package shz;

import shz.model.TopInfo;
import shz.queue.RedBlackBSTPQueue;

import java.util.*;
import java.util.function.*;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@SuppressWarnings("unchecked")
public final class ToMap<K, V> {
    private final Map<K, V> map;

    private ToMap(int initialCapacity, int idx, boolean dummy) {
        map = dummy ? new LinkedHashMap<>(reduce(initialCapacity, idx), 1.0f)
                : new HashMap<>(reduce(initialCapacity, idx), 1.0f);
    }

    private static final int MAX = 1 << 30;

    static int reduce(int cap, int idx) {
        if (idx <= 0 || cap == 1) return (int) Math.ceil(cap / 0.75f);
        int n = cap - 1;
        n |= n >>> 1;
        n |= n >>> 2;
        n |= n >>> 4;
        n |= n >>> 8;
        n |= n >>> 16;
        n = n >= MAX ? MAX : n + 1;
        while (idx-- > 0) {
            if (n < 2) break;
            n >>>= 1;
        }
        return (int) Math.ceil(n / 0.75f);
    }

    public static <K, V> ToMap<K, V> get(int initialCapacity, int idx, boolean dummy) {
        return new ToMap<>(initialCapacity, idx, dummy);
    }

    public static <K, V> ToMap<K, V> get(int initialCapacity, int idx) {
        return new ToMap<>(initialCapacity, idx, false);
    }

    public static <K, V> ToMap<K, V> get(int initialCapacity) {
        return new ToMap<>(initialCapacity, 0, false);
    }

    public <T> T build() {
        return (T) map;
    }

    public ToMap<K, V> put(Map<? extends K, ? extends V> m) {
        map.putAll(m);
        return this;
    }

    public <KK extends K, VV extends V> ToMap<K, V> put(KK k, VV v) {
        map.put(k, v);
        return this;
    }

    public static <E, K, V, M extends Map<K, V>> Collector<E, ?, M> collector(
            Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper,
            BinaryOperator<V> mergeFunction, Supplier<M> mapSupplier) {
        return Collector.of(
                mapSupplier,
                //替换默认merge方法
                (map, element) -> merge(keyMapper.apply(element), valueMapper.apply(element), map, mergeFunction),
                (m1, m2) -> {
                    for (Map.Entry<K, V> e : m2.entrySet()) merge(e.getKey(), e.getValue(), m1, mergeFunction);
                    return m1;
                }
        );
    }

    private static <K, V, M extends Map<K, V>> void merge(K key, V value, M map, BinaryOperator<V> mergeFunction) {
        V oldValue;
        if ((oldValue = map.get(key)) == null) {
            //原key的值为null则使用新值（包括null）
            map.put(key, value);
            return;
        }
        //原key的值不为null则使用具体的取值策略
        map.put(key, mergeFunction.apply(oldValue, value));
    }

    public static <E, K, V, M extends Map<K, V>> Collector<E, ?, M> collector(
            Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper,
            Supplier<M> mapSupplier) {
        return collector(keyMapper, valueMapper,
                //原key的值不为null时的取值策略，默认策略为抛出异常，这里选择若新值不为null则替换原来的值
                (oldValue, newValue) -> newValue == null ? oldValue : newValue,
                mapSupplier
        );
    }

    /**
     * 大致确定元素个数时的收集器
     *
     * @param keyMapper       键映射器
     * @param valueMapper     值映射器
     * @param initialCapacity 大致确定的初始容量
     * @param idx             初始容量衰减次数
     * @param dummy           是否排序的
     */
    public static <E, K, V> Collector<E, ?, Map<K, V>> collector(
            Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int initialCapacity, int idx, boolean dummy) {
        //提供初始化容量的HashMap（若dummy为true则为LinkedHashMap）
        return collector(keyMapper, valueMapper, () -> get(initialCapacity, idx, dummy).build());
    }

    public static <E, K, V> Collector<E, ?, Map<K, V>> collector(
            Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int initialCapacity, int idx) {
        return collector(keyMapper, valueMapper, initialCapacity, idx, false);
    }

    public static <E, K, V> Collector<E, ?, Map<K, V>> collector(
            Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int initialCapacity) {
        return collector(keyMapper, valueMapper, initialCapacity, 0, false);
    }

    /**
     * 大致确定元素个数时的流收集
     */
    public static <E, K, V> Map<K, V> explicitCollect(
            Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int initialCapacity, int idx, boolean dummy) {
        Map<K, V> map = stream.collect(collector(keyMapper, valueMapper, initialCapacity, idx, dummy));
        return map.isEmpty() ? Collections.emptyMap() : map;
    }

    public static <E, K, V> Map<K, V> explicitCollect(
            Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int initialCapacity) {
        return explicitCollect(stream, keyMapper, valueMapper, initialCapacity, 0, false);
    }

    /**
     * 无法确定元素个数时的收集器
     */
    public static <E, K, V> Collector<E, ?, Map<K, V>> collector(Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, boolean dummy) {
        return collector(keyMapper, valueMapper, dummy ? LinkedHashMap::new : HashMap::new);
    }

    public static <E, K, V> Collector<E, ?, Map<K, V>> collector(Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper) {
        return collector(keyMapper, valueMapper, false);
    }

    /**
     * 无法确定元素个数时的流收集
     *
     * @param extra 申请额外空间
     */
    public static <E, K, V> Map<K, V> collect(Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, boolean dummy, int extra) {
        int size = Math.max(extra, 0);
        Map<K, V> map = stream.collect(collector(keyMapper, valueMapper, dummy));
        if (map.isEmpty()) return size == 0 ? Collections.emptyMap() : get(size, 0, dummy).build();
        if (size == 0) return map;
        //原来的初始容量
        int oldCap = reduce(map.size(), 0);
        //申请额外空间后计算需扩容一次达到指定容量的初始容量
        int newCap = reduce(size + map.size(), 1);
        //如果添加额外元素最多导致扩容一次则直接返回
        // 考虑到申请额外空间不一定会真正的去执行，因此这一次的扩容留给具体执行的方法(不一定会扩容)
        if (oldCap >= newCap) return map;
        //下面代码基本不会执行，除非申请额外空间非常大（即需要扩容两次以上）

        //初始容量避免添加额外元素时导致频繁扩容
        newCap = reduce(size + map.size(), 0);
        Map<K, V> result = dummy ? new LinkedHashMap<>(newCap, 1.0f) : new HashMap<>(newCap, 1.0f);
        result.putAll(map);
        return result;
    }

    public static <E, K, V> Map<K, V> collect(Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, boolean dummy) {
        return collect(stream, keyMapper, valueMapper, dummy, 0);
    }

    public static <E, K, V> Map<K, V> collect(Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper, int extra) {
        return collect(stream, keyMapper, valueMapper, false, extra);
    }

    /**
     * 如果流中具有重复的key又不想抛异常则使用该方法，否则使用对应的Collectors.toMap()即可
     */
    public static <E, K, V> Map<K, V> collect(Stream<E> stream, Function<? super E, ? extends K> keyMapper, Function<? super E, ? extends V> valueMapper) {
        return collect(stream, keyMapper, valueMapper, false, 0);
    }

    /**
     * 分组列表
     *
     * @param list        需要分组的数据
     * @param classifiers 分组函数集
     */
    @SafeVarargs
    public static <E, M extends Map<?, ?>> M unique(List<E> list, Function<E, ?>... classifiers) {
        if (Validator.isEmpty(list)) return (M) Collections.emptyMap();
        Collector<E, ?, ? extends Map<?, ?>> collector = Collectors.groupingBy(classifiers[classifiers.length - 1], Collectors.reducing(null, (a, b) -> b));
        for (int i = classifiers.length - 2; i >= 0; --i) collector = Collectors.groupingBy(classifiers[i], collector);
        return (M) list.stream().collect(collector);
    }

    /**
     * 获取topN
     */
    public static <E, K> Map<K, List<E>> topN(Consumer<Consumer<E>> runnable, Function<E, TopInfo<K>> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        Map<K, RedBlackBSTPQueue<E>> queues = new HashMap<>();
        runnable.accept(t -> {
            if (t == null || (filter != null && !filter.test(t))) return;
            TopInfo<K> topInfo = group.apply(t);
            RedBlackBSTPQueue<E> queue = queues.computeIfAbsent(topInfo.key, k -> RedBlackBSTPQueue.Min.of(comparator::apply));
            while (queue.size() > topInfo.n) queue.poll();
            if (queue.size() == topInfo.n) {
                if (comparator.apply(t, queue.peek()) > 0) {
                    queue.poll();
                    queue.offer(t);
                }
            } else queue.offer(t);
        });
        return ToMap.explicitCollect(queues.keySet().stream(), Function.identity(), key -> queues.get(key).reverse(), queues.size());
    }

    public static <E, K> Map<K, List<E>> topN(Supplier<E> supplier, Function<E, TopInfo<K>> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        return topN(consumer -> {
            E e;
            while ((e = supplier.get()) != null) consumer.accept(e);
        }, group, filter, comparator);
    }

    public static <E, K> Map<K, List<E>> topN(Collection<E> collection, Function<E, TopInfo<K>> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        return topN(consumer -> {
            for (E e : collection) consumer.accept(e);
        }, group, filter, comparator);
    }

    public static <E, K> Map<K, E> topOne(Consumer<Consumer<E>> runnable, Function<E, K> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        Map<K, RedBlackBSTPQueue<E>> queues = new HashMap<>();
        runnable.accept(t -> {
            if (t == null || (filter != null && !filter.test(t))) return;
            K key = group.apply(t);
            RedBlackBSTPQueue<E> queue = queues.computeIfAbsent(key, k -> RedBlackBSTPQueue.Min.of(comparator::apply));
            while (queue.size() > 1) queue.poll();
            if (queue.size() == 1) {
                if (comparator.apply(t, queue.peek()) > 0) {
                    queue.poll();
                    queue.offer(t);
                }
            } else queue.offer(t);
        });
        return ToMap.explicitCollect(queues.keySet().stream(), Function.identity(), key -> queues.get(key).poll(), queues.size());
    }

    public static <E, K> Map<K, E> topOne(Supplier<E> supplier, Function<E, K> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        return topOne(consumer -> {
            E e;
            while ((e = supplier.get()) != null) consumer.accept(e);
        }, group, filter, comparator);
    }

    public static <E, K> Map<K, E> topOne(Collection<E> collection, Function<E, K> group, Predicate<E> filter, BiFunction<E, E, Integer> comparator) {
        return topOne(consumer -> {
            for (E e : collection) consumer.accept(e);
        }, group, filter, comparator);
    }
}
