package shz.accept;

import shz.AccessibleHelp;
import shz.ThreadHelp;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * 数据接收执行器
 *
 * @param <T> 执行参数
 */
public abstract class AcceptExecutor<T> {
    private final String threadName;
    private Executor executor;
    private final Lock lock;
    private final Condition sleep;
    private final Condition finish;
    private final Condition delay;
    private volatile boolean stop = true;
    private volatile boolean wait;
    protected final Map<String, AcceptCallback> callbacks;

    protected AcceptExecutor(String threadName) {
        this.threadName = threadName;
        lock = new ReentrantLock();
        sleep = lock.newCondition();
        finish = lock.newCondition();
        delay = lock.newCondition();
        callbacks = new HashMap<>();
    }

    protected AcceptExecutor() {
        this.threadName = AccessibleHelp.identify(getClass());
        lock = new ReentrantLock();
        sleep = lock.newCondition();
        finish = lock.newCondition();
        delay = lock.newCondition();
        callbacks = new HashMap<>();
    }

    protected boolean acceptable(T t) {
        return true;
    }

    /**
     * 数据消费业务方法
     */
    protected abstract void consumer(T t);

    protected long sleepMicroSeconds(T t) {
        return 0L;
    }

    protected boolean stoppable(T t) {
        return false;
    }

    protected long gapMilliSeconds(T t) {
        return 0L;
    }

    protected long gapSleepMilliSeconds(T t) {
        return 50L;
    }

    protected long finishMilliSeconds(T t) {
        return 0L;
    }

    protected long delayMicroSeconds(T t) {
        return 5000L;
    }

    protected void log(Throwable cause) {
        cause.printStackTrace();
    }

    private void run(T t) {
        while (!stop) {
            if (!lock.tryLock()) break;
            try {
                long idleMillis = 0L, gapMillis = gapMilliSeconds(t), gapSleepMillis = gapSleepMilliSeconds(t);
                if (gapMillis > 0L) gapSleepMillis = Math.min(gapMillis, gapSleepMillis <= 0 ? 50L : gapSleepMillis);
                else gapSleepMillis = 0L;
                while (!stop) {
                    boolean acceptable = false;
                    try {
                        acceptable = acceptable(t);
                    } catch (Throwable e) {
                        log(e);
                    }
                    if (!acceptable) {
                        //无消费数据
                        if (gapSleepMillis == 0L || idleMillis >= gapMillis) break;
                        try {
                            TimeUnit.MILLISECONDS.sleep(gapSleepMillis);
                        } catch (InterruptedException ignored) {
                        }
                        //累计空闲时间
                        idleMillis += gapSleepMillis;
                        continue;
                    }
                    //重置空闲时间
                    idleMillis = 0L;

                    try {
                        //执行消费方法
                        consumer(t);
                    } catch (Throwable e) {
                        log(e);
                    }

                    try {
                        long time = sleepMicroSeconds(t);
                        //消费睡眠
                        if (time > 0L) sleep.await(time, TimeUnit.MICROSECONDS);
                    } catch (Throwable e) {
                        log(e);
                    }
                }

                //判断是否结束线程任务
                if (!stop) try {
                    stop = stoppable(t);
                } catch (Throwable e) {
                    log(e);
                }

                if (!stop) {
                    wait = true;
                    if (!stop) {
                        long finishTime = 100L;
                        long curMillis = System.currentTimeMillis();
                        try {
                            //当前没有可消费，等待被唤醒或等待超时
                            finishTime = finishMilliSeconds(t);

                            //执行回调
                            callbacks.values().forEach(AcceptCallback::execute);
                        } catch (Throwable e) {
                            log(e);
                        }

                        if (finishTime > 0L) {
                            long delta = finishTime + curMillis - System.currentTimeMillis();
                            finishTime = delta != 0L ? delta : 100L;
                        } else finishTime = 0L;

                        if (wait) {
                            try {
                                if (finishTime > 0L) finish.await(finishTime, TimeUnit.MILLISECONDS);
                                else if (finishTime == 0L) finish.await();
                            } catch (InterruptedException e) {
                                log(e);
                            }
                            wait = false;
                        }

                        long delayTime = 0L;
                        try {
                            //延迟执行
                            delayTime = delayMicroSeconds(t);
                        } catch (Throwable e) {
                            log(e);
                        }

                        if (delayTime > 0L && finishTime < 0L) {
                            long delta = delayTime + finishTime;
                            delayTime = delta > 0 ? delta : 100L;
                        }

                        try {
                            if (delayTime > 0L) delay.await(delayTime, TimeUnit.MICROSECONDS);
                        } catch (InterruptedException e) {
                            log(e);
                        }
                    }
                }
            } finally {
                lock.unlock();
            }
        }
    }

    public final void execute(T target) {
        if (!stop || !lock.tryLock()) return;
        try {
            if (!stop) return;
            stop = false;
            if (executor == null) executor = ThreadHelp.getExecutor(ThreadHelp.TPConfig.of(threadName)
                    .tpType(ThreadHelp.TPType.SINGLE_THREAD_EXECUTOR)
                    .daemon(true)
                    .workQueue(new LinkedBlockingQueue<>(1))
            );
            executor.execute(() -> run(target));
        } finally {
            lock.unlock();
        }
    }

    private void awaken() {
        if (!wait || !lock.tryLock()) return;
        try {
            if (!wait) return;
            wait = false;
            finish.signal();
        } catch (Throwable t) {
            log(t);
        } finally {
            lock.unlock();
        }
    }

    public final void awaken(T target) {
        execute(target);
        awaken();
    }

    public final boolean isStop() {
        return stop;
    }

    public synchronized void stop() {
        if (!stop) {
            stop = true;
            awaken();
        }
    }

    public final void registerCallback(String name, AcceptCallback callback) {
        callbacks.put(name, callback);
    }

    public final void unRegisterCallback(String name) {
        callbacks.remove(name);
    }
}