package shz.encrypt;

import shz.*;
import shz.msg.ClientFailure;

import java.io.File;
import java.text.Collator;
import java.util.*;

public abstract class AbstractEncryptManager implements EncryptManager {
    protected final RsaEncipher rsaEncipher;
    protected final byte[] privateKey;
    protected final byte[] publicKey;

    protected AbstractEncryptManager() {
        rsaEncipher = RsaEncipher.getInstance();
        SimpleKeyPair keyPair = getKeyPair();
        if (keyPair == null
                || Validator.isBlank(keyPair.privateKey)
                || Validator.isBlank(keyPair.publicKey)) {
            rsaEncipher.initKeyPair();
            privateKey = rsaEncipher.getKeyPair().getPrivate().getEncoded();
            publicKey = rsaEncipher.getKeyPair().getPublic().getEncoded();
            saveKeyPair(new SimpleKeyPair(encodeBytes(privateKey), encodeBytes(publicKey)));
        } else {
            privateKey = decodeString(keyPair.privateKey);
            publicKey = decodeString(keyPair.publicKey);
        }
    }

    protected SimpleKeyPair getKeyPair() {
        try {
            String rsaKeyDir = System.getProperty("user.dir");

            String privateKey = FileHelp.readString(new File(rsaKeyDir, "privateKey")).replaceAll("\\s+", "");
            if (Validator.isBlank(privateKey)) return null;

            String publicKey = FileHelp.readString(new File(rsaKeyDir, "publicKey")).replaceAll("\\s+", "");
            if (Validator.isBlank(publicKey)) return null;

            return new SimpleKeyPair(privateKey, publicKey);
        } catch (Throwable t) {
            return null;
        }
    }

    protected void saveKeyPair(SimpleKeyPair keyPair) {
        if (keyPair == null
                || Validator.isBlank(keyPair.getPrivateKey())
                || Validator.isBlank(keyPair.getPublicKey())) return;
        String rsaKeyDir = System.getProperty("user.dir");

        FileHelp.writeChars(new File(rsaKeyDir, "privateKey"), keyPair.getPrivateKey().toCharArray());
        FileHelp.writeChars(new File(rsaKeyDir, "publicKey"), keyPair.getPublicKey().toCharArray());
    }

    public final void resetKeyPair() {
        rsaEncipher.initKeyPair();
        System.arraycopy(rsaEncipher.getKeyPair().getPrivate().getEncoded(), 0, privateKey, 0, privateKey.length);
        System.arraycopy(rsaEncipher.getKeyPair().getPublic().getEncoded(), 0, publicKey, 0, publicKey.length);
        saveKeyPair(new SimpleKeyPair(encodeBytes(privateKey), encodeBytes(publicKey)));
    }

    public final String getPublicKey() {
        return encodeBytes(publicKey);
    }

    public final String encryptKey(String key, String publicKey) {
        return encodeBytes(rsaEncipher.encryptByPublicKey(stringToBytes(key), decodeString(publicKey)));
    }

    protected final String decryptKey(String key) {
        return bytesToString(rsaEncipher.decryptByPrivateKey(decodeString(key), privateKey));
    }

    /**
     * 使用私钥对数据签名再次签名
     */
    public final String sign(String signature) {
        return encodeBytes(rsaEncipher.encryptByPrivateKey(stringToBytes(signature), privateKey));
    }

    /**
     * 默认的数据签名方法
     */
    @Override
    public String sign(EncryptParam encryptParam, Object data) {
        Map<String, String> map = PropHelp.fieldValueMap(data);
        Set<String> keys = map.keySet();
        List<String> list = ToList.explicitCollect(keys.stream().filter(k -> !"null".equals(k) && !"sign".equals(k) && !"pl_sign".equals(k)), keys.size());
        list.sort(Collator.getInstance(Locale.CHINA));
        return Coder.md5(stringToBytes(String.join("", ToList.explicitCollect(list.stream().map(map::get), list.size())) + encryptParam.signKey));
    }

    @Override
    public final void checkSign(EncryptParam encryptParam, String signature, Object data) {
        if (Validator.nonBlank(encryptParam.publicKey))
            signature = bytesToString(rsaEncipher.decryptByPublicKey(decodeString(signature), decodeString(encryptParam.publicKey)));
        ClientFailure.INVALID_SIGNATURE.requireNon(!Objects.equals(sign(encryptParam, data), signature));
    }
}