package shz.linked;

import shz.UnsafeHelp;

/**
 * 元素类型为double支持并发的双向链表节点
 * <p>
 * 24*n(n为元素个数)
 * <p>
 * B=16+24*n
 */
@SuppressWarnings({"restriction"})
public class ConcurrentDDNode implements DNode {
    public volatile double val;
    protected volatile ConcurrentDDNode next;
    protected volatile ConcurrentDDNode prev;

    protected ConcurrentDDNode(double val) {
        this.val = val;
    }

    public static ConcurrentDDNode of(double e) {
        return new ConcurrentDDNode(e);
    }

    public static ConcurrentDDNode of() {
        return of(0d);
    }

    private static final long valOffset;
    private static final long nextOffset;
    private static final long prevOffset;

    static {
        try {
            Class<?> k = ConcurrentDDNode.class;
            valOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("val"));
            nextOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("next"));
            prevOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("prev"));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public boolean casVal(double expect, double val) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, valOffset, expect, val);
    }

    public boolean casNext(ConcurrentDDNode expect, ConcurrentDDNode next) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, nextOffset, expect, next);
    }

    public boolean casPrev(ConcurrentDDNode expect, ConcurrentDDNode prev) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, prevOffset, expect, prev);
    }

    @Override
    public final ConcurrentDDNode next() {
        return next;
    }

    @Override
    public final void next(DNode node) {
        next = (ConcurrentDDNode) node;
    }

    @Override
    public final ConcurrentDDNode prev() {
        return prev;
    }

    @Override
    public final void prev(DNode node) {
        prev = (ConcurrentDDNode) node;
    }

    @Override
    public final ConcurrentDDNode addNext(DNode node) {
        ConcurrentDDNode cddNode = (ConcurrentDDNode) node;
        while (true) {
            ConcurrentDDNode next = this.next;
            cddNode.prev = this;
            cddNode.next = next;
            if (!casNext(next, cddNode)) continue;
            if (next != null) next.prev = cddNode;
            break;
        }
        return cddNode;
    }

    @Override
    public final ConcurrentDDNode addPrev(DNode node) {
        ConcurrentDDNode cddNode = (ConcurrentDDNode) node;
        while (true) {
            ConcurrentDDNode prev = this.prev;
            cddNode.prev = prev;
            cddNode.next = this;
            if (!casPrev(prev, cddNode)) continue;
            if (prev != null) prev.next = cddNode;
            break;
        }
        return cddNode;
    }

    @Override
    public final void poll() {
        ConcurrentDDNode prev = this.prev;
        ConcurrentDDNode next = this.next;
        if (prev != null) {
            if (prev.casNext(this, next)) if (next != null) next.prev = prev;
        } else if (next != null) next.casPrev(this, null);
    }

    public final ConcurrentDDNode addNext(double e) {
        return addNext(of(e));
    }

    public final ConcurrentDDNode addNext(double... es) {
        ConcurrentDDNode next = this;
        for (double e : es) next = next.addNext(e);
        return next;
    }

    public final ConcurrentDDNode addPrev(double e) {
        return addPrev(of(e));
    }

    public final ConcurrentDDNode addPrev(double... es) {
        ConcurrentDDNode prev = this;
        for (double e : es) prev = prev.addPrev(e);
        return prev;
    }
}