package shz.linked;

import shz.UnsafeHelp;

/**
 * 元素类型为long支持并发的双向链表节点
 * <p>
 * 24*n(n为元素个数)
 * <p>
 * B=16+24*n
 */
@SuppressWarnings("restriction")
public class ConcurrentJDNode implements DNode {
    public volatile long val;
    protected volatile ConcurrentJDNode next;
    protected volatile ConcurrentJDNode prev;

    protected ConcurrentJDNode(long val) {
        this.val = val;
    }

    public static ConcurrentJDNode of(long e) {
        return new ConcurrentJDNode(e);
    }

    public static ConcurrentJDNode of() {
        return of(0L);
    }

    private static final long valOffset;
    private static final long nextOffset;
    private static final long prevOffset;

    static {
        try {
            Class<?> k = ConcurrentJDNode.class;
            valOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("val"));
            nextOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("next"));
            prevOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("prev"));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public boolean casVal(long expect, long val) {
        return UnsafeHelp.getUnsafe().compareAndSwapLong(this, valOffset, expect, val);
    }

    public boolean casNext(ConcurrentJDNode expect, ConcurrentJDNode next) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, nextOffset, expect, next);
    }

    public boolean casPrev(ConcurrentJDNode expect, ConcurrentJDNode prev) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, prevOffset, expect, prev);
    }

    @Override
    public final ConcurrentJDNode next() {
        return next;
    }

    @Override
    public final void next(DNode node) {
        next = (ConcurrentJDNode) node;
    }

    @Override
    public final ConcurrentJDNode prev() {
        return prev;
    }

    @Override
    public final void prev(DNode node) {
        prev = (ConcurrentJDNode) node;
    }

    @Override
    public final ConcurrentJDNode addNext(DNode node) {
        ConcurrentJDNode cjdNode = (ConcurrentJDNode) node;
        while (true) {
            ConcurrentJDNode next = this.next;
            cjdNode.prev = this;
            cjdNode.next = next;
            if (!casNext(next, cjdNode)) continue;
            if (next != null) next.prev = cjdNode;
            break;
        }
        return cjdNode;
    }

    @Override
    public final ConcurrentJDNode addPrev(DNode node) {
        ConcurrentJDNode cjdNode = (ConcurrentJDNode) node;
        while (true) {
            ConcurrentJDNode prev = this.prev;
            cjdNode.prev = prev;
            cjdNode.next = this;
            if (!casPrev(prev, cjdNode)) continue;
            if (prev != null) prev.next = cjdNode;
            break;
        }
        return cjdNode;
    }

    @Override
    public final void poll() {
        ConcurrentJDNode prev = this.prev;
        ConcurrentJDNode next = this.next;
        if (prev != null) {
            if (prev.casNext(this, next)) if (next != null) next.prev = prev;
        } else if (next != null) next.casPrev(this, null);
    }

    public final ConcurrentJDNode addNext(long e) {
        return addNext(of(e));
    }

    public final ConcurrentJDNode addNext(long... es) {
        ConcurrentJDNode next = this;
        for (long e : es) next = next.addNext(e);
        return next;
    }

    public final ConcurrentJDNode addPrev(long e) {
        return addPrev(of(e));
    }

    public final ConcurrentJDNode addPrev(long... es) {
        ConcurrentJDNode prev = this;
        for (long e : es) prev = prev.addPrev(e);
        return prev;
    }
}