package shz.linked;

import shz.UnsafeHelp;

/**
 * 元素类型为long支持并发的单向链表节点
 * <p>
 * 16*n(n为元素个数)
 * <p>
 * B=16*(n+1)
 */
@SuppressWarnings("restriction")
public class ConcurrentJSNode implements SNode {
    public volatile long val;
    protected volatile ConcurrentJSNode next;

    protected ConcurrentJSNode(long val) {
        this.val = val;
    }

    public static ConcurrentJSNode of(long e) {
        return new ConcurrentJSNode(e);
    }

    public static ConcurrentJSNode of() {
        return of(0L);
    }

    private static final long valOffset;
    private static final long nextOffset;

    static {
        try {
            Class<?> k = ConcurrentJSNode.class;
            valOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("val"));
            nextOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("next"));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public final boolean casVal(long expect, long val) {
        return UnsafeHelp.getUnsafe().compareAndSwapLong(this, valOffset, expect, val);
    }

    public final boolean casNext(ConcurrentJSNode expect, ConcurrentJSNode next) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, nextOffset, expect, next);
    }

    @Override
    public final ConcurrentJSNode next() {
        return next;
    }

    @Override
    public final void next(SNode node) {
        next = (ConcurrentJSNode) node;
    }

    @Override
    public final ConcurrentJSNode addNext(SNode node) {
        ConcurrentJSNode cjsNode = (ConcurrentJSNode) node;
        ConcurrentJSNode next;
        do {
            next = this.next;
            cjsNode.next = next;
        } while (!casNext(next, cjsNode));
        return cjsNode;
    }

    @Override
    public final ConcurrentJSNode addPrev(SNode node) {
        ConcurrentJSNode cjsNode = addNext(node);
        long val;
        do {
            val = this.val;
        } while (!casVal(val, cjsNode.val));
        cjsNode.val = val;
        return this;
    }

    @Override
    public final void poll() {
        if (next == null) this.val = 0L;
        else {
            ConcurrentJSNode next = this.next;
            if (casNext(next, next.next)) this.val = next.val;
        }
    }

    public final ConcurrentJSNode addNext(long e) {
        return addNext(of(e));
    }

    public final ConcurrentJSNode addNext(long... es) {
        ConcurrentJSNode next = this;
        for (long e : es) next = next.addNext(e);
        return next;
    }

    public final ConcurrentJSNode addPrev(long e) {
        return addPrev(of(e));
    }

    public final ConcurrentJSNode addPrev(long... es) {
        for (long e : es) addPrev(e);
        return this;
    }
}