package shz.linked;

import shz.UnsafeHelp;

/**
 * 元素类型为E支持并发的双向链表节点
 * <p>
 * [24+E(类型字节)+对齐填充]*n(n为元素个数)
 * <p>
 * B=16+24*n+(E+对齐填充)*n
 */
@SuppressWarnings({"restriction", "unchecked"})
public class ConcurrentLDNode<E> implements DNode {
    public volatile E val;
    protected volatile ConcurrentLDNode<E> next;
    protected volatile ConcurrentLDNode<E> prev;

    protected ConcurrentLDNode(E val) {
        this.val = val;
    }

    public static <E> ConcurrentLDNode<E> of(E e) {
        return new ConcurrentLDNode<>(e);
    }

    public static <E> ConcurrentLDNode<E> of() {
        return of(null);
    }

    private static final long valOffset;
    private static final long nextOffset;
    private static final long prevOffset;

    static {
        try {
            Class<?> k = ConcurrentLDNode.class;
            valOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("val"));
            nextOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("next"));
            prevOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("prev"));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public boolean casVal(E expect, E val) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, valOffset, expect, val);
    }

    public boolean casNext(ConcurrentLDNode<E> expect, ConcurrentLDNode<E> next) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, nextOffset, expect, next);
    }

    public boolean casPrev(ConcurrentLDNode<E> expect, ConcurrentLDNode<E> prev) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, prevOffset, expect, prev);
    }

    @Override
    public final ConcurrentLDNode<E> next() {
        return next;
    }

    @Override
    public final void next(DNode node) {
        next = (ConcurrentLDNode<E>) node;
    }

    @Override
    public final ConcurrentLDNode<E> prev() {
        return prev;
    }

    @Override
    public final void prev(DNode node) {
        prev = (ConcurrentLDNode<E>) node;
    }

    @Override
    public final ConcurrentLDNode<E> addNext(DNode node) {
        ConcurrentLDNode<E> cldNode = (ConcurrentLDNode<E>) node;
        while (true) {
            ConcurrentLDNode<E> next = this.next;
            cldNode.prev = this;
            cldNode.next = next;
            if (!casNext(next, cldNode)) continue;
            if (next != null) next.prev = cldNode;
            break;
        }
        return cldNode;
    }

    @Override
    public final ConcurrentLDNode<E> addPrev(DNode node) {
        ConcurrentLDNode<E> cldNode = (ConcurrentLDNode<E>) node;
        while (true) {
            ConcurrentLDNode<E> prev = this.prev;
            cldNode.prev = prev;
            cldNode.next = this;
            if (!casPrev(prev, cldNode)) continue;
            if (prev != null) prev.next = cldNode;
            break;
        }
        return cldNode;
    }

    @Override
    public final void poll() {
        ConcurrentLDNode<E> prev = this.prev;
        ConcurrentLDNode<E> next = this.next;
        if (prev != null) {
            if (prev.casNext(this, next)) if (next != null) next.prev = prev;
        } else if (next != null) next.casPrev(this, null);
    }

    public final ConcurrentLDNode<E> addNext(E e) {
        return addNext(of(e));
    }

    @SafeVarargs
    public final ConcurrentLDNode<E> addNext(E... es) {
        ConcurrentLDNode<E> next = this;
        for (E e : es) next = next.addNext(e);
        return next;
    }

    public final ConcurrentLDNode<E> addPrev(E e) {
        return addPrev(of(e));
    }

    @SafeVarargs
    public final ConcurrentLDNode<E> addPrev(E... es) {
        ConcurrentLDNode<E> prev = this;
        for (E e : es) prev = prev.addPrev(e);
        return prev;
    }
}