package shz.linked;

import shz.UnsafeHelp;

/**
 * 元素类型为E支持并发的单向链表节点
 * <p>
 * [16+E(类型字节)+对齐填充]*n(n为元素个数)
 * <p>
 * B=16*(n+1)+(E+对齐填充)*n
 */
@SuppressWarnings({"restriction", "unchecked"})
public class ConcurrentLSNode<E> implements SNode {
    public volatile E val;
    protected volatile ConcurrentLSNode<E> next;

    protected ConcurrentLSNode(E val) {
        this.val = val;
    }

    public static <E> ConcurrentLSNode<E> of(E e) {
        return new ConcurrentLSNode<>(e);
    }

    public static <E> ConcurrentLSNode<E> of() {
        return of(null);
    }

    private static final long valOffset;
    private static final long nextOffset;

    static {
        try {
            Class<?> k = ConcurrentLSNode.class;
            valOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("val"));
            nextOffset = UnsafeHelp.getUnsafe().objectFieldOffset(k.getDeclaredField("next"));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public final boolean casVal(E expect, E val) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, valOffset, expect, val);
    }

    public final boolean casNext(ConcurrentLSNode<E> expect, ConcurrentLSNode<E> next) {
        return UnsafeHelp.getUnsafe().compareAndSwapObject(this, nextOffset, expect, next);
    }

    @Override
    public final ConcurrentLSNode<E> next() {
        return next;
    }

    @Override
    public final void next(SNode node) {
        next = (ConcurrentLSNode<E>) node;
    }

    @Override
    public final ConcurrentLSNode<E> addNext(SNode node) {
        ConcurrentLSNode<E> clsNode = (ConcurrentLSNode<E>) node;
        ConcurrentLSNode<E> next;
        do {
            next = this.next;
            clsNode.next = next;
        } while (!casNext(next, clsNode));
        return clsNode;
    }

    @Override
    public final ConcurrentLSNode<E> addPrev(SNode node) {
        ConcurrentLSNode<E> clsNode = addNext(node);
        E val;
        do {
            val = this.val;
        } while (!casVal(val, clsNode.val));
        clsNode.val = val;
        return this;
    }

    @Override
    public final void poll() {
        if (next == null) this.val = null;
        else {
            ConcurrentLSNode<E> next = this.next;
            if (casNext(next, next.next)) this.val = next.val;
        }
    }

    public final ConcurrentLSNode<E> addNext(E e) {
        return addNext(of(e));
    }

    @SafeVarargs
    public final ConcurrentLSNode<E> addNext(E... es) {
        ConcurrentLSNode<E> next = this;
        for (E e : es) next = next.addNext(e);
        return next;
    }

    public final ConcurrentLSNode<E> addPrev(E e) {
        return addPrev(of(e));
    }

    @SafeVarargs
    public final ConcurrentLSNode<E> addPrev(E... es) {
        for (E e : es) addPrev(e);
        return this;
    }
}