package shz.msg;

import shz.PRException;
import shz.Validator;

import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

public interface FailureMsg extends Message {
    @Override
    default boolean isOk() {
        return false;
    }

    static FailureMsg fail(int code, String msg) {
        return new FailureMsg() {
            @Override
            public final int code() {
                return code;
            }

            @Override
            public final String msg() {
                return msg;
            }
        };
    }

    static FailureMsg fail(Throwable cause) {
        if (cause == null) return ServerFailure.INTERNAL_ERROR;
        if (cause instanceof InvocationTargetException)
            cause = ((InvocationTargetException) cause).getTargetException();
        if (cause instanceof PRException) return ((PRException) cause).msg();

        if (cause instanceof IllegalThreadStateException) return ServerFailure.IllegalThreadStateException;
        if (cause instanceof NumberFormatException) return ServerFailure.NumberFormatException;
        if (cause instanceof IllegalArgumentException) return ServerFailure.IllegalArgumentException;

        if (cause instanceof SecurityException) return ServerFailure.SecurityException;
        if (cause instanceof UnsupportedOperationException) return ServerFailure.UnsupportedOperationException;
        if (cause instanceof IllegalAccessException) return ServerFailure.IllegalAccessException;

        if (cause instanceof IllegalStateException) return ServerFailure.IllegalStateException;
        if (cause instanceof ClassNotFoundException) return ServerFailure.ClassNotFoundException;
        if (cause instanceof CloneNotSupportedException) return ServerFailure.CloneNotSupportedException;
        if (cause instanceof InstantiationException) return ServerFailure.InstantiationException;
        if (cause instanceof InterruptedException) return ServerFailure.InterruptedException;
        if (cause instanceof NoSuchFieldException) return ServerFailure.NoSuchFieldException;
        if (cause instanceof NoSuchMethodException) return ServerFailure.NoSuchMethodException;
        if (cause instanceof ArithmeticException) return ServerFailure.ArithmeticException;
        if (cause instanceof ArrayIndexOutOfBoundsException) return ServerFailure.ArrayIndexOutOfBoundsException;
        if (cause instanceof ArrayStoreException) return ServerFailure.ArrayStoreException;
        if (cause instanceof ClassCastException) return ServerFailure.ClassCastException;
        if (cause instanceof IllegalMonitorStateException) return ServerFailure.IllegalMonitorStateException;

        if (cause instanceof StringIndexOutOfBoundsException) return ServerFailure.StringIndexOutOfBoundsException;
        if (cause instanceof IndexOutOfBoundsException) return ServerFailure.IndexOutOfBoundsException;

        if (cause instanceof NegativeArraySizeException) return ServerFailure.NegativeArraySizeException;

        if (cause instanceof TimeoutException) return ServerFailure.GATEWAY_TIMEOUT;

        if (Validator.nonBlank(cause.getMessage())) return ServerFailureMsg.fail(cause.getMessage());
        return ServerFailureMsg.fail(Arrays.stream(cause.getStackTrace()).map(Object::toString).collect(Collectors.joining("\n")));
    }

    default void requireNon(boolean condition) {
        if (condition) throw PRException.of(this);
    }

    default void requireNonNull(Object obj) {
        if (obj == null) throw PRException.of(this);
    }

    default void requireNonEmpty(Object obj) {
        if (Validator.isEmpty(obj)) throw PRException.of(this);
    }

    default void requireNonBlank(Object obj) {
        if (Validator.isBlank(obj)) throw PRException.of(this);
    }
}
