package shz.queue;

import java.util.Iterator;

/**
 * 基于循环动态数组的队列
 */
public abstract class ArrayQueue<E> implements Iterable<E> {
    protected static final int DEFAULT_CAPACITY = 8;
    protected int capacity, size, tail, head;

    protected ArrayQueue(int capacity) {
        if (capacity < 1) throw new IllegalArgumentException();
        this.capacity = capacity + 1;
    }

    public final class ArrayQueueIterator implements Iterator<E> {
        private int current = head;

        @Override
        public boolean hasNext() {
            return current != tail;
        }

        @Override
        public E next() {
            E e = get(current);
            current = (current + 1) % capacity;
            return e;
        }
    }

    protected abstract E get(int i);

    @Override
    public final Iterator<E> iterator() {
        return new ArrayQueueIterator();
    }

    public final int size() {
        return size;
    }

    public final boolean isEmpty() {
        return size == 0;
    }

    protected final void beforeOffer() {
        if (size == capacity - 1) {
            if (capacity == Integer.MAX_VALUE) throw new OutOfMemoryError();
            int newCap = capacity >= 64 ? capacity + (capacity >> 1) : capacity << 1;
            resize(newCap < 0 ? Integer.MAX_VALUE : newCap);
        }
    }

    protected abstract void resize(int capacity);

    protected final void afterOffer() {
        tail = (tail + 1) % capacity;
        ++size;
    }

    protected final void afterPoll() {
        setNull(head);
        head = (head + 1) % capacity;
        --size;
        reduce();
    }

    private void reduce() {
        if (capacity > 64) {
            int newCap = size + 1 + (size + 1 >> 1);
            if (newCap + (newCap >> 1) <= capacity) resize(newCap);
        } else if (size + 1 <= capacity >>> 2) resize(capacity >> 1);
    }

    protected abstract void setNull(int i);

    public final void removeTail() {
        if (tail == 0) tail = capacity - 1;
        else --tail;
        setNull(tail);
        --size;
        reduce();
    }
}
