package shz.queue;

import shz.stack.LArrayStack;

import java.util.*;

public abstract class RedBlackBSTPQueue<E> {
    static final class Node<E> {
        E e;
        Node<E> left, right;
        int size = 1;
        boolean red;

        Node(E e, boolean red) {
            this.e = e;
            this.red = red;
        }
    }

    Node<E> root;
    final Comparator<? super E> comparator;

    RedBlackBSTPQueue(Comparator<? super E> comparator) {
        Objects.requireNonNull(comparator);
        this.comparator = comparator;
    }

    public final int size() {
        return size(root);
    }

    private int size(Node<E> h) {
        return h == null ? 0 : h.size;
    }

    public final boolean isEmpty() {
        return root == null || root.size == 0;
    }

    final Node<E> put(Node<E> h, E e) {
        if (h == null) return new Node<>(e, true);
        int cmp = comparator.compare(e, h.e);
        if (cmp <= 0) h.left = put(h.left, e);
        else h.right = put(h.right, e);
        if (isRed(h.right) && !isRed(h.left)) h = rotateLeft(h);
        if (isRed(h.left) && isRed(h.left.left)) h = rotateRight(h);
        if (isRed(h.left) && isRed(h.right)) flipColors(h);
        h.size = size(h.left) + size(h.right) + 1;
        return h;
    }

    final boolean isRed(Node<E> h) {
        return h != null && h.red;
    }

    final Node<E> rotateLeft(Node<E> h) {
        Node<E> x = h.right;
        h.right = x.left;
        x.left = h;
        x.red = h.red;
        h.red = true;
        x.size = h.size;
        h.size = 1 + size(h.left) + size(h.right);
        return x;
    }

    final Node<E> rotateRight(Node<E> h) {
        Node<E> x = h.left;
        h.left = x.right;
        x.right = h;
        x.red = h.red;
        h.red = true;
        x.size = h.size;
        h.size = 1 + size(h.left) + size(h.right);
        return x;
    }

    final void flipColors(Node<E> h) {
        h.red = true;
        if (h.left != null) h.left.red = false;
        if (h.right != null) h.right.red = false;
    }

    final Node<E> balance(Node<E> h) {
        if (isRed(h.right)) h = rotateLeft(h);
        if (isRed(h.right) && !isRed(h.left)) h = rotateLeft(h);
        if (isRed(h.left) && isRed(h.left.left)) h = rotateRight(h);
        if (isRed(h.left) && isRed(h.right)) flipColors(h);
        h.size = size(h.left) + size(h.right) + 1;
        return h;
    }

    public abstract void offer(E e);

    public abstract E poll();

    public abstract E peek();

    public static final class Min<E> extends RedBlackBSTPQueue<E> {
        private Min(Comparator<? super E> comparator) {
            super(comparator);
        }

        public static <E> Min<E> of(Comparator<? super E> comparator) {
            return new Min<>(comparator);
        }

        @Override
        public void offer(E e) {
            Objects.requireNonNull(e);
            if (root == null) {
                root = new Node<>(e, false);
                return;
            }
            root = put(root, e);
            root.red = false;
        }

        @Override
        public E poll() {
            E result = peek();
            if (result == null) return null;
            if (!isRed(root.left) && !isRed(root.right)) root.red = true;
            root = delTop(root);
            if (root != null && root.size > 0) root.red = false;
            return result;
        }

        @Override
        public E peek() {
            if (root == null) return null;
            Node<E> h = root;
            while (h.left != null) h = h.left;
            return h.e;
        }

        private Node<E> delTop(Node<E> h) {
            if (h.left == null) return h.right;
            if (!isRed(h.left) && !isRed(h.left.left)) h = moveRedLeft(h);
            h.left = delTop(h.left);
            return balance(h);
        }

        private Node<E> moveRedLeft(Node<E> h) {
            flipColors(h);
            if (h.right != null && isRed(h.right.left)) {
                h.right = rotateRight(h.right);
                h = rotateLeft(h);
            }
            return h;
        }
    }

    public static final class Max<E> extends RedBlackBSTPQueue<E> {
        private Max(Comparator<? super E> comparator) {
            super(comparator);
        }

        public static <E> Max<E> of(Comparator<? super E> comparator) {
            return new Max<>(comparator);
        }

        @Override
        public void offer(E e) {
            Objects.requireNonNull(e);
            if (root == null) {
                root = new Node<>(e, false);
                return;
            }
            root = put(root, e);
            root.red = false;
        }

        @Override
        public E poll() {
            E result = peek();
            if (result == null) return null;
            if (!isRed(root.left) && !isRed(root.right)) root.red = true;
            root = delTop(root);
            if (root != null && root.size > 0) root.red = false;
            return result;
        }

        @Override
        public E peek() {
            if (root == null) return null;
            Node<E> h = root;
            while (h.right != null) h = h.right;
            return h.e;
        }

        private Node<E> delTop(Node<E> h) {
            if (isRed(h.left)) h = rotateRight(h);
            if (h.right == null) return h.left;
            if (!isRed(h.right) && !isRed(h.right.left)) h = moveRedRight(h);
            h.right = delTop(h.right);
            return balance(h);
        }

        private Node<E> moveRedRight(Node<E> h) {
            flipColors(h);
            if (h.left != null && !isRed(h.left.left)) h = rotateRight(h);
            return h;
        }
    }

    public final List<E> reverse() {
        int size = size();
        if (size == 0) return Collections.emptyList();
        LArrayStack<E> stack = LArrayStack.of(size);
        for (int i = 0; i < size; ++i) stack.push(poll());
        List<E> result = new ArrayList<>(size);
        for (int i = 0; i < size; ++i) result.add(stack.pop());
        return result;
    }
}
