package shz.st.bst;

import java.util.Comparator;
import java.util.function.Consumer;
import java.util.function.Supplier;

/**
 * 元素为E类型的外部多路排序树(败者树及胜者树)
 */
public abstract class LWBST<E> {
    protected final Comparator<? super E> comparator;
    protected final E[] leaf;
    protected final int[] tree;
    private final Consumer<E> offer;

    private LWBST(Comparator<? super E> comparator, E[] leaf, int type) {
        if (comparator == null || leaf == null || leaf.length == 0) throw new NullPointerException();
        this.comparator = comparator;
        this.leaf = leaf;
        tree = getTree(type);
        offer = getOffer(type);
    }

    protected abstract int[] getTree(int type);

    protected abstract Consumer<E> getOffer(int type);

    public static <E> LWBST<E> of(Comparator<? super E> comparator, E[] leaf, int type) {
        return new LWBST<E>(comparator, leaf, type) {
            @Override
            protected int[] getTree(int type) {
                int[] tree;
                if (type == 1) {
                    tree = new int[(leaf.length << 1)];
                    for (int i = 0; i < leaf.length; ++i) tree[leaf.length + i] = i;
                    tree[0] = postVisit1(tree, 1);
                } else {
                    tree = new int[(leaf.length << 1) - 1];
                    for (int i = 0; i < leaf.length; ++i) tree[leaf.length - 1 + i] = i;
                    postVisit0(tree, 0);
                }
                return tree;
            }

            private int postVisit1(int[] tree, int i) {
                int left = i << 1, lwinner = -1, rwinner = -1;
                if (left < leaf.length) lwinner = postVisit1(tree, left);
                if (left < leaf.length - 1) rwinner = postVisit1(tree, left + 1);
                int li = lwinner < 0 ? tree[i << 1] : lwinner;
                int ri = rwinner < 0 ? tree[(i << 1) + 1] : rwinner;
                if (leaf[li] == null) {
                    tree[i] = li;
                    return ri;
                }
                if (leaf[ri] == null) {
                    tree[i] = ri;
                    return li;
                }
                if (comparator.compare(leaf[li], leaf[ri]) < 0) {
                    tree[i] = ri;
                    return li;
                }
                tree[i] = li;
                return ri;
            }

            private void postVisit0(int[] tree, int i) {
                int left = (i << 1) + 1;
                if (left < leaf.length - 1) postVisit0(tree, left);
                if (left < leaf.length - 2) postVisit0(tree, left + 1);

                int li = tree[(i << 1) + 1];
                int ri = tree[(i << 1) + 2];

                if (leaf[li] != null || leaf[ri] != null)
                    tree[i] = leaf[li] == null || (leaf[ri] != null && comparator.compare(leaf[li], leaf[ri]) >= 0) ? ri : li;
            }

            @Override
            protected Consumer<E> getOffer(int type) {
                Consumer<E> offer;
                if (type == 1) {
                    offer = e -> {
                        leaf[tree[0]] = e;
                        int parent = (tree[0] + leaf.length) >> 1;
                        int winner = tree[0];
                        while (parent > 0) {
                            if (leaf[tree[parent]] != null && (leaf[winner] == null || comparator.compare(leaf[winner], leaf[tree[parent]]) > 0)) {
                                int loser = tree[parent];
                                tree[parent] = winner;
                                winner = loser;
                            }
                            parent >>= 1;
                        }
                        tree[0] = winner;
                    };
                } else {
                    offer = e -> {
                        leaf[tree[0]] = e;
                        int parent = (tree[0] + leaf.length - 2) >> 1;
                        while (parent >= 0) {
                            int li = tree[(parent << 1) + 1];
                            int ri = tree[(parent << 1) + 2];
                            if (leaf[li] != null || leaf[ri] != null)
                                tree[parent] = leaf[li] == null || (leaf[ri] != null && comparator.compare(leaf[li], leaf[ri]) >= 0) ? ri : li;
                            parent = (parent - 1) >> 1;
                        }
                    };
                }
                return offer;
            }
        };
    }

    public static <E> LWBST<E> of(Comparator<? super E> comparator, E[] leaf) {
        return of(comparator, leaf, 1);
    }

    public final void sort(Supplier<E>[] sources, Consumer<E> consumer) {
        if (sources == null || consumer == null) throw new NullPointerException();
        if (sources.length != leaf.length) throw new IllegalArgumentException();
        while (leaf[tree[0]] != null) {
            consumer.accept(leaf[tree[0]]);
            offer.accept(sources[tree[0]].get());
        }
    }
}
