package shz.st.bst;

/**
 * 红黑二叉查找树（红黑树，2-3树的一种等价定义）
 * <p>
 * 在一棵大小为N的2-3树中，查找和插入操作访问的结点必然不超过lgN个
 * <p>
 * 满足条件：
 * 1：红链接均为左链接
 * 2：没有任何一个结点同时和两条红链接相连
 * 3：该树是完美黑色平衡的，即任意空链接到根结点的路径上的黑链接数量相同
 * 4：根结点总是黑色
 * <p>
 * 一棵大小为N的红黑树的高度不会超过2lgN
 * 一棵大小为N的红黑树中，根结点到任意结点的平均路径长度为～1.00lgN
 */
@SuppressWarnings("unchecked")
public abstract class RedBlackBST<K extends Comparable<K>> {
    protected static abstract class Node {
        public Node left, right;
        public int size = 1;
        public boolean red;

        public Node(boolean red) {
            this.red = red;
        }

        public final <T extends Node> T left() {
            return (T) left;
        }

        public final <T extends Node> T right() {
            return (T) right;
        }
    }

    protected Node root;

    protected RedBlackBST(Node root) {
        if (root == null) throw new NullPointerException();
        this.root = root;
    }

    protected final <T extends Node> T root() {
        return (T) root;
    }

    public final int size() {
        return size(root);
    }

    protected final int size(Node h) {
        return h == null ? 0 : h.size;
    }

    public final boolean isEmpty() {
        return size() == 0;
    }

    public final boolean isLeaf() {
        return size() == 1;
    }

    protected final boolean isRed(Node h) {
        return h != null && h.red;
    }

    protected final <T extends Node> T rotateLeft(Node h) {
        Node x = h.right;
        h.right = x.left;
        x.left = h;
        x.red = h.red;
        h.red = true;
        x.size = h.size;
        h.size = 1 + size(h.left) + size(h.right);
        return (T) x;
    }

    protected final <T extends Node> T rotateRight(Node h) {
        Node x = h.left;
        h.left = x.right;
        x.right = h;
        x.red = h.red;
        h.red = true;
        x.size = h.size;
        h.size = 1 + size(h.left) + size(h.right);
        return (T) x;
    }

    protected final void flipColors(Node h) {
        h.red = true;
        if (h.left != null) h.left.red = false;
        if (h.right != null) h.right.red = false;
    }

    protected abstract K key(Node h);

    public final K min() {
        return root == null ? null : key(min(root));
    }

    protected final <T extends Node> T min(Node h) {
        Node l;
        while ((l = h.left) != null) h = l;
        return (T) h;
    }

    public final K max() {
        return root == null ? null : key(max(root));
    }

    protected final <T extends Node> T max(Node h) {
        Node r;
        while ((r = h.right) != null) h = r;
        return (T) h;
    }

    /**
     * 树的高度
     */
    public final int depth() {
        return depth(root);
    }

    protected final int depth(Node h) {
        if (h == null) return 0;
        return Math.max(depth(h.left), depth(h.right)) + 1;
    }

    /**
     * 返回排名为k的节点
     */
    public final K select(int k) {
        if (k < 1) throw new IllegalArgumentException();
        Node h = select(root, k);
        return h == null ? null : key(h);
    }

    protected final <T extends Node> T select(Node h, int k) {
        while (h != null) {
            int k0 = k - size(h.left) - 1;
            if (k0 == 0) break;
            if (k0 < 0) h = h.left;
            else {
                h = h.right;
                k = k0;
            }
        }
        return (T) h;
    }

    public final void deleteMin() {
        if (root == null) return;
        if (!isRed(root.left) && !isRed(root.right)) root.red = true;
        root = deleteMin(root);
        if (root == null) return;
        if (!isEmpty()) root.red = false;
    }

    protected final <T extends Node> T deleteMin(Node h) {
        if (h.left == null) return h.right();
        if (!isRed(h.left) && !isRed(h.left.left)) h = moveRedLeft(h);
        h.left = deleteMin(h.left);
        return balance(h);
    }

    protected final <T extends Node> T moveRedLeft(Node h) {
        //假设节点h为红色，h.left和H.left.left都为黑色
        //将h.left或者h.left的子节点之一变红
        flipColors(h);
        if (h.right != null && isRed(h.right.left)) {
            h.right = rotateRight(h.right);
            h = rotateLeft(h);
        }
        return (T) h;
    }

    protected final <T extends Node> T balance(Node h) {
        if (isRed(h.right)) h = rotateLeft(h);
        if (isRed(h.right) && !isRed(h.left)) h = rotateLeft(h);
        if (isRed(h.left) && isRed(h.left.left)) h = rotateRight(h);
        if (isRed(h.left) && isRed(h.right)) flipColors(h);
        h.size = size(h.left) + size(h.right) + 1;
        return (T) h;
    }

    public final void deleteMax() {
        if (root == null) return;
        if (!isRed(root.left) && !isRed(root.right)) root.red = true;
        root = deleteMax(root);
        if (root == null) return;
        if (!isEmpty()) root.red = false;
    }

    protected final <T extends Node> T deleteMax(Node h) {
        if (isRed(h.left)) h = rotateRight(h);
        if (h.right == null) return h.left();
        if (!isRed(h.right) && !isRed(h.right.left)) h = moveRedRight(h);
        h.right = deleteMax(h.right);
        return balance(h);
    }

    protected final <T extends Node> T moveRedRight(Node h) {
        //假设节点h为红色，h.right和H.right.left都为黑色
        //将h.right或者h.right的子节点之一变红
        flipColors(h);
        if (h.left != null && !isRed(h.left.left)) h = rotateRight(h);
        return (T) h;
    }
}
