package shz.stack;

import java.util.Iterator;

/**
 * 基于动态数组的栈
 */
public abstract class ArrayStack<E> implements Iterable<E> {
    protected static final int DEFAULT_CAPACITY = 8;
    protected int capacity, size;

    protected ArrayStack(int capacity) {
        if (capacity < 1) throw new IllegalArgumentException();
        this.capacity = capacity;
    }

    public final class ArrayStackIterator implements Iterator<E> {
        private int current = size;

        @Override
        public boolean hasNext() {
            return current > 0;
        }

        @Override
        public E next() {
            return get(--current);
        }
    }

    protected abstract E get(int i);

    @Override
    public Iterator<E> iterator() {
        return new ArrayStackIterator();
    }

    public final int size() {
        return size;
    }

    public final boolean isEmpty() {
        return size == 0;
    }

    protected final void beforePush() {
        if (size == capacity) {
            if (size == Integer.MAX_VALUE) throw new OutOfMemoryError();
            int newCap = capacity >= 64 ? capacity + (capacity >> 1) : capacity << 1;
            resize(newCap < 0 ? Integer.MAX_VALUE : newCap);
        }
    }

    protected abstract void resize(int capacity);

    protected final void afterPop() {
        setNull(size);
        reduce();
    }

    private void reduce() {
        if (capacity > 64) {
            int newCap = size + 1 + (size + 1 >> 1);
            if (newCap + (newCap >> 1) <= capacity) resize(newCap);
        } else if (size + 1 <= capacity >>> 2) resize(capacity >> 1);
    }

    protected abstract void setNull(int i);
}
