package shz.third;

import shz.*;
import shz.msg.ServerFailureMsg;
import shz.tuple.Tuple2;

import javax.net.ssl.SSLContext;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * API请求抽象类
 *
 * @param <D> 请求执行参数
 * @param <T> 响应
 */
public abstract class AbstractApiRequest<D, T extends ApiResponse> implements ApiRequest<D, T> {
    @ApiIgnore
    protected final Class<T> tCls;
    @ApiIgnore
    protected final ApiRequestConfig config;
    @ApiIgnore
    protected final Map<String, String> headers;

    protected AbstractApiRequest() {
        tCls = AccessibleHelp.getParameterizedType(getClass(), AbstractApiRequest.class, "T");
        config = getConfig();
        headers = new HashMap<>();
    }

    protected ApiRequestConfig getConfig() {
        Class<?> cls = getClass();
        ApiConfig apiConfig = cls.getAnnotation(ApiConfig.class);
        ServerFailureMsg.requireNon(apiConfig == null, "API请求类:%s缺少ApiConfig注解", cls.getTypeName());
        ApiRequestConfig config = new ApiRequestConfig();
        config.url = apiConfig.url();
        config.method = apiConfig.method();
        config.proxyHost = apiConfig.proxyHost();
        config.proxyPort = apiConfig.proxyPort();
        config.connectTimeoutMills = apiConfig.connectTimeoutMills();
        config.readTimeoutMills = apiConfig.readTimeoutMills();
        return config;
    }

    @Override
    public final T execute(D data) {
        //获取请求参数
        Map<String, Object> dataMap = dataMap();
        //初始化请求参数
        init(data, dataMap);
        String url = url(dataMap);
        String body = body(dataMap);
        //请求前置方法，可用于日志
        before(url, body);
        try {
            //执行http请求
            String response = read(HttpHelp.requestIs(
                    url, this::sslContext, headers, config.method,
                    body == null ? null : body.getBytes(charset()),
                    config.proxyHost,
                    Integer.parseInt(config.proxyPort),
                    Integer.parseInt(config.connectTimeoutMills),
                    Integer.parseInt(config.readTimeoutMills)
            ));
            //请求后置方法，可用于日志
            after(url, body, response);
            return analysis(response);
        } catch (Exception e) {
            throw PRException.of(e);
        }
    }

    /**
     * 获取请求参数
     */
    private Map<String, Object> dataMap() {
        List<Field> fields = AccessibleHelp.fields(getClass(), f -> !ignoreField(f));
        if (fields.isEmpty()) return Collections.emptyMap();
        Map<String, Object> dataMap = ToMap.get(fields.size()).build();
        for (Field field : fields)
            try {
                String name = field.getName();
                Object obj = field.get(this);

                if (obj == null) {
                    ApiValue apiValue = field.getAnnotation(ApiValue.class);
                    if (apiValue != null) obj = ToObject.parse(apiValue(apiValue.value()), field.getType());
                }

                Tuple2<Boolean, String> key = nameFilter(name, obj);
                if (key == null || Validator.isTrue(key._1)) continue;
                Tuple2<Boolean, Object> value = valueFilter(name, obj);
                if (value == null || Validator.isTrue(value._1)) continue;
                dataMap.put(key._2, value._2);
            } catch (IllegalAccessException ignored) {
            }
        return dataMap.isEmpty() ? Collections.emptyMap() : dataMap;
    }

    protected boolean ignoreField(Field field) {
        return field.isAnnotationPresent(ApiIgnore.class);
    }

    protected String apiValue(String value) {
        return value;
    }

    protected Tuple2<Boolean, String> nameFilter(String name, Object value) {
        return Tuple2.apply(Boolean.FALSE, name);
    }

    protected Tuple2<Boolean, Object> valueFilter(String name, Object value) {
        return Tuple2.apply(value == null, value);
    }

    protected void init(D data, Map<String, Object> dataMap) {
    }

    protected String url(Map<String, Object> dataMap) {
        return config.url;
    }

    protected SSLContext sslContext() {
        return null;
    }

    protected String body(Map<String, Object> dataMap) {
        return null;
    }

    protected Charset charset() {
        return StandardCharsets.UTF_8;
    }

    protected void before(String url, String body) {
    }

    protected String read(InputStream is) {
        return IOHelp.read(IOHelp.getBr(is, charset()));
    }

    protected void after(String url, String body, String response) {
    }

    protected abstract T analysis(String response);
}
