package shz.xml;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import shz.*;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import java.io.*;
import java.lang.reflect.Field;
import java.util.*;
import java.util.function.BiFunction;
import java.util.function.Function;

public final class W3cXmlHelp {
    private W3cXmlHelp() {
        throw new IllegalStateException();
    }

    public static Document getXmlDocument(InputSource in) {
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true);
        try {
            DocumentBuilder db = dbf.newDocumentBuilder();
            return db.parse(in);
        } catch (ParserConfigurationException | SAXException | IOException e) {
            throw PRException.of(e);
        }
    }

    public static Document getXmlDocument(String uri) {
        return getXmlDocument(new InputSource(uri));
    }

    public static Document getXmlDocument(File file) {
        return getXmlDocument(new InputSource(file.toURI().toASCIIString()));
    }

    public static Document getXmlDocument(InputStream is, String systemId) {
        InputSource in = new InputSource(is);
        in.setSystemId(systemId);
        return getXmlDocument(in);
    }

    public static Document getXmlDocument(InputStream is) {
        return getXmlDocument(new InputSource(is));
    }

    public static Document getXmlDocument(Reader reader) {
        return getXmlDocument(new InputSource(reader));
    }

    public static Node getXmlNode(Document document) {
        return document.getDocumentElement();
    }

    public static Map<String, Node> getNodeChildNodes(Node node) {
        NodeList nodelist = node.getChildNodes();
        if (nodelist.getLength() == 0) return Collections.emptyMap();
        Map<String, Node> map = ToMap.get(nodelist.getLength()).build();
        for (int i = 0; i < nodelist.getLength(); ++i)
            map.put(getKeyNum(nodelist.item(i).getNodeName(), 0), nodelist.item(i));
        return map.isEmpty() ? Collections.emptyMap() : map;
    }

    private static final String NAME_SEPARATOR = "--";
    private static final String NUM_SEPARATOR = "N";

    private static String getKeyNum(String k, int num) {
        k = k.trim();
        int length = k.length() - (NUM_SEPARATOR.length() << 1) - 3;
        int indexOf = k.indexOf(NUM_SEPARATOR, Math.max(length, 0));
        indexOf = indexOf < 0 ? k.length() : indexOf;
        return k.substring(0, indexOf) + NUM_SEPARATOR + num + NUM_SEPARATOR;
    }

    public static Map<String, Node> getNodeAllChildNodes(Map<String, Node> map, String k, Node v, boolean pc) {
        NodeList nodelist = v.getChildNodes();
        if (nodelist.getLength() != 0) {
            if (pc) map.put(getKey(k, map, 0), v);
            for (int i = 0; i < nodelist.getLength(); ++i)
                getNodeAllChildNodes(map, getKey(k + NAME_SEPARATOR + nodelist.item(i).getNodeName(), map, 0), nodelist.item(i), pc);
        } else map.put(getKey(k, map, 0), v);
        return map.isEmpty() ? Collections.emptyMap() : map;
    }

    public static Map<String, Node> getNodeAllChildNodes(Map<String, Node> map, String k, Node v) {
        return getNodeAllChildNodes(map, k, v, false);
    }

    private static String getKey(String k, Map<String, Node> map, int num) {
        String keyNum = getKeyNum(k, num);
        return map.containsKey(keyNum) ? getKey(k, map, num + 1) : keyNum;
    }

    public static Map<String, Node> getXmlMap(Document document, boolean pc) {
        Map<String, Node> map = new HashMap<>();
        getNodeChildNodes(getXmlNode(document)).forEach((k, v) -> getNodeAllChildNodes(map, k, v, pc));
        return map.isEmpty() ? Collections.emptyMap() : map;
    }

    public static Map<String, Node> getXmlMap(Document document) {
        return getXmlMap(document, false);
    }

    public static Map<String, Node> getXmlMap(String uri, boolean pc) {
        return getXmlMap(getXmlDocument(uri), pc);
    }

    public static Map<String, Node> getXmlMap(String uri) {
        return getXmlMap(getXmlDocument(uri));
    }

    public static Map<String, Node> getXmlMap(File file, boolean pc) {
        return getXmlMap(getXmlDocument(file), pc);
    }

    public static Map<String, Node> getXmlMap(File file) {
        return getXmlMap(getXmlDocument(file));
    }

    public static List<Node> getXmlNodes(Map<String, Node> map, String names) {
        List<Node> result = new LinkedList<>();
        String[] allNames = names.trim().split(" +");
        String name = allNames[allNames.length - 1];
        String[] parentNames = new String[allNames.length - 1];
        System.arraycopy(allNames, 0, parentNames, 0, parentNames.length);
        int[] index = new int[parentNames.length];
        map.forEach((k, v) -> {
            String p_s = k.trim().replaceAll(NUM_SEPARATOR + "\\d+" + NUM_SEPARATOR, "");
            if (p_s.endsWith(name)) {
                if (parentNames.length == 0) result.add(v);
                else {
                    index[index.length - 1] = -1;
                    for (int i = 0; i < index.length; i++) if ((index[i] = p_s.indexOf(parentNames[i])) == -1) break;
                    if (index[index.length - 1] != -1 && isMinToMax(index, 0)) result.add(v);
                }
            }
        });
        return result.isEmpty() ? Collections.emptyList() : result;
    }

    private static boolean isMinToMax(int[] index, int n) {
        if (n + 1 < index.length)
            if (index[n] < index[n + 1]) return isMinToMax(index, n + 1);
            else return false;
        return true;
    }

    public static List<Node> getXmlNodes(String xml, boolean pc, String name) {
        return getXmlNodes(getXmlMap(xml, pc), name);
    }

    static final class W3cXmlSetFieldStrategy extends ToObject.SetFieldStrategy {
        @Override
        public Function<Field, BiFunction<Function<Field, String>, Class<?>, Collection<?>>> supCollection() {
            return t -> (u, r) -> ToSet.collect(keys(t).parallelStream().map(k -> executor(k, r)));
        }

        @Override
        public Function<Field, BiFunction<Function<Field, String>, Class<?>[], Map<?, ?>>> supMap() {
            return t -> (u, r) -> ToMap.collect(keys(t).parallelStream(), k -> executor(k, r[0]), k -> executor(k, r[1]));
        }

        @Override
        public Function<Field, BiFunction<Function<Field, String>, Class<?>, Object>> supObject() {
            return t -> (u, r) -> executor(u.apply(t), r);
        }

        Map<String, Node> map;

        @Override
        protected Collection<String> keys(Field field) {
            List<String> keys = new ArrayList<>(map.size());
            String key = field.getName();
            int num = 0;
            String keyNum;
            while (map.containsKey(keyNum = getKeyNum(key, num))) {
                keys.add(keyNum);
                ++num;
            }
            return keys;
        }

        @Override
        protected Object executor(String key, Class<?> cls) {
            return ToObject.fromMap(cls, map, f -> key(key + NAME_SEPARATOR + f.getName()), n -> n == null ? null : ((Node) n).getTextContent(), this);
        }

        private String key(String key) {
            String keyNum = getKeyNum(key, 0);
            return map.containsKey(keyNum) ? keyNum : null;
        }

        <T> T fromMap(Class<? extends T> cls, Map<String, Node> map) {
            this.map = map;
            return ToObject.fromMap(cls, map, f -> key(f.getName()), n -> n == null ? null : ((Node) n).getTextContent(), this);
        }

        <T> T fromMap(T t, Map<String, Node> map) {
            this.map = map;
            return ToObject.fromMap(t, map, f -> key(f.getName()), n -> n == null ? null : ((Node) n).getTextContent(), this);
        }
    }

    private static final W3cXmlSetFieldStrategy STRATEGY = new W3cXmlSetFieldStrategy();

    public static <T> T toObject(Class<? extends T> cls, String uri) {
        return ((W3cXmlSetFieldStrategy) STRATEGY.clone()).fromMap(cls, getXmlMap(uri, true));
    }

    public static <T> T toObject(T t, String uri) {
        return ((W3cXmlSetFieldStrategy) STRATEGY.clone()).fromMap(t, getXmlMap(uri, true));
    }

    public static <T> T toObject(Class<T> cls, File file) {
        return ((W3cXmlSetFieldStrategy) STRATEGY.clone()).fromMap(cls, getXmlMap(file, true));
    }

    public static <T> T toObject(T t, File file) {
        return ((W3cXmlSetFieldStrategy) STRATEGY.clone()).fromMap(t, getXmlMap(file, true));
    }

    public static String getValueFromXmlStrByKey(String xmlStr, String key) {
        if (Validator.isAnyBlank(xmlStr, key)) return null;
        return ((Element) getXmlNode(getXmlDocument(new StringReader(xmlStr)))).getElementsByTagName(key).item(0).getTextContent();
    }

    public static Map<String, String> getValuesFromXmlStrByKeys(String xmlStr, String... keys) {
        if (Validator.isBlank(xmlStr) || Validator.isEmpty(keys)) return Collections.emptyMap();
        Element element = (Element) getXmlNode(getXmlDocument(new StringReader(xmlStr)));
        return ToMap.explicitCollect(Arrays.stream(keys), Function.identity(), k -> element.getElementsByTagName(k).item(0).getTextContent(), keys.length);
    }
}