/*
 * Decompiled with CFR 0.152.
 */
package jadx.zip.parser;

import jadx.zip.IZipEntry;
import jadx.zip.IZipParser;
import jadx.zip.ZipContent;
import jadx.zip.ZipReaderFlags;
import jadx.zip.ZipReaderOptions;
import jadx.zip.fallback.FallbackZipParser;
import jadx.zip.io.ByteBufferBackedInputStream;
import jadx.zip.io.LimitedInputStream;
import jadx.zip.parser.JadxZipEntry;
import jadx.zip.parser.ZipDeflate;
import jadx.zip.security.IJadxZipSecurity;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.channels.FileChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JadxZipParser
implements IZipParser {
    private static final Logger LOG = LoggerFactory.getLogger(JadxZipParser.class);
    private static final byte LOCAL_FILE_HEADER_START = 80;
    private static final int LOCAL_FILE_HEADER_SIGN = 67324752;
    private static final int CD_SIGN = 33639248;
    private static final int END_OF_CD_SIGN = 101010256;
    private final File zipFile;
    private final ZipReaderOptions options;
    private final IJadxZipSecurity zipSecurity;
    private final Set<ZipReaderFlags> flags;
    private final boolean verify;
    private final boolean useLimitedDataStream;
    private RandomAccessFile file;
    private FileChannel fileChannel;
    private ByteBuffer byteBuffer;
    private int endOfCDStart = -2;
    @Nullable
    private ZipContent fallbackZipContent;

    public JadxZipParser(File zipFile, ZipReaderOptions options) {
        this.zipFile = zipFile;
        this.options = options;
        this.zipSecurity = options.getZipSecurity();
        this.flags = options.getFlags();
        this.verify = options.getFlags().contains((Object)ZipReaderFlags.REPORT_TAMPERING);
        this.useLimitedDataStream = this.zipSecurity.useLimitedDataStream();
    }

    @Override
    public ZipContent open() throws IOException {
        this.load();
        try {
            int maxEntriesCount = this.zipSecurity.getMaxEntriesCount();
            if (maxEntriesCount == -1) {
                maxEntriesCount = Integer.MAX_VALUE;
            }
            List<IZipEntry> entries = this.flags.contains((Object)ZipReaderFlags.IGNORE_CENTRAL_DIR_ENTRIES) ? this.searchLocalFileHeaders(maxEntriesCount) : this.loadFromCentralDirs(maxEntriesCount);
            return new ZipContent(this, entries);
        }
        catch (Exception e) {
            if (this.flags.contains((Object)ZipReaderFlags.DONT_USE_FALLBACK)) {
                throw new IOException("Failed to open zip: " + this.zipFile + ", error: " + e.getMessage(), e);
            }
            LOG.warn("Zip open failed, switching to fallback parser, zip: {}", (Object)this.zipFile, (Object)e);
            return this.initFallbackParser();
        }
    }

    public boolean canOpen() {
        try {
            this.load();
            int eocdStart = this.searchEndOfCDStart();
            ByteBuffer buf = this.byteBuffer;
            buf.position(eocdStart + 4);
            int diskNum = JadxZipParser.readU2(buf);
            return diskNum != 65535;
        }
        catch (Exception e) {
            LOG.warn("Jadx parser can't open zip file: {}", (Object)this.zipFile, (Object)e);
            return false;
        }
    }

    private boolean isValidEntry(JadxZipEntry zipEntry) {
        boolean validEntry = this.zipSecurity.isValidEntry(zipEntry);
        if (!validEntry) {
            LOG.warn("Zip entry '{}' is invalid and excluded from processing", (Object)zipEntry);
        }
        return validEntry;
    }

    private void load() throws IOException {
        if (this.byteBuffer != null) {
            return;
        }
        this.file = new RandomAccessFile(this.zipFile, "r");
        long size = this.file.length();
        if (size >= Integer.MAX_VALUE) {
            throw new IOException("Zip file is too big");
        }
        int fileLen = (int)size;
        if (fileLen < 0x6400000) {
            byte[] bytes = new byte[fileLen];
            this.file.readFully(bytes);
            this.byteBuffer = ByteBuffer.wrap(bytes).asReadOnlyBuffer();
            this.file.close();
            this.file = null;
        } else {
            this.fileChannel = this.file.getChannel();
            this.byteBuffer = this.fileChannel.map(FileChannel.MapMode.READ_ONLY, 0L, this.fileChannel.size());
        }
        this.byteBuffer.order(ByteOrder.LITTLE_ENDIAN);
    }

    private List<IZipEntry> searchLocalFileHeaders(int maxEntriesCount) {
        ArrayList<IZipEntry> entries = new ArrayList<IZipEntry>();
        while (true) {
            int start;
            if ((start = this.searchEntryStart()) == -1) {
                return entries;
            }
            JadxZipEntry zipEntry = this.loadFileEntry(start);
            if (!this.isValidEntry(zipEntry)) continue;
            entries.add(zipEntry);
            if (entries.size() > maxEntriesCount) break;
        }
        throw new IllegalStateException("Max entries count limit exceeded: " + entries.size());
    }

    private List<IZipEntry> loadFromCentralDirs(int maxEntriesCount) throws IOException {
        int eocdStart = this.searchEndOfCDStart();
        if (eocdStart < 0) {
            throw new RuntimeException("End of central directory not found");
        }
        ByteBuffer buf = this.byteBuffer;
        buf.position(eocdStart + 10);
        int entriesCount = JadxZipParser.readU2(buf);
        buf.position(eocdStart + 16);
        int cdOffset = buf.getInt();
        if (entriesCount > maxEntriesCount) {
            throw new IllegalStateException("Max entries count limit exceeded: " + entriesCount);
        }
        ArrayList<IZipEntry> entries = new ArrayList<IZipEntry>(entriesCount);
        buf.position(cdOffset);
        for (int i = 0; i < entriesCount; ++i) {
            JadxZipEntry zipEntry = this.loadCDEntry();
            if (!this.isValidEntry(zipEntry)) continue;
            entries.add(zipEntry);
        }
        return entries;
    }

    private JadxZipEntry loadCDEntry() {
        ByteBuffer buf = this.byteBuffer;
        int start = buf.position();
        buf.position(start + 28);
        int fileNameLen = JadxZipParser.readU2(buf);
        int extraFieldLen = JadxZipParser.readU2(buf);
        int commentLen = JadxZipParser.readU2(buf);
        buf.position(start + 42);
        int fileEntryStart = buf.getInt();
        int entryEnd = start + 46 + fileNameLen + extraFieldLen + commentLen;
        JadxZipEntry entry = this.loadFileEntry(fileEntryStart);
        if (this.verify) {
            JadxZipParser.compareCDAndLFH(buf, start, entry);
        }
        if (!entry.isSizesValid()) {
            entry = this.fixEntryFromCD(entry, start);
        }
        buf.position(entryEnd);
        return entry;
    }

    private JadxZipEntry fixEntryFromCD(JadxZipEntry entry, int start) {
        ByteBuffer buf = this.byteBuffer;
        buf.position(start + 10);
        int comprMethod = JadxZipParser.readU2(buf);
        buf.position(start + 20);
        int comprSize = buf.getInt();
        int unComprSize = buf.getInt();
        return new JadxZipEntry(this, entry.getName(), start, entry.getDataStart(), comprMethod, comprSize, unComprSize);
    }

    private static void compareCDAndLFH(ByteBuffer buf, int start, JadxZipEntry entry) {
        buf.position(start + 10);
        int comprMethod = JadxZipParser.readU2(buf);
        if (comprMethod != entry.getCompressMethod()) {
            LOG.warn("Compression method differ in CD {} and LFH {} for {}", new Object[]{comprMethod, entry.getCompressMethod(), entry});
        }
        buf.position(start + 20);
        int comprSize = buf.getInt();
        int unComprSize = buf.getInt();
        if ((long)comprSize != entry.getCompressedSize()) {
            LOG.warn("Compressed size differ in CD {} and LFH {} for {}", new Object[]{comprSize, entry.getCompressedSize(), entry});
        }
        if ((long)unComprSize != entry.getUncompressedSize()) {
            LOG.warn("Uncompressed size differ in CD {} and LFH {} for {}", new Object[]{unComprSize, entry.getUncompressedSize(), entry});
        }
    }

    private JadxZipEntry loadFileEntry(int start) {
        ByteBuffer buf = this.byteBuffer;
        buf.position(start + 8);
        int comprMethod = JadxZipParser.readU2(buf);
        buf.position(start + 18);
        int comprSize = buf.getInt();
        int unComprSize = buf.getInt();
        int fileNameLen = JadxZipParser.readU2(buf);
        int extraFieldLen = JadxZipParser.readU2(buf);
        String fileName = JadxZipParser.readString(buf, fileNameLen);
        int dataStart = start + 30 + fileNameLen + extraFieldLen;
        buf.position(dataStart + comprSize);
        return new JadxZipEntry(this, fileName, start, dataStart, comprMethod, comprSize, unComprSize);
    }

    private int searchEndOfCDStart() throws IOException {
        if (this.endOfCDStart != -2) {
            return this.endOfCDStart;
        }
        ByteBuffer buf = this.byteBuffer;
        int pos = buf.limit() - 22;
        int minPos = Math.max(0, pos - 65535);
        do {
            buf.position(pos);
            int sign = buf.getInt();
            if (sign != 101010256) continue;
            this.endOfCDStart = pos;
            return pos;
        } while (--pos >= minPos);
        throw new IOException("End of central directory record not found");
    }

    private int searchEntryStart() {
        int start;
        ByteBuffer buf = this.byteBuffer;
        while (true) {
            if ((start = buf.position()) + 4 > buf.limit()) {
                return -1;
            }
            byte b = buf.get();
            if (b != 80) continue;
            buf.position(start);
            int sign = buf.getInt();
            if (sign == 67324752) break;
        }
        return start;
    }

    synchronized InputStream getInputStream(JadxZipEntry entry) {
        InputStream stream;
        if (this.verify) {
            JadxZipParser.verifyEntry(entry);
        }
        if (entry.getCompressMethod() == 8) {
            try {
                stream = ZipDeflate.decompressEntryToStream(this.byteBuffer, entry);
            }
            catch (Exception e) {
                this.entryParseFailed(entry, e);
                return this.useFallbackParser(entry).getInputStream();
            }
        } else {
            stream = JadxZipParser.bufferToStream(this.byteBuffer, entry.getDataStart(), (int)entry.getUncompressedSize());
        }
        if (this.useLimitedDataStream) {
            return new LimitedInputStream(stream, entry.getUncompressedSize());
        }
        return stream;
    }

    synchronized byte[] getBytes(JadxZipEntry entry) {
        if (this.verify) {
            JadxZipParser.verifyEntry(entry);
        }
        if (entry.getCompressMethod() == 8) {
            try {
                return ZipDeflate.decompressEntryToBytes(this.byteBuffer, entry);
            }
            catch (Exception e) {
                this.entryParseFailed(entry, e);
                return this.useFallbackParser(entry).getBytes();
            }
        }
        return JadxZipParser.bufferToBytes(this.byteBuffer, entry.getDataStart(), (int)entry.getUncompressedSize());
    }

    private static void verifyEntry(JadxZipEntry entry) {
        int compressMethod = entry.getCompressMethod();
        if (compressMethod == 0) {
            if (entry.getCompressedSize() != entry.getUncompressedSize()) {
                LOG.warn("Not equal sizes for STORE method: compressed: {}, uncompressed: {}, entry: {}", new Object[]{entry.getCompressedSize(), entry.getUncompressedSize(), entry});
            }
        } else if (compressMethod != 8) {
            LOG.warn("Unknown compress method: {} in entry: {}", (Object)compressMethod, (Object)entry);
        }
    }

    private void entryParseFailed(JadxZipEntry entry, Exception e) {
        if (this.isEncrypted(entry)) {
            throw new RuntimeException("Entry is encrypted, failed to decompress: " + entry, e);
        }
        if (this.flags.contains((Object)ZipReaderFlags.DONT_USE_FALLBACK)) {
            throw new RuntimeException("Failed to decompress zip entry: " + entry + ", error: " + e.getMessage(), e);
        }
        LOG.warn("Entry '{}' parse failed, switching to fallback parser", (Object)entry, (Object)e);
    }

    private IZipEntry useFallbackParser(JadxZipEntry entry) {
        LOG.debug("useFallbackParser used for {}", (Object)entry);
        IZipEntry zipEntry = this.initFallbackParser().searchEntry(entry.getName());
        if (zipEntry == null) {
            throw new RuntimeException("Fallback parser can't find entry: " + entry);
        }
        return zipEntry;
    }

    private ZipContent initFallbackParser() {
        if (this.fallbackZipContent == null) {
            try {
                this.fallbackZipContent = new FallbackZipParser(this.zipFile, this.options).open();
            }
            catch (Exception e) {
                throw new RuntimeException("Fallback parser failed to open file: " + this.zipFile, e);
            }
        }
        return this.fallbackZipContent;
    }

    private boolean isEncrypted(JadxZipEntry entry) {
        int flags = this.readFlags(entry);
        return (flags & 1) != 0;
    }

    private int readFlags(JadxZipEntry entry) {
        ByteBuffer buf = this.byteBuffer;
        buf.position(entry.getEntryStart() + 6);
        return JadxZipParser.readU2(buf);
    }

    static byte[] bufferToBytes(ByteBuffer buf, int start, int size) {
        byte[] data = new byte[size];
        buf.position(start);
        buf.get(data);
        return data;
    }

    static InputStream bufferToStream(ByteBuffer buf, int start, int size) {
        buf.position(start);
        ByteBuffer streamBuf = buf.slice();
        streamBuf.limit(size);
        return new ByteBufferBackedInputStream(streamBuf);
    }

    private static int readU2(ByteBuffer buf) {
        return buf.getShort() & 0xFFFF;
    }

    private static String readString(ByteBuffer buf, int fileNameLen) {
        byte[] bytes = new byte[fileNameLen];
        buf.get(bytes);
        return new String(bytes, StandardCharsets.UTF_8);
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.fileChannel != null) {
                this.fileChannel.close();
            }
            if (this.file != null) {
                this.file.close();
            }
            if (this.fallbackZipContent != null) {
                this.fallbackZipContent.close();
            }
        }
        finally {
            this.fileChannel = null;
            this.file = null;
            this.byteBuffer = null;
            this.endOfCDStart = -2;
            this.fallbackZipContent = null;
        }
    }

    public File getZipFile() {
        return this.zipFile;
    }

    public String toString() {
        return "JadxZipParser{" + this.zipFile + "}";
    }
}

