package io.github.thoroldvix.api;

import java.util.Set;

/**
 * Represents a single transcript for a YouTube video, including its metadata.
 * <p>
 * This interface provides methods to access the transcript content and to perform translations into different languages.
 * Individual transcripts can be obtained through the {@link TranscriptList} class.
 * </p>
 */
public interface Transcript {

    /**
     * Retrieves the content of the transcript.
     *
     * @return The content of the transcript as a {@link TranscriptContent} object.
     * @throws TranscriptRetrievalException If the transcript content cannot be retrieved.
     */
    TranscriptContent fetch() throws TranscriptRetrievalException;

    /**
     * Gets the video id of the transcript.
     *
     * @return The video id as a {@link String}.
     */
    String getVideoId();

    /**
     * Gets the language of the transcript.
     *
     * @return The language as a {@link String}.
     */
    String getLanguage();

    /**
     * Gets the language code of the transcript.
     *
     * @return The language code as a {@link String}.
     */
    String getLanguageCode();

    /**
     * Returns API URL which needs to be called to fetch transcript content.
     *
     * @return {@link String} API URL to fetch transcript content
     */
    String getApiUrl();

    /**
     * Determines if the transcript was automatically generated by YouTube.
     *
     * @return {@code true} if the transcript was automatically generated; {@code false} otherwise.
     */
    boolean isGenerated();

    /**
     * Lists all available translation languages for the transcript.
     *
     * @return A set of language codes representing available translation languages.
     */
    Set<String> getTranslationLanguages();

    /**
     * Indicates whether the transcript can be translated.
     *
     * @return {@code true} if the transcript is translatable; {@code false} otherwise.
     */
    boolean isTranslatable();

    /**
     * Translates the transcript into the specified language.
     *
     * @param languageCode The language code to which the transcript should be translated.
     * @return A {@link Transcript} representing the translated transcript.
     * @throws TranscriptRetrievalException If the transcript cannot be translated.
     */
    Transcript translate(String languageCode) throws TranscriptRetrievalException;
}
