/*
 * Decompiled with CFR 0.152.
 */
package de.janno.evaluator.dice;

import com.google.common.collect.ImmutableList;
import de.janno.evaluator.dice.BracketPair;
import de.janno.evaluator.dice.ExpressionException;
import de.janno.evaluator.dice.Operator;
import de.janno.evaluator.dice.Parameters;
import de.janno.evaluator.dice.Token;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import lombok.NonNull;

public class Tokenizer {
    private final ImmutableList<TokenBuilder> tokenBuilders;
    private final String escapeCharacter;

    public Tokenizer(Parameters parameters) {
        this.escapeCharacter = parameters.getEscapeBrackets().stream().map(BracketPair::toString).collect(Collectors.joining(" or "));
        ImmutableList.Builder builder = ImmutableList.builder();
        Stream.concat(parameters.getExpressionBrackets().stream(), parameters.getFunctionBrackets().stream()).distinct().forEach(c -> {
            builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(c.getOpen()), s -> Token.openTokenOf(c)));
            builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(c.getClose()), s -> Token.closeTokenOf(c)));
        });
        parameters.getFunctions().forEach(function -> builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(function.getName()), s -> Token.of(function))));
        parameters.getOperators().forEach(operator -> builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(operator.getName()), s -> Token.of(operator))));
        builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(parameters.getSeparator()), s -> Token.separator()));
        parameters.getEscapeBrackets().forEach(b -> builder.add((Object)new TokenBuilder(Tokenizer.buildEscapeBracketsRegex(b), s -> Token.of(s.substring(1, s.length() - 1)))));
        builder.add((Object)new TokenBuilder("[0-9]+", s -> {
            try {
                Integer.parseInt(s);
                return Token.of(s);
            }
            catch (Throwable t) {
                throw new ExpressionException("The number '%s' was to big".formatted(s));
            }
        }));
        this.tokenBuilders = builder.build();
        List<String> duplicateRegex = this.tokenBuilders.stream().collect(Collectors.groupingBy(TokenBuilder::regex)).entrySet().stream().filter(e -> ((List)e.getValue()).size() > 1).map(Map.Entry::getKey).toList();
        if (!duplicateRegex.isEmpty()) {
            throw new IllegalArgumentException("The following regex for tokenizing where used more then once: " + duplicateRegex);
        }
    }

    private static String buildEscapeBracketsRegex(BracketPair bracketPair) {
        return String.format("%s.*?%s", Tokenizer.escapeForRegexAndAddCaseInsensitivity(bracketPair.getOpen()), Tokenizer.escapeForRegexAndAddCaseInsensitivity(bracketPair.getClose()));
    }

    private static String escapeForRegexAndAddCaseInsensitivity(String in) {
        return "(?i)\\Q%s\\E(?-i)".formatted(in);
    }

    public List<Token> tokenize(String input) throws ExpressionException {
        Optional<Match> currentMatch;
        ArrayList<Token> preTokens = new ArrayList<Token>();
        String current = input.trim();
        do {
            if (!(currentMatch = this.getBestMatch(current)).isPresent()) continue;
            Match match = currentMatch.get();
            Token token = match.token();
            preTokens.add(token);
            current = current.substring(match.match().length()).trim();
        } while (currentMatch.isPresent());
        if (!current.isEmpty()) {
            throw new ExpressionException("No matching operator for '%s', non-functional text and value names must to be surrounded by %s".formatted(current, this.escapeCharacter));
        }
        return this.setOperatorType(preTokens);
    }

    private List<Token> setOperatorType(List<Token> in) throws ExpressionException {
        ImmutableList.Builder builder = ImmutableList.builder();
        boolean lastOperatorWasUnaryLeft = false;
        for (int i = 0; i < in.size(); ++i) {
            Token token = in.get(i);
            if (token.getOperator().isPresent()) {
                Token left = i == 0 ? null : in.get(i - 1);
                Token right = i == in.size() - 1 ? null : in.get(i + 1);
                Operator.OperatorType type = this.determineAndValidateOperatorType(token.getOperator().get(), left, right, lastOperatorWasUnaryLeft);
                builder.add((Object)Token.of(token.getOperator().get(), type));
                lastOperatorWasUnaryLeft = type == Operator.OperatorType.UNARY && token.getOperator().get().getAssociativityForOperantType(Operator.OperatorType.UNARY) == Operator.Associativity.LEFT;
                continue;
            }
            builder.add((Object)token);
            lastOperatorWasUnaryLeft = false;
        }
        return builder.build();
    }

    private Operator.OperatorType determineAndValidateOperatorType(@NonNull Operator operator, @Nullable Token left, @Nullable Token right, boolean lastOperatorWasUnaryLeft) throws ExpressionException {
        boolean rightLiteralOrBracket;
        if (operator == null) {
            throw new NullPointerException("operator is marked non-null but is null");
        }
        boolean leftLiteralOrBracket = left != null && (left.getLiteral().isPresent() || left.isCloseBracket() || left.getOperator().isPresent() && lastOperatorWasUnaryLeft);
        boolean bl = rightLiteralOrBracket = right != null && (right.getLiteral().isPresent() || right.isOpenBracket() || right.getOperator().isPresent() && right.getOperator().get().getAssociativityForOperantType(Operator.OperatorType.UNARY) == Operator.Associativity.RIGHT || right.getFunction().isPresent());
        if (leftLiteralOrBracket && rightLiteralOrBracket) {
            if (!operator.supportBinaryOperation()) {
                throw new ExpressionException("Operator %s does not support binary operations".formatted(operator.getName()));
            }
            return Operator.OperatorType.BINARY;
        }
        if (!operator.supportUnaryOperation()) {
            throw new ExpressionException("Operator %s does not support unary operations".formatted(operator.getName()));
        }
        Operator.Associativity operatorAssociativity = operator.getAssociativityForOperantType(Operator.OperatorType.UNARY);
        if (operatorAssociativity == Operator.Associativity.LEFT && !leftLiteralOrBracket) {
            throw new ExpressionException("Operator %s has left associativity but the left value was: %s".formatted(operator.getName(), Optional.ofNullable(left).map(Object::toString).orElse("empty")));
        }
        if (operatorAssociativity == Operator.Associativity.RIGHT && !rightLiteralOrBracket) {
            throw new ExpressionException("Operator %s has right associativity but the right value was: %s".formatted(operator.getName(), Optional.ofNullable(right).map(Object::toString).orElse("empty")));
        }
        return Operator.OperatorType.UNARY;
    }

    private Optional<Match> getBestMatch(String input) throws ExpressionException {
        List<Match> allMatches = this.getAllMatches(input);
        int maxLength = allMatches.stream().mapToInt(Match::length).max().orElse(0);
        List<Match> maxLengthMatches = allMatches.stream().filter(m -> m.length() == maxLength).toList();
        if (maxLengthMatches.isEmpty()) {
            return Optional.empty();
        }
        if (maxLengthMatches.size() > 1) {
            throw new IllegalStateException("More then one operator matched the input %s: %s".formatted(input, maxLengthMatches.stream().map(Match::token).map(Token::toString).toList()));
        }
        return Optional.of(maxLengthMatches.get(0));
    }

    private List<Match> getAllMatches(String input) throws ExpressionException {
        ImmutableList.Builder matchBuilder = ImmutableList.builder();
        for (TokenBuilder tokenBuilder : this.tokenBuilders) {
            Optional<Match> firstMatch = this.getFirstMatch(input, tokenBuilder);
            firstMatch.ifPresent(arg_0 -> ((ImmutableList.Builder)matchBuilder).add(arg_0));
        }
        return matchBuilder.build();
    }

    private Optional<Match> getFirstMatch(String input, TokenBuilder tokenBuilder) throws ExpressionException {
        Matcher matcher = tokenBuilder.pattern().matcher(input);
        if (matcher.find()) {
            String matchGroup = matcher.group().trim();
            return Optional.of(new Match(matcher.start(), matchGroup, tokenBuilder.toToken().apply(matchGroup)));
        }
        return Optional.empty();
    }

    private record TokenBuilder(String regex, ToToken toToken) {
        Pattern pattern() {
            return Pattern.compile("^\\s*%s\\s*".formatted(this.regex));
        }
    }

    private static interface ToToken {
        public Token apply(String var1) throws ExpressionException;
    }

    private record Match(int start, String match, Token token) {
        public int length() {
            return this.match.length();
        }
    }
}

