/*
 * Decompiled with CFR 0.152.
 */
package de.janno.evaluator.dice;

import com.google.common.collect.ImmutableList;
import de.janno.evaluator.dice.BracketPair;
import de.janno.evaluator.dice.ExpressionException;
import de.janno.evaluator.dice.ExpressionPosition;
import de.janno.evaluator.dice.Function;
import de.janno.evaluator.dice.Operator;
import de.janno.evaluator.dice.Parameters;
import de.janno.evaluator.dice.Token;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nullable;
import lombok.NonNull;

public class Tokenizer {
    private static final String ALL_NUMBER_REGEX = "\\d+\\.?\\d*";
    private static final Pattern SMALL_DECIMAL_PATTERN = Pattern.compile("\\d{1,9}\\.\\d{1,9}");
    private static final Pattern SMALL_INTEGER_PATTERN = Pattern.compile("\\d{1,9}");
    private final ImmutableList<TokenBuilder> tokenBuilders;
    private final String escapeCharacter;
    private final ImmutableList<Pattern> allOperatorAndFunctionNamePatterns;

    public Tokenizer(Parameters parameters) {
        this.escapeCharacter = parameters.getEscapeBrackets().stream().map(BracketPair::toString).collect(Collectors.joining(" or "));
        ImmutableList.Builder builder = ImmutableList.builder();
        Stream.concat(parameters.getExpressionBrackets().stream(), parameters.getFunctionBrackets().stream()).distinct().forEach(c -> {
            builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(c.getOpen()), expressionPosition -> Token.openTokenOf(c, expressionPosition), false));
            builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(c.getClose()), expressionPosition -> Token.closeTokenOf(c, expressionPosition), false));
        });
        parameters.getFunctions().forEach(function -> builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(function.getName()), expressionPosition -> Token.of(function, expressionPosition), false)));
        parameters.getOperators().forEach(operator -> builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(operator.getName()), expressionPosition -> Token.of(operator, expressionPosition), false)));
        builder.add((Object)new TokenBuilder(Tokenizer.escapeForRegexAndAddCaseInsensitivity(parameters.getSeparator()), Token::separator, false));
        parameters.getEscapeBrackets().forEach(b -> builder.add((Object)new TokenBuilder(Tokenizer.buildEscapeBracketsRegex(b), expressionPosition -> Token.of(expressionPosition.getValue().substring(1, expressionPosition.getValue().length() - 1), expressionPosition), true)));
        builder.add((Object)new TokenBuilder(ALL_NUMBER_REGEX, expressionPosition -> {
            if (SMALL_INTEGER_PATTERN.matcher(expressionPosition.getValue()).matches() || SMALL_DECIMAL_PATTERN.matcher(expressionPosition.getValue()).matches()) {
                return Token.of(expressionPosition.getValue(), expressionPosition);
            }
            throw new ExpressionException("The number '%s' is too big".formatted(expressionPosition.getValue()), expressionPosition);
        }, false));
        this.tokenBuilders = builder.build();
        List<String> duplicateRegex = this.tokenBuilders.stream().collect(Collectors.groupingBy(TokenBuilder::regex)).entrySet().stream().filter(e -> ((List)e.getValue()).size() > 1).map(Map.Entry::getKey).toList();
        if (!duplicateRegex.isEmpty()) {
            throw new IllegalArgumentException("The following regex for tokenizing where used more then once: " + String.valueOf(duplicateRegex));
        }
        this.allOperatorAndFunctionNamePatterns = (ImmutableList)Stream.concat(parameters.getOperators().stream().map(Operator::getName), parameters.getFunctions().stream().map(Function::getName)).map(n -> Pattern.compile(Tokenizer.escapeForRegexAndAddCaseInsensitivity(n))).collect(ImmutableList.toImmutableList());
    }

    private static String buildEscapeBracketsRegex(BracketPair bracketPair) {
        return String.format("%s.*?%s", Tokenizer.escapeForRegexAndAddCaseInsensitivity(bracketPair.getOpen()), Tokenizer.escapeForRegexAndAddCaseInsensitivity(bracketPair.getClose()));
    }

    private static String escapeForRegexAndAddCaseInsensitivity(String in) {
        return "(?i)\\Q%s\\E(?-i)".formatted(in);
    }

    private static int countLeadingWhitespaces(String input) {
        int spaceCount = 0;
        for (char c : input.toCharArray()) {
            if (c != ' ') break;
            ++spaceCount;
        }
        return spaceCount;
    }

    public List<Token> tokenize(String input) throws ExpressionException {
        Optional<Token> currentMatch;
        ArrayList<Token> preTokens = new ArrayList<Token>();
        String current = input.trim();
        int currentPositionWithSpace = 0;
        do {
            if (!(currentMatch = this.getBestMatch(current, currentPositionWithSpace)).isPresent()) continue;
            Token token = currentMatch.get();
            preTokens.add(token);
            int matchLength = token.getExpressionPosition().getValue().length();
            currentPositionWithSpace += matchLength;
            String substringWithSpace = current.substring(matchLength);
            currentPositionWithSpace += Tokenizer.countLeadingWhitespaces(substringWithSpace);
            current = substringWithSpace.trim();
        } while (currentMatch.isPresent());
        if (!current.isEmpty()) {
            int nextPositionWithMatch = this.findNextMatchForErrorMessage(current);
            String nonMatchingString = current.substring(0, nextPositionWithMatch);
            throw new ExpressionException("No matching operator for '%s', non-functional text and value names must to be surrounded by %s".formatted(nonMatchingString, this.escapeCharacter), ExpressionPosition.of(currentPositionWithSpace, nonMatchingString));
        }
        return this.setOperatorType(preTokens);
    }

    private int findNextMatchForErrorMessage(String current) {
        int i = 0;
        while (!current.isEmpty()) {
            ++i;
            current = current.substring(1);
            try {
                Optional<Token> match = this.getBestMatch(current, i);
                if (!match.isPresent()) continue;
                return i;
            }
            catch (ExpressionException e) {
                return i;
            }
        }
        return i;
    }

    private List<Token> setOperatorType(List<Token> in) throws ExpressionException {
        ImmutableList.Builder builder = ImmutableList.builder();
        boolean lastOperatorWasUnaryLeft = false;
        for (int i = 0; i < in.size(); ++i) {
            Optional<String> followingCloseBracket;
            Token token = in.get(i);
            Optional<String> previousOpenBracket = this.getPreviousOpenBrackets(in, i);
            if (previousOpenBracket.isPresent()) {
                token = Token.addOpenBracket(token, previousOpenBracket.get());
            }
            if ((followingCloseBracket = this.getFollowingCloseBrackets(in, i)).isPresent()) {
                token = Token.addCloseBracket(token, followingCloseBracket.get());
            }
            if (token.getOperator().isPresent()) {
                Token left = i == 0 ? null : in.get(i - 1);
                Token right = i == in.size() - 1 ? null : in.get(i + 1);
                Operator.OperatorType type = this.determineAndValidateOperatorType(token, left, right, lastOperatorWasUnaryLeft);
                builder.add((Object)Token.of(token.getOperator().get(), type, token.getExpressionPosition()));
                lastOperatorWasUnaryLeft = type == Operator.OperatorType.UNARY && token.getOperator().get().getAssociativityForOperantType(Operator.OperatorType.UNARY) == Operator.Associativity.LEFT;
                continue;
            }
            builder.add((Object)token);
            lastOperatorWasUnaryLeft = false;
        }
        return builder.build();
    }

    private Optional<String> getPreviousOpenBrackets(List<Token> in, int index) {
        if (index <= 0) {
            return Optional.empty();
        }
        StringBuilder brackets = new StringBuilder();
        for (int i = index - 1; i >= 0 && in.get(i).getBrackets().isPresent() && in.get(i).isOpenBracket(); --i) {
            brackets.append(in.get(i).getBrackets().get().getOpen());
        }
        if (brackets.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(brackets.toString());
    }

    private Optional<String> getFollowingCloseBrackets(List<Token> in, int index) {
        if (index > in.size() - 2) {
            return Optional.empty();
        }
        StringBuilder brackets = new StringBuilder();
        for (int i = index + 1; i < in.size() && in.get(i).getBrackets().isPresent() && in.get(i).isCloseBracket(); ++i) {
            brackets.append(in.get(i).getBrackets().get().getClose());
        }
        if (brackets.isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(brackets.toString());
    }

    private Operator.OperatorType determineAndValidateOperatorType(@NonNull Token token, @Nullable Token left, @Nullable Token right, boolean lastOperatorWasUnaryLeft) throws ExpressionException {
        boolean rightLiteralOrBracket;
        if (token == null) {
            throw new NullPointerException("token is marked non-null but is null");
        }
        Operator operator = token.getOperator().orElseThrow();
        boolean leftLiteralOrBracket = left != null && (left.getLiteral().isPresent() || left.isCloseBracket() || left.getOperator().isPresent() && lastOperatorWasUnaryLeft);
        boolean bl = rightLiteralOrBracket = right != null && (right.getLiteral().isPresent() || right.isOpenBracket() || right.getOperator().isPresent() && right.getOperator().get().getAssociativityForOperantType(Operator.OperatorType.UNARY) == Operator.Associativity.RIGHT || right.getFunction().isPresent());
        if (leftLiteralOrBracket && rightLiteralOrBracket) {
            if (!operator.supportBinaryOperation()) {
                throw new ExpressionException("Operator %s does not support binary operations".formatted(operator.getName()), token.getExpressionPosition());
            }
            return Operator.OperatorType.BINARY;
        }
        if (!operator.supportUnaryOperation()) {
            throw new ExpressionException("Operator %s does not support unary operations".formatted(operator.getName()), token.getExpressionPosition());
        }
        Operator.Associativity operatorAssociativity = operator.getAssociativityForOperantType(Operator.OperatorType.UNARY);
        if (operatorAssociativity == Operator.Associativity.LEFT && !leftLiteralOrBracket) {
            throw new ExpressionException("Operator %s has left associativity but the left value was: %s".formatted(operator.getName(), Optional.ofNullable(left).map(Object::toString).orElse("empty")), token.getExpressionPosition());
        }
        if (operatorAssociativity == Operator.Associativity.RIGHT && !rightLiteralOrBracket) {
            throw new ExpressionException("Operator %s has right associativity but the right value was: %s".formatted(operator.getName(), Optional.ofNullable(right).map(Object::toString).orElse("empty")), token.getExpressionPosition());
        }
        return Operator.OperatorType.UNARY;
    }

    private Optional<Token> getBestMatch(String input, int position) throws ExpressionException {
        List<Token> allMatches = this.getAllMatches(input, position);
        int maxLength = allMatches.stream().map(Token::getExpressionPosition).map(ExpressionPosition::getValue).mapToInt(String::length).max().orElse(0);
        List<Token> maxLengthMatches = allMatches.stream().filter(m -> m.getExpressionPosition().getValue().length() == maxLength).toList();
        if (maxLengthMatches.isEmpty()) {
            return Optional.empty();
        }
        if (maxLengthMatches.size() > 1) {
            throw new IllegalStateException("More then one operator matched the input %s: %s".formatted(input, maxLengthMatches.stream().map(Token::toString).toList()));
        }
        return Optional.of(maxLengthMatches.getFirst());
    }

    private List<Token> getAllMatches(String input, int position) throws ExpressionException {
        ImmutableList.Builder matchBuilder = ImmutableList.builder();
        for (TokenBuilder tokenBuilder : this.tokenBuilders) {
            Optional<Token> firstMatch = this.getFirstMatch(input, tokenBuilder, position);
            firstMatch.ifPresent(arg_0 -> ((ImmutableList.Builder)matchBuilder).add(arg_0));
        }
        return matchBuilder.build();
    }

    private Optional<Token> getFirstMatch(String input, TokenBuilder tokenBuilder, int position) throws ExpressionException {
        Matcher matcher = tokenBuilder.pattern().matcher(input);
        if (matcher.find()) {
            String matchGroup = matcher.group().trim();
            return Optional.of(tokenBuilder.toToken().apply(ExpressionPosition.of(position, matchGroup)));
        }
        return Optional.empty();
    }

    public boolean expressionContainsOperatorOrFunction(String expression) {
        return this.allOperatorAndFunctionNamePatterns.stream().anyMatch(p -> p.matcher(expression).find());
    }

    private record TokenBuilder(String regex, ToToken toToken, boolean multiLine) {
        Pattern pattern() {
            if (this.multiLine) {
                return Pattern.compile("^\\s*%s\\s*".formatted(this.regex), 32);
            }
            return Pattern.compile("^\\s*%s\\s*".formatted(this.regex));
        }
    }

    private static interface ToToken {
        public Token apply(ExpressionPosition var1) throws ExpressionException;
    }
}

