package transpropify;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Map;
import java.util.Map.Entry;
import java.util.function.Consumer;
import java.util.TreeMap;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class PropertySerializerDefault implements PropertySerializer {

    private static final Logger log = LogManager.getLogger();

    private static final String SEPARATOR = "=";

    private Charset charset;

    private Consumer<KeyValue> preSerialization = null;

    private Consumer<KeyValue> posDeserialization = null;

    private boolean rejectNullProperties = true;

    public PropertySerializerDefault(Consumer<KeyValue> preSerialization, Consumer<KeyValue> posDeserialization,
            Charset charset) {
        this.preSerialization = preSerialization;
        this.posDeserialization = posDeserialization;
        this.charset = (charset != null) ? charset : StandardCharsets.UTF_8;
    }

    public PropertySerializerDefault() {
        this(null, null, null);
    }

    public PropertySerializerDefault(Consumer<KeyValue> preSerialization, Consumer<KeyValue> posDeserialization) {
        this(preSerialization, posDeserialization, null);
    }

    public PropertySerializerDefault(Charset charset) {
        this(null, null, charset);
    }

    public static String format(String key, String value) {
        return key + SEPARATOR + value;
    }

    @Override
    public void serialize(Entry<String, String> entry, OutputStream outputStream) throws IOException {
        serialize(entry, outputStream, this.charset);
    }

    @Override
    public void serialize(Entry<String, String> entry, OutputStream outputStream, Charset charset) throws IOException {
        KeyValue kv = new KeyValue(entry);
        serialize(kv, outputStream, charset);
    }

    @Override
    public void serialize(KeyValue keyValue, OutputStream outputStream, Charset charset) throws IOException {
        if (isRejectNullProperties() && StringUtils.isBlank(keyValue.getKey())) {
            log.debug(String.format("Rejeitando nome de propriedade nulo em '%s", keyValue.toString()));
            return;
        }
        
        if (preSerialization != null) {
            preSerialization.accept(keyValue);
        }

        String prop = PropertySerializerDefault.format(keyValue.getKey(), keyValue.getValue());
        outputStream.write(prop.getBytes(charset));
    }

    @Override
    public Entry<String, String> deserialize(String keyValueArg) {
        String keyValue = keyValueArg.trim();

        int sepIndex = keyValue.indexOf(SEPARATOR);
        if (isRejectNullProperties() && sepIndex <= 0) {
            log.debug(String.format("Rejeitando nome de propriedade nulo em '%s", keyValueArg));
            return null;
        }

        KeyValue kv = new KeyValue();
        kv.setKey(keyValue.substring(0, sepIndex));
        kv.setValue(keyValue.substring(sepIndex + 1));

        if (posDeserialization != null) {
            posDeserialization.accept(kv);
        }

        Pair<String, String> pair = new ImmutablePair<String, String>(kv.getKey(), kv.getValue());
        return pair;
    }

    public Map<String, String> load(File file) throws FileNotFoundException, IOException {
        try (FileReader reader = new FileReader(file)) {
            return load(reader);
        }
    }

    public Map<String, String> load(InputStream inputStream) throws IOException {
        try (InputStreamReader reader = new InputStreamReader(inputStream)) {
            return load(reader);
        }
    }

    public Map<String, String> load(InputStreamReader inputStream) throws IOException {
        try (BufferedReader stream = new BufferedReader(inputStream)) {
            Map<String, String> map = new TreeMap<>();

            String line;
            while ((line = stream.readLine()) != null) {
                deserializeToMaṕ(map, line);
            }

            return map;
        }
    }

    protected void deserializeToMaṕ(Map<String, String> map, String contextKeyValue) {
        Entry<String, String> entry = deserialize(contextKeyValue);
        if (entry == null) {
            return;
        }
        map.put(entry.getKey(), entry.getValue());
    }

    public boolean isRejectNullProperties() {
        return rejectNullProperties;
    }

    public void setRejectNullProperties(boolean nullPropertySupport) {
        this.rejectNullProperties = nullPropertySupport;
    }
}
