package transpropify.converters;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonToken;
import com.google.gson.stream.JsonWriter;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.util.Map;
import java.util.TreeMap;
import org.apache.commons.lang3.StringUtils;

public class ConverterJsonToMap {

    private Map<String, String> mapresult;
    private JsonReader jsonReader;

    private boolean ignoreEmpty = true;

    public ConverterJsonToMap() {
    }

    public Map<String, String> convert(JsonElement obj) {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        try {
            new Gson().toJson(obj, new JsonWriter(new OutputStreamWriter(outputStream, ConvertionConstants.DEFAULT_ENCODING)));
        } catch (IOException ex) {
            throw new RuntimeException("Falha inesperada na codificação do objeto json", ex);
        }

        ByteArrayInputStream inputStream = new ByteArrayInputStream(outputStream.toByteArray());

        try {
            return convert(inputStream);
        } catch (IOException ex) {
            throw new RuntimeException("Falha inesperada na codificação do objeto json", ex);
        }
    }

    public Map<String, String> convert(String json) {
        InputStream stream = new ByteArrayInputStream(json.getBytes());
        try {
            return convert(stream);
        } catch (IOException ex) {
            throw new RuntimeException("Falha inesperada na codificação do objeto json", ex);
        }
    }

    public Map<String, String> convert(InputStream stream) throws IOException {
        jsonReader = new JsonReader(new InputStreamReader(stream));
        mapresult = new TreeMap<>();

        handleJsonObject();

        return mapresult;
    }

    private void handleJsonObject() throws IOException {
        handleJsonObject(StringUtils.EMPTY, StringUtils.EMPTY);
    }

    private void handleJsonObject(String lastKey, String currentKey) throws IOException {
        handleJsonObject(lastKey, currentKey, StringUtils.EMPTY);
    }

    private void handleJsonObject(String parentKey, String nodeKey, String nodeIndex) throws IOException {
        jsonReader.beginObject();

        StringBuilder parentKeyStr = new StringBuilder(parentKey);
        if (nodeKey.length() > 0) {
            if (parentKeyStr.length() > 0) {
                parentKeyStr.append(".");
            }
            parentKeyStr.append(nodeKey);
        }

        String newNodeKey = StringUtils.EMPTY;

        StringBuilder newParentKeyStr=null;

        while (jsonReader.hasNext()) {
            JsonToken token = jsonReader.peek();

            switch (token) {
                case BEGIN_ARRAY:
                    if (StringUtils.isEmpty(newNodeKey)) {
                        throw new RuntimeException(String.format("Falha separando array json. Elemento sem nome (parentKey:'%s', nodeKey:'%s', nodeIndex:'%s')", parentKey, nodeKey, nodeIndex));
                    }
                    
                    newParentKeyStr = new StringBuilder(parentKeyStr);
                    if (StringUtils.isNotEmpty(nodeIndex)) {
                        newParentKeyStr.append("[").append(nodeIndex).append("]");
                    }
                    if (handleJsonArray(newParentKeyStr.toString(), newNodeKey) == 0) {
                        String newItemKey;
                        newItemKey = newParentKeyStr.toString();
                        newItemKey += StringUtils.isNotEmpty(newItemKey) ? "." : "";
                        newItemKey += newNodeKey + "[]";
                        mapresult.put(newItemKey, "");
                    }
                    break;
                case END_ARRAY:
                    jsonReader.endArray();
                    break;
                case BEGIN_OBJECT:
                    if (StringUtils.isEmpty(newNodeKey)) {
                        throw new RuntimeException(String.format("Falha separando json. Objeto json sem nome (parentKey:'%s', nodeKey:'%s', nodeIndex:'%s')", parentKey, nodeKey, nodeIndex));
                    }
                    
                    newParentKeyStr = new StringBuilder(parentKeyStr);
                    if (StringUtils.isNotEmpty(nodeIndex)) {
                        newParentKeyStr.append("[").append(nodeIndex).append("]");
                    }

                    handleJsonObject(newParentKeyStr.toString(), newNodeKey);
                    break;
                case END_OBJECT:
                    jsonReader.endObject();
                    break;
                case NAME:
                    newNodeKey = jsonReader.nextName();
                    if (newNodeKey.equals(ConvertionConstants.INDEX_NAME)) {
                        nodeIndex = jsonReader.nextString();
                    }
                    break;
                default:
                    if (StringUtils.isEmpty(newNodeKey)) {
                        throw new RuntimeException(String.format("Falha separando json. Valor json não suportado (parentKey:'%s', nodeKey:'%s', nodeIndex:'%s', token:'%s')", parentKey, nodeKey, nodeIndex, token.name()));
                    }

                    StringBuilder newItemKeyStr = new StringBuilder();
                    if (!newNodeKey.equals(ConvertionConstants.VALUE_NAME)) {
                        if (parentKeyStr.length() > 0) {
                            newItemKeyStr.append(parentKeyStr);
                            if (StringUtils.isNotEmpty(nodeIndex)) {
                                newItemKeyStr.append("[");
                                newItemKeyStr.append(nodeIndex);
                                newItemKeyStr.append("]");
                            }
                            newItemKeyStr.append(".");
                            newItemKeyStr.append(newNodeKey);
                        } else {
                            newItemKeyStr.append(newNodeKey);
                        }
                    } else if (StringUtils.isNotEmpty(nodeIndex)) {
                        if (newItemKeyStr.length() <= 0 && parentKeyStr.length() > 0) {
                            newItemKeyStr.append(parentKeyStr);
                        }
                        newItemKeyStr.append("[");
                        newItemKeyStr.append(nodeIndex);
                        newItemKeyStr.append("]");
                    }

                    String newItemKey = newItemKeyStr.toString();

                    switch (token) {
                        case STRING:
                        case NUMBER: {
                            String value = jsonReader.nextString();

                            if (!ignoreEmpty || StringUtils.isNotEmpty(value)) {
                                //TODO: call an external converter to suit the property consumer localization?
                                mapresult.put(newItemKey, value);
                            }
                            break;
                        }
                        case BOOLEAN: {
                            boolean value = jsonReader.nextBoolean();
                            mapresult.put(newItemKey, value ? "S" : "N");
                            break;
                        }
                        case NULL:
                            jsonReader.skipValue();
                            if (!ignoreEmpty) {
                                mapresult.put(newItemKey, StringUtils.EMPTY);
                            }
                            break;
                        default:
                            throw new RuntimeException(String.format("Falha separando json. Token não suportado (parentKey:'%s', nodeKey:'%s', nodeIndex:'%s', token:'%s')", parentKey, nodeKey, nodeIndex, token.name()));
                    }
                    break;
            }
        }
        jsonReader.endObject();
    }

    private void handleJsonString(String parentKey, String nodeKey) throws IOException {
        StringBuilder parentKeyStr = new StringBuilder(parentKey);
        if (nodeKey.length() > 0) {
            if (parentKeyStr.length() > 0) {
                parentKeyStr.append(".");
            }
            parentKeyStr.append(nodeKey);
        }

        for (int autoindex = 0; jsonReader.hasNext(); autoindex++) {
            JsonToken token = jsonReader.peek();

            StringBuilder newItemKeyStr = new StringBuilder();
            if (parentKeyStr.length() > 0) {
                newItemKeyStr.append(parentKeyStr);
            }
            newItemKeyStr.append("[");
            newItemKeyStr.append(autoindex);
            newItemKeyStr.append("]");

            String newItemKey = newItemKeyStr.toString();

            switch (token) {
                case STRING:
                case NUMBER: {
                    String value = jsonReader.nextString();
                    if (!ignoreEmpty || StringUtils.isNotEmpty(value)) {
                        //TODO: call an external converter to suit the property consumer localization?
                        mapresult.put(newItemKey, value);
                    }
                    break;
                }
                case BOOLEAN: {
                    boolean value = jsonReader.nextBoolean();
                    mapresult.put(newItemKey, value ? "S" : "N");
                    break;
                }
                case NULL: {
                    jsonReader.skipValue();
                    if (!ignoreEmpty) {
                        mapresult.put(newItemKey, StringUtils.EMPTY);
                    }
                    break;
                }
                default:
                    throw new RuntimeException(String.format("Falha separando json. Token não suportado (parentKey:'%s', nodeKey:'%s', autoindex:'%d', token:'%s')", parentKey, nodeKey, autoindex, token.name()));
            }
        }
    }

    private int handleJsonArray(String parentKey, String currentKey) throws IOException {
        jsonReader.beginArray();

        StringBuilder strkey = new StringBuilder();
        if (parentKey.length() > 0) {
            strkey.append(parentKey);
            strkey.append(".");
        }
        if (currentKey.length() > 0) {
            strkey.append(currentKey);
        }

        String newNodeKey = StringUtils.EMPTY;

        JsonToken token;
        int autoindex = 0;
        for (; ((token = jsonReader.peek()) != null) && !token.equals(JsonToken.END_ARRAY); autoindex++) {
            switch (token) {
                case BEGIN_OBJECT:
                    handleJsonObject(strkey.toString(), newNodeKey, String.valueOf(autoindex));
                    break;
                case END_OBJECT:
                    jsonReader.endObject();
                    return autoindex;
                case END_ARRAY:
                    jsonReader.endArray();
                    return autoindex;
                case BOOLEAN:
                case STRING:
                case NUMBER:
                case NULL:
                    handleJsonString(strkey.toString(), newNodeKey);
                    break;
                default:
                    throw new RuntimeException(String.format("Falha separando array json. Token inesperado (lastKey:'%s', currentKey:'%s', strkey:'%s', token:%s, autoindex:%d)", parentKey, currentKey, strkey.toString(), token.toString(), autoindex));
                //break;
            }
        }
        jsonReader.endArray();
        return autoindex;
    }

    public ConverterJsonToMap setKeepEmpties() {
        ignoreEmpty = false;
        return this;
    }
}
