package org.silbertb.proto.domainconverter.util;

import com.google.protobuf.Message;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.*;
import javax.lang.model.type.*;
import javax.lang.model.util.Types;
import javax.tools.Diagnostic;
import java.lang.annotation.Annotation;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class LangModelUtil {

    private final ProcessingEnvironment processingEnv;

    public LangModelUtil(ProcessingEnvironment processingEnv) {
        this.processingEnv = processingEnv;
    }

    public TypeMirror getClassFromAnnotation(Runnable classGetterFromAnnotation) {
        try {
            //A hack. It always throws the exception, and this is the easiest way to get the TypeMirror of the class
            classGetterFromAnnotation.run();
        }
        catch( MirroredTypeException mte ) {
            return mte.getTypeMirror();
        }
        return null;
    }

    public VariableElement getMemberField(TypeMirror typeMirror, String fieldName) {
        for(Element enclosedElement : processingEnv.getTypeUtils().asElement(typeMirror).getEnclosedElements()) {
            if(enclosedElement.getKind().equals(ElementKind.FIELD) && enclosedElement.getSimpleName().contentEquals(fieldName)) {
                return (VariableElement) enclosedElement;
            }
        }
        return null;
    }

    public boolean isList(TypeMirror typeMirror) {
        return isAssignedFrom(typeMirror, List.class);
    }

    public boolean isMap(TypeMirror typeMirror) {
        return isAssignedFrom(typeMirror, Map.class);
    }

    public boolean isMessage(TypeMirror typeMirror) {
        return isAssignedFrom(typeMirror, Message.class);
    }

    public boolean isAssignedFrom(TypeMirror typeMirror, Class<?> supposedSuperClass) {
        String superClassName = supposedSuperClass.getName();
        TypeElement superTypeElement = processingEnv.getElementUtils().getTypeElement(superClassName);
        Types typeUtil = processingEnv.getTypeUtils();

        return typeUtil.isAssignable(typeMirror, typeUtil.erasure(superTypeElement.asType()));
    }

    public boolean isConcreteType(Element element) {
        Element e = processingEnv.getTypeUtils().asElement(element.asType());
        return !isAbstractType(e) && !isInterfaceType(e);
    }

    public boolean isAbstractType(Element element) {
        return element.getModifiers().contains(Modifier.ABSTRACT);
    }

    public boolean isInterfaceType(Element element) {
        return element.getKind() == ElementKind.INTERFACE;
    }

    public boolean isSameType(TypeMirror typeMirror, Class<?> clazz) {
        TypeElement otherTypeElement = processingEnv.getElementUtils().getTypeElement(clazz.getName());
        Types typeUtil = processingEnv.getTypeUtils();
        return typeUtil.isSameType(typeMirror, otherTypeElement.asType());
    }

    public boolean isByteArray(TypeMirror typeMirror) {
        if (typeMirror.getKind() == TypeKind.ARRAY) {
            return ((ArrayType)typeMirror).getComponentType().getKind().equals(TypeKind.BYTE);
        }
        return false;
    }

    public <A extends Annotation> A getAnnotation(TypeMirror typeMirror, Class<A> annotationClass) {
        Types typeUtils = processingEnv.getTypeUtils();
        TypeElement typeElement = (TypeElement) typeUtils.asElement(typeMirror);
        return typeElement.getAnnotation(annotationClass);
    }

    public List<? extends TypeMirror> getGenericsTypes(TypeMirror typeMirror) {
        if(!typeMirror.getKind().equals(TypeKind.DECLARED)) {
            return Collections.emptyList();
        }

        DeclaredType fieldDeclaredType = (DeclaredType)typeMirror;
        return fieldDeclaredType.getTypeArguments();
    }

    public TypeMirror getInterfaceOf(TypeMirror typeMirror, String interfaceName) {
        if (processingEnv.getTypeUtils().erasure(typeMirror).toString().equals(interfaceName)) {
            return typeMirror;
        }

        for(TypeMirror superclass : processingEnv.getTypeUtils().directSupertypes(typeMirror)) {
            TypeMirror result = getInterfaceOf(superclass, interfaceName);
            if(result != null) {
                return result;
            }
        }

        return null;
    }

    public List<ExecutableElement> getAllMethods(TypeElement typeElement) {
        return typeElement.getEnclosedElements().stream()
                .filter(e -> e.getKind().equals(ElementKind.METHOD))
                .map(e -> (ExecutableElement)e)
                .collect(Collectors.toList());
    }

    private void info(String msg) {
        processingEnv.getMessager().printMessage(Diagnostic.Kind.NOTE, msg);
    }
}
