package co.legato.deeplink.networks.interceptors

import android.util.Log
import co.legato.deeplink.exts.catching
import co.legato.deeplink.exts.toJson
import co.legato.deeplink.networks.responses.BaseResponse
import com.google.gson.Gson
import okhttp3.Interceptor
import okhttp3.MediaType
import okhttp3.Response
import okhttp3.ResponseBody.Companion.toResponseBody
import java.net.HttpURLConnection.HTTP_OK
import java.net.HttpURLConnection.HTTP_UNAUTHORIZED

internal class ExceptionInterceptor(private val gson: Gson) : Interceptor {

    override fun intercept(chain: Interceptor.Chain): Response {
        val response = chain.proceed(chain.request())
        val body = response.body?.string() ?: ""
        val error = body.catching(showLog = false) {
            gson.fromJson(this, BaseResponse::class.java)
                .apply { code = response.code }
        } ?: BaseResponse<Any>(
            code = response.code,
            message = response.message
        )
        if (error.code != 200) {
            val message = ErrorLog(response.request.url.toString(), body)
            Log.d("Error", " Response: ${message.toJson()}")
        }
        // throw exception when call request failed.
        handleExceptionInternal(error)?.apply { throw this }

        val contentType: MediaType? = response.body?.contentType()
        return response.newBuilder()
            .body(body.toResponseBody(contentType))
            .build()
    }

    private fun handleExceptionInternal(error: BaseResponse<Any>): Throwable? =
        when (error.code) {
            HTTP_OK -> null
            /*HTTP_CONFLICT -> if (error.data is LinkedTreeMap<*, *>) {
                val version: String = error.data.getOrDefault("version", "").toString()
                UpgradeVersionException(message = error.message, version = version)
            } else null*/
            HTTP_UNAUTHORIZED -> UnauthorizedException(error.message ?: "")
            else -> RequestException(
                status = error.code,
                message = error.message ?: "",
                messageCode = ""
            )
        }

    internal data class ErrorLog(
        val url: String = "",
        val message: String = ""
    )
}