package co.legato.deeplink

import android.content.Context
import androidx.annotation.Keep
import co.legato.deeplink.data.AppConfiguration
import co.legato.deeplink.data.Environment
import co.legato.deeplink.data.GenerateLinkData
import co.legato.deeplink.data.GenerateLinkRequest
import co.legato.deeplink.networks.APIGenerator
import co.legato.deeplink.networks.client.CentralizeClient
import co.legato.deeplink.tasks.TaskImpl
import co.legato.deeplink.tasks.interfaces.Task
import co.legato.deeplink.tools.JsonGenerator.serializeToMapString
import com.google.gson.JsonElement
import com.google.gson.JsonNull
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class CentralizeDeepLink internal constructor(
    context: Context,
    private val config: AppConfiguration,
    environment: Environment
) {
    private var client: CentralizeClient

    init {
        val api = APIGenerator(context, environment.apiUrl)
        client = api.createService(CentralizeClient::class.java)
    }

    @Keep
    companion object {
        @Volatile
        private var sInstance: CentralizeDeepLink? = null

        @JvmStatic
        fun initialize(
            config: AppConfiguration,
            env: Environment,
            context: Context
        ): CentralizeDeepLink {
            return sInstance ?: synchronized(CentralizeDeepLink::class) {
                sInstance ?: CentralizeDeepLink(context, config, env).also { sInstance = it }
            }
        }

        fun getInstance(): CentralizeDeepLink {
            return sInstance
                ?: throw RuntimeException("CentralizeDeepLink.initialize haven't called yet")
        }
    }

    fun generateDeepLink(request: GenerateLinkData): Task<String> {
        val taskImpl = TaskImpl<String>()
        CoroutineScope(Dispatchers.IO).launch {
            try {

                val response =
                    client.generateDeepLink(
                        config.serializeToMapString(),
                        request = GenerateLinkRequest.from(request)
                    )
                withContext(Dispatchers.Main) {
                    taskImpl.result = response.data?.link
                }
            } catch (ex: Exception) {
                ex.printStackTrace()
                withContext(Dispatchers.Main) {
                    taskImpl.exception = ex
                }
            }
        }
        return taskImpl
    }

    fun extraShortLink(shortToken: String, metaData: Map<String, Boolean>? = null): Task<String> {
        val taskImpl = TaskImpl<String>()
        CoroutineScope(Dispatchers.IO).launch {
            try {
                val appConfig = config.serializeToMapString()
                val response = client.extraParams(appConfig, metaData, shortToken);
                withContext(Dispatchers.Main) {
                    val data: JsonElement? = response.data?.metaData
                    taskImpl.result =
                        if (response.data == null || data is JsonNull) "" else data?.toString()
                            ?: ""
                }
            } catch (ex: Exception) {
                ex.printStackTrace()
                withContext(Dispatchers.Main) {
                    taskImpl.exception = ex
                }
            }
        }
        return taskImpl
    }
}