/*
 * Decompiled with CFR 0.152.
 */
package io.gravitee.reporter.file.vertx;

import io.gravitee.reporter.api.Reportable;
import io.gravitee.reporter.file.MetricsType;
import io.gravitee.reporter.file.config.FileReporterConfiguration;
import io.gravitee.reporter.file.formatter.Formatter;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.buffer.Buffer;
import io.vertx.core.file.AsyncFile;
import io.vertx.core.file.OpenOptions;
import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;
import java.util.Timer;
import java.util.TimerTask;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class VertxFileWriter<T extends Reportable> {
    private static final Logger LOGGER = LoggerFactory.getLogger(VertxFileWriter.class);
    private static final char LF = '\n';
    private static final char CR = '\r';
    private static final byte[] END_OF_LINE = new byte[]{13, 10};
    private final Vertx vertx;
    private String filename;
    private final MetricsType type;
    private final Formatter<T> formatter;
    private AsyncFile asyncFile;
    private static final String ROLLOVER_FILE_DATE_FORMAT = "yyyy_MM_dd";
    private static final String YYYY_MM_DD = "yyyy_mm_dd";
    private static Timer __rollover;
    private RollTask _rollTask;
    private final SimpleDateFormat fileDateFormat = new SimpleDateFormat("yyyy_MM_dd");
    private FileReporterConfiguration configuration;
    private final long flushId;
    private final Pattern rolloverFiles;

    public VertxFileWriter(Vertx vertx, MetricsType type, Formatter<T> formatter, String filename, FileReporterConfiguration configuration) {
        this.vertx = vertx;
        this.type = type;
        this.formatter = formatter;
        this.configuration = configuration;
        if (filename != null && (filename = filename.trim()).length() == 0) {
            filename = null;
        }
        if (filename == null) {
            throw new IllegalArgumentException("Invalid filename");
        }
        this.filename = filename;
        File file = new File(this.filename);
        int datePattern = configuration.getFilename().toLowerCase(Locale.ENGLISH).indexOf(YYYY_MM_DD);
        this.rolloverFiles = datePattern >= 0 ? Pattern.compile(String.format(file.getName(), this.type.getType()).replaceFirst(YYYY_MM_DD, "([0-9]{4}_[0-9]{2}_[0-9]{2})")) : null;
        __rollover = new Timer(VertxFileWriter.class.getName(), true);
        this.flushId = vertx.setPeriodic(configuration.getFlushInterval(), event -> {
            LOGGER.debug("Flush the content to file");
            if (this.asyncFile != null) {
                this.asyncFile.flush(event1 -> {
                    if (event1.failed()) {
                        LOGGER.error("An error occurs while flushing the content of the file", event1.cause());
                    }
                });
            }
        });
    }

    public Future<Void> initialize() {
        ZonedDateTime now = ZonedDateTime.now(TimeZone.getDefault().toZoneId());
        this.scheduleNextRollover(now);
        return this.setFile(now);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Future<Void> setFile(ZonedDateTime now) {
        Promise promise = Promise.promise();
        VertxFileWriter vertxFileWriter = this;
        synchronized (vertxFileWriter) {
            File file = new File(this.filename);
            try {
                this.filename = file.getCanonicalPath();
                file = new File(this.filename);
                File dir = new File(file.getParent());
                if (!dir.isDirectory() || !dir.canWrite()) {
                    LOGGER.error("Cannot write reporter data to directory " + dir);
                    promise.fail((Throwable)new IOException("Cannot write reporter data to directory " + dir));
                    return promise.future();
                }
                Object filename = String.format(file.getName(), this.type.getType());
                int datePattern = ((String)filename).toLowerCase(Locale.ENGLISH).indexOf(YYYY_MM_DD);
                if (datePattern >= 0) {
                    filename = dir.getAbsolutePath() + File.separatorChar + ((String)filename).substring(0, datePattern) + this.fileDateFormat.format(new Date(now.toInstant().toEpochMilli())) + ((String)filename).substring(datePattern + YYYY_MM_DD.length());
                }
                LOGGER.info("Initializing file reporter to write into file: {}", filename);
                AsyncFile oldAsyncFile = this.asyncFile;
                OpenOptions options = new OpenOptions().setAppend(true).setCreate(true);
                if (this.configuration.getFlushInterval() <= 0L) {
                    options.setDsync(true);
                }
                this.vertx.fileSystem().open((String)filename, options, event -> {
                    if (event.succeeded()) {
                        this.asyncFile = (AsyncFile)event.result();
                        if (oldAsyncFile != null) {
                            this.stop(oldAsyncFile).onComplete(closeEvent -> {
                                if (!closeEvent.succeeded()) {
                                    LOGGER.error("An error occurs while closing file writer for type[{}]", (Object)this.type, (Object)closeEvent.cause());
                                }
                            });
                        }
                        promise.complete();
                    } else {
                        LOGGER.error("An error occurs while starting file writer for type[{}]", (Object)this.type, (Object)event.cause());
                        promise.fail(event.cause());
                    }
                });
            }
            catch (IOException ioe) {
                promise.fail((Throwable)ioe);
            }
        }
        return promise.future();
    }

    public void write(T data) {
        if (this.asyncFile != null && !this.asyncFile.writeQueueFull()) {
            this.vertx.executeBlocking(event -> {
                Buffer buffer = this.formatter.format((Reportable)data);
                if (buffer != null) {
                    event.complete((Object)buffer);
                } else {
                    event.fail("Invalid data");
                }
            }, event -> {
                if (event.succeeded() && !this.asyncFile.writeQueueFull()) {
                    this.asyncFile.write((Object)((Buffer)event.result()).appendBytes(END_OF_LINE));
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Future<Void> stop() {
        Promise promise = Promise.promise();
        Class<VertxFileWriter> clazz = VertxFileWriter.class;
        synchronized (VertxFileWriter.class) {
            if (this._rollTask != null) {
                this._rollTask.cancel();
            }
            // ** MonitorExit[var2_2] (shouldn't be in output)
            this.stop(this.asyncFile).onComplete(event -> {
                this.vertx.cancelTimer(this.flushId);
                if (event.succeeded()) {
                    this.asyncFile = null;
                    promise.complete();
                } else {
                    promise.fail(event.cause());
                }
            });
            return promise.future();
        }
    }

    private Future<Void> stop(AsyncFile asyncFile) {
        Promise promise = Promise.promise();
        if (asyncFile != null) {
            asyncFile.flush(flushEvent -> asyncFile.close(event -> {
                if (event.succeeded()) {
                    LOGGER.info("File writer is now closed for type [{}]", (Object)this.type);
                    promise.complete();
                } else {
                    LOGGER.error("An error occurs while closing file writer for type[{}]", (Object)this.type, (Object)event.cause());
                    promise.fail(event.cause());
                }
            }));
        } else {
            promise.complete();
        }
        return promise.future();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void scheduleNextRollover(ZonedDateTime now) {
        this._rollTask = new RollTask();
        ZonedDateTime midnight = VertxFileWriter.toMidnight(now);
        long delay = midnight.toInstant().toEpochMilli() - now.toInstant().toEpochMilli();
        Class<VertxFileWriter> clazz = VertxFileWriter.class;
        synchronized (VertxFileWriter.class) {
            __rollover.schedule((TimerTask)this._rollTask, delay);
            // ** MonitorExit[var5_4] (shouldn't be in output)
            return;
        }
    }

    private static ZonedDateTime toMidnight(ZonedDateTime now) {
        return now.toLocalDate().atStartOfDay(now.getZone()).plus(1L, ChronoUnit.DAYS);
    }

    private void removeOldFiles() {
        if (this.configuration.getRetainDays() > 0L) {
            long now = System.currentTimeMillis();
            File file = new File(this.filename);
            if (this.rolloverFiles != null) {
                File dir = new File(file.getParent());
                String[] logList = dir.list();
                for (int i = 0; i < logList.length; ++i) {
                    File f;
                    String fn = logList[i];
                    if (!this.rolloverFiles.matcher(fn).matches() || !this.shouldDeleteFile(f = new File(dir, fn), now)) continue;
                    f.delete();
                }
            }
        }
    }

    protected boolean shouldDeleteFile(File file, long currentTimeInMs) {
        long retainDays = this.configuration.getRetainDays();
        return retainDays > 0L && currentTimeInMs - file.lastModified() > retainDays * 1000L * 60L * 60L * 24L;
    }

    private class RollTask
    extends TimerTask {
        private RollTask() {
        }

        @Override
        public void run() {
            try {
                ZonedDateTime now = ZonedDateTime.now(VertxFileWriter.this.fileDateFormat.getTimeZone().toZoneId());
                VertxFileWriter.this.setFile(now);
                VertxFileWriter.this.scheduleNextRollover(now);
                VertxFileWriter.this.removeOldFiles();
            }
            catch (Throwable t) {
                LOGGER.error("Unexpected error while moving to a new reporter file", t);
            }
        }
    }
}

