/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.transport.netty;

import com.google.common.base.Preconditions;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.transport.HttpUtil;
import io.grpc.transport.netty.BufferingHttp2ConnectionEncoder;
import io.grpc.transport.netty.CancelStreamCommand;
import io.grpc.transport.netty.CreateStreamCommand;
import io.grpc.transport.netty.GoAwayClosedStreamException;
import io.grpc.transport.netty.NettyClientStream;
import io.grpc.transport.netty.RequestMessagesCommand;
import io.grpc.transport.netty.SendGrpcFrameCommand;
import io.grpc.transport.netty.WriteQueue;
import io.netty.buffer.ByteBuf;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.http2.DefaultHttp2ConnectionDecoder;
import io.netty.handler.codec.http2.Http2Connection;
import io.netty.handler.codec.http2.Http2ConnectionAdapter;
import io.netty.handler.codec.http2.Http2ConnectionDecoder;
import io.netty.handler.codec.http2.Http2ConnectionEncoder;
import io.netty.handler.codec.http2.Http2ConnectionHandler;
import io.netty.handler.codec.http2.Http2Error;
import io.netty.handler.codec.http2.Http2Exception;
import io.netty.handler.codec.http2.Http2FrameAdapter;
import io.netty.handler.codec.http2.Http2FrameListener;
import io.netty.handler.codec.http2.Http2FrameReader;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2LocalFlowController;
import io.netty.handler.codec.http2.Http2Stream;
import io.netty.handler.codec.http2.Http2StreamVisitor;
import io.netty.util.CharsetUtil;
import io.netty.util.concurrent.GenericFutureListener;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;

class NettyClientHandler
extends Http2ConnectionHandler {
    private static final Logger logger = Logger.getLogger(NettyClientHandler.class.getName());
    private final Http2Connection.PropertyKey streamKey;
    private WriteQueue clientWriteQueue;
    private int connectionWindowSize;
    private Throwable connectionError;
    private Status goAwayStatus;
    private ChannelHandlerContext ctx;
    private int nextStreamId;

    public NettyClientHandler(BufferingHttp2ConnectionEncoder encoder, Http2Connection connection, Http2FrameReader frameReader, int connectionWindowSize, int streamWindowSize) {
        super((Http2ConnectionDecoder)new DefaultHttp2ConnectionDecoder(connection, (Http2ConnectionEncoder)encoder, frameReader, (Http2FrameListener)new LazyFrameListener()), (Http2ConnectionEncoder)encoder);
        Preconditions.checkArgument((connectionWindowSize > 0 ? 1 : 0) != 0, (Object)"connectionWindowSize must be positive");
        this.connectionWindowSize = connectionWindowSize;
        try {
            this.decoder().flowController().initialWindowSize(streamWindowSize);
        }
        catch (Http2Exception e) {
            throw new RuntimeException(e);
        }
        this.initListener();
        this.streamKey = connection.newKey();
        connection.remote().maxActiveStreams(0);
        connection.local().allowPushTo(false);
        this.nextStreamId = connection.local().nextStreamId();
        connection.addListener((Http2Connection.Listener)new Http2ConnectionAdapter(){

            public void onGoAwayReceived(int lastStreamId, long errorCode, ByteBuf debugData) {
                NettyClientHandler.this.goAwayStatus(NettyClientHandler.this.statusFromGoAway(errorCode, debugData));
                NettyClientHandler.this.goingAway();
            }
        });
    }

    @Nullable
    public Throwable connectionError() {
        return this.connectionError;
    }

    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
        this.ctx = ctx;
        super.handlerAdded(ctx);
        this.initConnectionWindow();
    }

    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        super.channelActive(ctx);
        this.initConnectionWindow();
        ctx.flush();
    }

    public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) {
        if (msg instanceof CreateStreamCommand) {
            this.createStream((CreateStreamCommand)msg, promise);
        } else if (msg instanceof SendGrpcFrameCommand) {
            this.sendGrpcFrame(ctx, (SendGrpcFrameCommand)((Object)msg), promise);
        } else if (msg instanceof CancelStreamCommand) {
            this.cancelStream(ctx, (CancelStreamCommand)msg, promise);
        } else if (msg instanceof RequestMessagesCommand) {
            ((RequestMessagesCommand)msg).requestMessages();
        } else {
            throw new AssertionError((Object)("Write called for unexpected type: " + msg.getClass().getName()));
        }
    }

    void startWriteQueue(Channel channel) {
        this.clientWriteQueue = new WriteQueue(channel);
    }

    WriteQueue getWriteQueue() {
        return this.clientWriteQueue;
    }

    void returnProcessedBytes(Http2Stream stream, int bytes) {
        try {
            this.decoder().flowController().consumeBytes(this.ctx, stream, bytes);
        }
        catch (Http2Exception e) {
            throw new RuntimeException(e);
        }
    }

    private void initListener() {
        ((LazyFrameListener)this.decoder().listener()).setHandler(this);
    }

    private void onHeadersRead(int streamId, Http2Headers headers, boolean endStream) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        stream.transportHeadersReceived(headers, endStream);
    }

    private void onDataRead(int streamId, ByteBuf data, boolean endOfStream) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        stream.transportDataReceived(data, endOfStream);
    }

    private void onRstStreamRead(int streamId) throws Http2Exception {
        NettyClientStream stream = this.clientStream(this.requireHttp2Stream(streamId));
        stream.transportReportStatus(Status.UNKNOWN, false, new Metadata.Trailers());
    }

    public void close(ChannelHandlerContext ctx, ChannelPromise promise) throws Exception {
        logger.fine("Network channel being closed by the application.");
        super.close(ctx, promise);
    }

    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        try {
            logger.fine("Network channel is closed");
            this.goAwayStatus(this.goAwayStatus().augmentDescription("Network channel closed"));
            this.connection().forEachActiveStream(new Http2StreamVisitor(){

                public boolean visit(Http2Stream stream) throws Http2Exception {
                    NettyClientHandler.this.clientStream(stream).transportReportStatus(NettyClientHandler.this.goAwayStatus, false, new Metadata.Trailers());
                    return true;
                }
            });
        }
        finally {
            super.channelInactive(ctx);
        }
    }

    protected void onConnectionError(ChannelHandlerContext ctx, Throwable cause, Http2Exception http2Ex) {
        logger.log(Level.FINE, "Caught a connection error", cause);
        this.connectionError = cause;
        this.goAwayStatus(Status.fromThrowable(this.connectionError));
        super.onConnectionError(ctx, cause, http2Ex);
    }

    protected void onStreamError(ChannelHandlerContext ctx, Throwable cause, Http2Exception.StreamException http2Ex) {
        Http2Stream stream = this.connection().stream(http2Ex.streamId());
        if (stream != null) {
            this.clientStream(stream).transportReportStatus(Status.fromThrowable(cause), false, new Metadata.Trailers());
        }
        super.onStreamError(ctx, cause, http2Ex);
    }

    protected boolean isGracefulShutdownComplete() {
        return super.isGracefulShutdownComplete() && ((BufferingHttp2ConnectionEncoder)this.encoder()).numBufferedStreams() == 0;
    }

    private void createStream(CreateStreamCommand command, ChannelPromise promise) {
        final int streamId = this.getAndIncrementNextStreamId();
        final NettyClientStream stream = command.stream();
        Http2Headers headers = command.headers();
        stream.id(streamId);
        this.encoder().writeHeaders(this.ctx, streamId, headers, 0, false, promise).addListener((GenericFutureListener)new ChannelFutureListener(){

            public void operationComplete(ChannelFuture future) throws Exception {
                if (future.isSuccess()) {
                    Http2Stream http2Stream = NettyClientHandler.this.connection().stream(streamId);
                    if (http2Stream != null) {
                        http2Stream.setProperty(NettyClientHandler.this.streamKey, (Object)stream);
                    }
                    stream.setHttp2Stream(http2Stream);
                } else if (future.cause() instanceof GoAwayClosedStreamException) {
                    GoAwayClosedStreamException e = (GoAwayClosedStreamException)future.cause();
                    NettyClientHandler.this.goAwayStatus(NettyClientHandler.this.statusFromGoAway(e.errorCode(), e.debugData()));
                    stream.transportReportStatus(NettyClientHandler.this.goAwayStatus, false, new Metadata.Trailers());
                } else {
                    stream.transportReportStatus(Status.fromThrowable(future.cause()), true, new Metadata.Trailers());
                }
            }
        });
    }

    private void cancelStream(ChannelHandlerContext ctx, CancelStreamCommand cmd, ChannelPromise promise) {
        NettyClientStream stream = cmd.stream();
        stream.transportReportStatus(Status.CANCELLED, true, new Metadata.Trailers());
        this.encoder().writeRstStream(ctx, stream.id().intValue(), Http2Error.CANCEL.code(), promise);
    }

    private void sendGrpcFrame(ChannelHandlerContext ctx, SendGrpcFrameCommand cmd, ChannelPromise promise) {
        this.encoder().writeData(ctx, cmd.streamId(), cmd.content(), 0, cmd.endStream(), promise);
    }

    private void goingAway() {
        final Status goAwayStatus = this.goAwayStatus();
        final int lastKnownStream = this.connection().local().lastStreamKnownByPeer();
        try {
            this.connection().forEachActiveStream(new Http2StreamVisitor(){

                public boolean visit(Http2Stream stream) throws Http2Exception {
                    if (stream.id() > lastKnownStream) {
                        NettyClientHandler.this.clientStream(stream).transportReportStatus(goAwayStatus, false, new Metadata.Trailers());
                        stream.close();
                    }
                    return true;
                }
            });
        }
        catch (Http2Exception e) {
            throw new RuntimeException(e);
        }
    }

    private Status goAwayStatus() {
        if (this.goAwayStatus != null) {
            return this.goAwayStatus;
        }
        return Status.UNAVAILABLE.withDescription("Connection going away, but for unknown reason");
    }

    private void goAwayStatus(Status status) {
        this.goAwayStatus = this.goAwayStatus == null ? status : this.goAwayStatus;
    }

    private Status statusFromGoAway(long errorCode, ByteBuf debugData) {
        Status status = HttpUtil.Http2Error.statusForCode((int)errorCode);
        if (debugData.isReadable()) {
            String msg = debugData.toString(CharsetUtil.UTF_8);
            status = status.augmentDescription(msg);
        }
        return status;
    }

    private NettyClientStream clientStream(Http2Stream stream) {
        return (NettyClientStream)stream.getProperty(this.streamKey);
    }

    private int getAndIncrementNextStreamId() {
        int id = this.nextStreamId;
        this.nextStreamId += 2;
        return id;
    }

    private Http2Stream requireHttp2Stream(int streamId) {
        Http2Stream stream = this.connection().stream(streamId);
        if (stream == null) {
            throw new AssertionError((Object)("Stream does not exist: " + streamId));
        }
        return stream;
    }

    private void initConnectionWindow() throws Http2Exception {
        if (this.connectionWindowSize > 0 && this.ctx.channel().isActive()) {
            Http2Stream connectionStream = this.connection().connectionStream();
            int currentSize = ((Http2LocalFlowController)this.connection().local().flowController()).windowSize(connectionStream);
            int delta = this.connectionWindowSize - currentSize;
            this.decoder().flowController().incrementWindowSize(this.ctx, connectionStream, delta);
            this.connectionWindowSize = -1;
        }
    }

    private static class LazyFrameListener
    extends Http2FrameAdapter {
        private NettyClientHandler handler;

        private LazyFrameListener() {
        }

        void setHandler(NettyClientHandler handler) {
            this.handler = handler;
        }

        public int onDataRead(ChannelHandlerContext ctx, int streamId, ByteBuf data, int padding, boolean endOfStream) throws Http2Exception {
            this.handler.onDataRead(streamId, data, endOfStream);
            return padding;
        }

        public void onHeadersRead(ChannelHandlerContext ctx, int streamId, Http2Headers headers, int streamDependency, short weight, boolean exclusive, int padding, boolean endStream) throws Http2Exception {
            this.handler.onHeadersRead(streamId, headers, endStream);
        }

        public void onRstStreamRead(ChannelHandlerContext ctx, int streamId, long errorCode) throws Http2Exception {
            this.handler.onRstStreamRead(streamId);
        }
    }
}

