/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.benchmarks.qps;

import com.google.common.base.Preconditions;
import com.google.protobuf.ByteString;
import io.grpc.Channel;
import io.grpc.ChannelImpl;
import io.grpc.Status;
import io.grpc.benchmarks.qps.ClientConfiguration;
import io.grpc.benchmarks.qps.Utils;
import io.grpc.stub.StreamObserver;
import io.grpc.testing.Payload;
import io.grpc.testing.SimpleRequest;
import io.grpc.testing.SimpleResponse;
import io.grpc.testing.TestServiceGrpc;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.HdrHistogram.Histogram;
import org.HdrHistogram.HistogramIterationValue;

public class AsyncClient {
    private final ClientConfiguration config;

    public AsyncClient(ClientConfiguration config) {
        this.config = config;
    }

    public void run() throws Exception {
        if (this.config == null) {
            return;
        }
        SimpleRequest req = this.newRequest();
        ArrayList<Channel> channels = new ArrayList<Channel>(this.config.channels);
        for (int i = 0; i < this.config.channels; ++i) {
            channels.add(Utils.newClientChannel(this.config));
        }
        this.warmup(req, channels);
        long startTime = System.nanoTime();
        long endTime = startTime + TimeUnit.SECONDS.toNanos(this.config.duration);
        List<Histogram> histograms = this.doBenchmark(req, channels, endTime);
        long elapsedTime = System.nanoTime() - startTime;
        Histogram merged = AsyncClient.merge(histograms);
        this.printStats(merged, elapsedTime);
        if (this.config.histogramFile != null) {
            Utils.saveHistogram(merged, this.config.histogramFile);
        }
        AsyncClient.shutdown(channels);
    }

    private SimpleRequest newRequest() {
        ByteString body = ByteString.copyFrom((byte[])new byte[this.config.clientPayload]);
        Payload payload = Payload.newBuilder().setType(this.config.payloadType).setBody(body).build();
        return SimpleRequest.newBuilder().setResponseType(this.config.payloadType).setResponseSize(this.config.serverPayload).setPayload(payload).build();
    }

    private void warmup(SimpleRequest req, List<Channel> channels) throws Exception {
        long endTime = System.nanoTime() + TimeUnit.SECONDS.toNanos(this.config.warmupDuration);
        this.doBenchmark(req, channels, endTime);
        System.gc();
    }

    private List<Histogram> doBenchmark(SimpleRequest req, List<Channel> channels, long endTime) throws Exception {
        ArrayList<Future<Histogram>> futures = new ArrayList<Future<Histogram>>(this.config.outstandingRpcsPerChannel);
        for (int i = 0; i < this.config.channels; ++i) {
            for (int j = 0; j < this.config.outstandingRpcsPerChannel; ++j) {
                Channel channel = channels.get(i);
                futures.add(this.doRpcs(channel, req, endTime));
            }
        }
        ArrayList<Histogram> histograms = new ArrayList<Histogram>(futures.size());
        for (Future future : futures) {
            histograms.add((Histogram)future.get());
        }
        return histograms;
    }

    private Future<Histogram> doRpcs(Channel channel, SimpleRequest request, long endTime) {
        switch (this.config.rpcType) {
            case UNARY: {
                return this.doUnaryCalls(channel, request, endTime);
            }
            case STREAMING: {
                return AsyncClient.doStreamingCalls(channel, request, endTime);
            }
        }
        throw new IllegalStateException("unsupported rpc type");
    }

    private Future<Histogram> doUnaryCalls(Channel channel, final SimpleRequest request, final long endTime) {
        final TestServiceGrpc.TestServiceStub stub = TestServiceGrpc.newStub(channel);
        final Histogram histogram = new Histogram(60000000L, 3);
        final HistogramFuture future = new HistogramFuture(histogram);
        stub.unaryCall(request, new StreamObserver<SimpleResponse>(){
            long lastCall = System.nanoTime();

            public void onValue(SimpleResponse value) {
            }

            public void onError(Throwable t) {
                Status status = Status.fromThrowable((Throwable)t);
                System.err.println("Encountered an error in unaryCall. Status is " + status);
                t.printStackTrace();
                future.cancel(true);
            }

            public void onCompleted() {
                long now = System.nanoTime();
                histogram.recordValue((now - this.lastCall) / 1000L);
                this.lastCall = now;
                if (endTime > now) {
                    stub.unaryCall(request, this);
                } else {
                    future.done();
                }
            }
        });
        return future;
    }

    private static Future<Histogram> doStreamingCalls(Channel channel, SimpleRequest request, long endTime) {
        TestServiceGrpc.TestServiceStub stub = TestServiceGrpc.newStub(channel);
        Histogram histogram = new Histogram(60000000L, 3);
        HistogramFuture future = new HistogramFuture(histogram);
        ThisIsAHackStreamObserver responseObserver = new ThisIsAHackStreamObserver(request, histogram, future, endTime);
        StreamObserver<SimpleRequest> requestObserver = stub.streamingCall(responseObserver);
        responseObserver.requestObserver = requestObserver;
        requestObserver.onValue((Object)request);
        return future;
    }

    private static Histogram merge(List<Histogram> histograms) {
        Histogram merged = new Histogram(60000000L, 3);
        for (Histogram histogram : histograms) {
            for (HistogramIterationValue value : histogram.allValues()) {
                long latency = value.getValueIteratedTo();
                long count = value.getCountAtValueIteratedTo();
                merged.recordValueWithCount(latency, count);
            }
        }
        return merged;
    }

    private void printStats(Histogram histogram, long elapsedTime) {
        long latency50 = histogram.getValueAtPercentile(50.0);
        long latency90 = histogram.getValueAtPercentile(90.0);
        long latency95 = histogram.getValueAtPercentile(95.0);
        long latency99 = histogram.getValueAtPercentile(99.0);
        long latency999 = histogram.getValueAtPercentile(99.9);
        long latencyMax = histogram.getValueAtPercentile(100.0);
        long queriesPerSecond = histogram.getTotalCount() * 1000000000L / elapsedTime;
        StringBuilder values = new StringBuilder();
        values.append("Channels:                       ").append(this.config.channels).append('\n').append("Outstanding RPCs per Channel:   ").append(this.config.outstandingRpcsPerChannel).append('\n').append("Server Payload Size:            ").append(this.config.serverPayload).append('\n').append("Client Payload Size:            ").append(this.config.clientPayload).append('\n').append("50%ile Latency (in micros):     ").append(latency50).append('\n').append("90%ile Latency (in micros):     ").append(latency90).append('\n').append("95%ile Latency (in micros):     ").append(latency95).append('\n').append("99%ile Latency (in micros):     ").append(latency99).append('\n').append("99.9%ile Latency (in micros):   ").append(latency999).append('\n').append("Maximum Latency (in micros):    ").append(latencyMax).append('\n').append("QPS:                            ").append(queriesPerSecond).append('\n');
        System.out.println(values);
    }

    private static void shutdown(List<Channel> channels) {
        for (Channel channel : channels) {
            ((ChannelImpl)channel).shutdown();
        }
    }

    public static void main(String ... args) throws Exception {
        ClientConfiguration config;
        ClientConfiguration.Builder configBuilder = ClientConfiguration.newBuilder(ClientConfiguration.ClientParam.ADDRESS, ClientConfiguration.ClientParam.CHANNELS, ClientConfiguration.ClientParam.OUTSTANDING_RPCS, ClientConfiguration.ClientParam.CLIENT_PAYLOAD, ClientConfiguration.ClientParam.SERVER_PAYLOAD, ClientConfiguration.ClientParam.TLS, ClientConfiguration.ClientParam.TESTCA, ClientConfiguration.ClientParam.TRANSPORT, ClientConfiguration.ClientParam.DURATION, ClientConfiguration.ClientParam.WARMUP_DURATION, ClientConfiguration.ClientParam.DIRECTEXECUTOR, ClientConfiguration.ClientParam.SAVE_HISTOGRAM, ClientConfiguration.ClientParam.STREAMING_RPCS, ClientConfiguration.ClientParam.FLOW_CONTROL_WINDOW);
        try {
            config = (ClientConfiguration)configBuilder.build(args);
        }
        catch (Exception e) {
            System.out.println(e.getMessage());
            configBuilder.printUsage();
            return;
        }
        AsyncClient client = new AsyncClient(config);
        client.run();
    }

    private static class HistogramFuture
    implements Future<Histogram> {
        private final Histogram histogram;
        private boolean canceled;
        private boolean done;

        HistogramFuture(Histogram histogram) {
            Preconditions.checkNotNull((Object)histogram, (Object)"histogram");
            this.histogram = histogram;
        }

        @Override
        public synchronized boolean cancel(boolean mayInterruptIfRunning) {
            if (!this.done && !this.canceled) {
                this.canceled = true;
                this.notifyAll();
                return true;
            }
            return false;
        }

        @Override
        public synchronized boolean isCancelled() {
            return this.canceled;
        }

        @Override
        public synchronized boolean isDone() {
            return this.done || this.canceled;
        }

        @Override
        public synchronized Histogram get() throws InterruptedException, ExecutionException {
            while (!this.isDone() && !this.isCancelled()) {
                this.wait();
            }
            if (this.isCancelled()) {
                throw new CancellationException();
            }
            this.done = true;
            return this.histogram;
        }

        @Override
        public Histogram get(long timeout, TimeUnit unit) throws InterruptedException, ExecutionException, TimeoutException {
            throw new UnsupportedOperationException();
        }

        private synchronized void done() {
            this.done = true;
            this.notifyAll();
        }
    }

    private static class ThisIsAHackStreamObserver
    implements StreamObserver<SimpleResponse> {
        final SimpleRequest request;
        final Histogram histogram;
        final HistogramFuture future;
        final long endTime;
        long lastCall = System.nanoTime();
        StreamObserver<SimpleRequest> requestObserver;

        ThisIsAHackStreamObserver(SimpleRequest request, Histogram histogram, HistogramFuture future, long endTime) {
            this.request = request;
            this.histogram = histogram;
            this.future = future;
            this.endTime = endTime;
        }

        public void onValue(SimpleResponse value) {
            long now = System.nanoTime();
            this.histogram.recordValue((now - this.lastCall) / 1000L);
            this.lastCall = now;
            if (this.endTime > now) {
                this.requestObserver.onValue((Object)this.request);
            } else {
                this.requestObserver.onCompleted();
            }
        }

        public void onError(Throwable t) {
            Status status = Status.fromThrowable((Throwable)t);
            System.err.println("Encountered an error in streamingCall. Status is " + status);
            t.printStackTrace();
            this.future.cancel(true);
        }

        public void onCompleted() {
            this.future.done();
        }
    }
}

