/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.binder;

import android.annotation.SuppressLint;
import android.app.admin.DevicePolicyManager;
import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.Signature;
import android.os.Build;
import android.os.Process;
import androidx.annotation.RequiresApi;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.hash.Hashing;
import com.google.errorprone.annotations.CheckReturnValue;
import io.grpc.ExperimentalApi;
import io.grpc.Status;
import io.grpc.binder.SecurityPolicy;
import io.grpc.binder.ServerSecurityPolicy;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

@CheckReturnValue
public final class SecurityPolicies {
    private static final int MY_UID = Process.myUid();
    private static final int SHA_256_BYTES_LENGTH = 32;

    private SecurityPolicies() {
    }

    @ExperimentalApi(value="https://github.com/grpc/grpc-java/issues/8022")
    public static ServerSecurityPolicy serverInternalOnly() {
        return new ServerSecurityPolicy();
    }

    public static SecurityPolicy internalOnly() {
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                return uid == MY_UID ? Status.OK : Status.PERMISSION_DENIED.withDescription("Rejected by (internal-only) security policy");
            }
        };
    }

    @ExperimentalApi(value="https://github.com/grpc/grpc-java/issues/8022")
    public static SecurityPolicy permissionDenied(String description) {
        final Status denied = Status.PERMISSION_DENIED.withDescription(description);
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                return denied;
            }
        };
    }

    @ExperimentalApi(value="https://github.com/grpc/grpc-java/issues/8022")
    public static SecurityPolicy hasSignature(PackageManager packageManager, String packageName, Signature requiredSignature) {
        return SecurityPolicies.oneOfSignatures(packageManager, packageName, (Collection<Signature>)ImmutableList.of((Object)requiredSignature));
    }

    @ExperimentalApi(value="https://github.com/grpc/grpc-java/issues/8022")
    public static SecurityPolicy hasSignatureSha256Hash(PackageManager packageManager, String packageName, byte[] requiredSignatureSha256Hash) {
        return SecurityPolicies.oneOfSignatureSha256Hash(packageManager, packageName, (List<byte[]>)ImmutableList.of((Object)requiredSignatureSha256Hash));
    }

    @ExperimentalApi(value="https://github.com/grpc/grpc-java/issues/8022")
    public static SecurityPolicy oneOfSignatures(final PackageManager packageManager, final String packageName, Collection<Signature> requiredSignatures) {
        Preconditions.checkNotNull((Object)packageManager, (Object)"packageManager");
        Preconditions.checkNotNull((Object)packageName, (Object)"packageName");
        Preconditions.checkNotNull(requiredSignatures, (Object)"requiredSignatures");
        Preconditions.checkArgument((!requiredSignatures.isEmpty() ? 1 : 0) != 0, (Object)"requiredSignatures");
        final ImmutableList requiredSignaturesImmutable = ImmutableList.copyOf(requiredSignatures);
        for (Signature requiredSignature : requiredSignaturesImmutable) {
            Preconditions.checkNotNull((Object)requiredSignature);
        }
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                return SecurityPolicies.checkUidSignature(packageManager, uid, packageName, (ImmutableList<Signature>)requiredSignaturesImmutable);
            }
        };
    }

    public static SecurityPolicy oneOfSignatureSha256Hash(final PackageManager packageManager, final String packageName, List<byte[]> requiredSignatureSha256Hashes) {
        Preconditions.checkNotNull((Object)packageManager);
        Preconditions.checkNotNull((Object)packageName);
        Preconditions.checkNotNull(requiredSignatureSha256Hashes);
        Preconditions.checkArgument((!requiredSignatureSha256Hashes.isEmpty() ? 1 : 0) != 0);
        ImmutableList.Builder immutableListBuilder = ImmutableList.builder();
        for (byte[] requiredSignatureSha256Hash : requiredSignatureSha256Hashes) {
            Preconditions.checkNotNull((Object)requiredSignatureSha256Hash);
            Preconditions.checkArgument((requiredSignatureSha256Hash.length == 32 ? 1 : 0) != 0);
            immutableListBuilder.add((Object)Arrays.copyOf(requiredSignatureSha256Hash, requiredSignatureSha256Hash.length));
        }
        final ImmutableList requiredSignaturesHashesImmutable = immutableListBuilder.build();
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                return SecurityPolicies.checkUidSha256Signature(packageManager, uid, packageName, (ImmutableList<byte[]>)requiredSignaturesHashesImmutable);
            }
        };
    }

    @RequiresApi(value=18)
    public static SecurityPolicy isDeviceOwner(Context applicationContext) {
        DevicePolicyManager devicePolicyManager = (DevicePolicyManager)applicationContext.getSystemService("device_policy");
        return SecurityPolicies.anyPackageWithUidSatisfies(applicationContext, (Predicate<String>)((Predicate)pkg -> devicePolicyManager.isDeviceOwnerApp(pkg)), "Rejected by device owner policy. No packages found for UID.", "Rejected by device owner policy");
    }

    public static SecurityPolicy isProfileOwner(Context applicationContext) {
        DevicePolicyManager devicePolicyManager = (DevicePolicyManager)applicationContext.getSystemService("device_policy");
        return SecurityPolicies.anyPackageWithUidSatisfies(applicationContext, (Predicate<String>)((Predicate)pkg -> Build.VERSION.SDK_INT >= 21 && devicePolicyManager.isProfileOwnerApp(pkg)), "Rejected by profile owner policy. No packages found for UID.", "Rejected by profile owner policy");
    }

    public static SecurityPolicy isProfileOwnerOnOrganizationOwnedDevice(Context applicationContext) {
        DevicePolicyManager devicePolicyManager = (DevicePolicyManager)applicationContext.getSystemService("device_policy");
        return SecurityPolicies.anyPackageWithUidSatisfies(applicationContext, (Predicate<String>)((Predicate)pkg -> Build.VERSION.SDK_INT >= 30 && devicePolicyManager.isProfileOwnerApp(pkg) && devicePolicyManager.isOrganizationOwnedDeviceWithManagedProfile()), "Rejected by profile owner on organization-owned device policy. No packages found for UID.", "Rejected by profile owner on organization-owned device policy");
    }

    private static Status checkUidSignature(PackageManager packageManager, int uid, String packageName, ImmutableList<Signature> requiredSignatures) {
        String[] packages = packageManager.getPackagesForUid(uid);
        if (packages == null) {
            return Status.UNAUTHENTICATED.withDescription("Rejected by signature check security policy");
        }
        boolean packageNameMatched = false;
        for (String pkg : packages) {
            if (!packageName.equals(pkg)) continue;
            packageNameMatched = true;
            if (!SecurityPolicies.checkPackageSignature(packageManager, pkg, (Predicate<Signature>)((Predicate)arg_0 -> requiredSignatures.contains(arg_0)))) continue;
            return Status.OK;
        }
        return Status.PERMISSION_DENIED.withDescription("Rejected by signature check security policy. Package name matched: " + packageNameMatched);
    }

    private static Status checkUidSha256Signature(PackageManager packageManager, int uid, String packageName, ImmutableList<byte[]> requiredSignatureSha256Hashes) {
        String[] packages = packageManager.getPackagesForUid(uid);
        if (packages == null) {
            return Status.UNAUTHENTICATED.withDescription("Rejected by (SHA-256 hash signature check) security policy");
        }
        boolean packageNameMatched = false;
        for (String pkg : packages) {
            if (!packageName.equals(pkg)) continue;
            packageNameMatched = true;
            if (!SecurityPolicies.checkPackageSignature(packageManager, pkg, (Predicate<Signature>)((Predicate)signature -> SecurityPolicies.checkSignatureSha256HashesMatch(signature, (List<byte[]>)requiredSignatureSha256Hashes)))) continue;
            return Status.OK;
        }
        return Status.PERMISSION_DENIED.withDescription("Rejected by (SHA-256 hash signature check) security policy. Package name matched: " + packageNameMatched);
    }

    @SuppressLint(value={"PackageManagerGetSignatures"})
    private static boolean checkPackageSignature(PackageManager packageManager, String packageName, Predicate<Signature> signatureCheckFunction) {
        try {
            if (Build.VERSION.SDK_INT >= 28) {
                Signature[] signatures;
                PackageInfo packageInfo = packageManager.getPackageInfo(packageName, 0x8000000);
                if (packageInfo.signingInfo == null) {
                    return false;
                }
                for (Signature signature : signatures = packageInfo.signingInfo.hasMultipleSigners() ? packageInfo.signingInfo.getApkContentsSigners() : packageInfo.signingInfo.getSigningCertificateHistory()) {
                    if (!signatureCheckFunction.apply((Object)signature)) continue;
                    return true;
                }
            } else {
                PackageInfo packageInfo = packageManager.getPackageInfo(packageName, 64);
                if (packageInfo.signatures == null || packageInfo.signatures.length != 1) {
                    return false;
                }
                if (signatureCheckFunction.apply((Object)packageInfo.signatures[0])) {
                    return true;
                }
            }
        }
        catch (PackageManager.NameNotFoundException nnfe) {
            return false;
        }
        return false;
    }

    public static SecurityPolicy allOf(SecurityPolicy ... securityPolicies) {
        Preconditions.checkNotNull((Object)securityPolicies, (Object)"securityPolicies");
        Preconditions.checkArgument((securityPolicies.length > 0 ? 1 : 0) != 0, (Object)"securityPolicies must not be empty");
        return SecurityPolicies.allOfSecurityPolicy(securityPolicies);
    }

    private static SecurityPolicy allOfSecurityPolicy(final SecurityPolicy ... securityPolicies) {
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                for (SecurityPolicy policy : securityPolicies) {
                    Status checkAuth = policy.checkAuthorization(uid);
                    if (checkAuth.isOk()) continue;
                    return checkAuth;
                }
                return Status.OK;
            }
        };
    }

    public static SecurityPolicy anyOf(SecurityPolicy ... securityPolicies) {
        Preconditions.checkNotNull((Object)securityPolicies, (Object)"securityPolicies");
        Preconditions.checkArgument((securityPolicies.length > 0 ? 1 : 0) != 0, (Object)"securityPolicies must not be empty");
        return SecurityPolicies.anyOfSecurityPolicy(securityPolicies);
    }

    private static SecurityPolicy anyOfSecurityPolicy(final SecurityPolicy ... securityPolicies) {
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                ArrayList<Status> failed = new ArrayList<Status>();
                for (SecurityPolicy policy : securityPolicies) {
                    Status checkAuth = policy.checkAuthorization(uid);
                    if (checkAuth.isOk()) {
                        return checkAuth;
                    }
                    failed.add(checkAuth);
                }
                Iterator iter = failed.iterator();
                Status toReturn = (Status)iter.next();
                while (iter.hasNext()) {
                    Status append = (Status)iter.next();
                    toReturn = toReturn.augmentDescription(append.getDescription());
                    if (append.getCause() == null) continue;
                    if (toReturn.getCause() != null) {
                        toReturn.getCause().addSuppressed(append.getCause());
                        continue;
                    }
                    toReturn = toReturn.withCause(append.getCause());
                }
                return toReturn;
            }
        };
    }

    public static SecurityPolicy hasPermissions(final PackageManager packageManager, final ImmutableSet<String> permissions) {
        Preconditions.checkNotNull((Object)packageManager, (Object)"packageManager");
        Preconditions.checkNotNull(permissions, (Object)"permissions");
        Preconditions.checkArgument((!permissions.isEmpty() ? 1 : 0) != 0, (Object)"permissions");
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                return SecurityPolicies.checkPermissions(uid, packageManager, (ImmutableSet<String>)permissions);
            }
        };
    }

    private static Status checkPermissions(int uid, PackageManager packageManager, ImmutableSet<String> permissions) {
        String[] packages = packageManager.getPackagesForUid(uid);
        if (packages == null || packages.length == 0) {
            return Status.UNAUTHENTICATED.withDescription("Rejected by permission check security policy. No packages found for uid");
        }
        for (String pkg : packages) {
            for (String permission : permissions) {
                if (packageManager.checkPermission(permission, pkg) == 0) continue;
                return Status.PERMISSION_DENIED.withDescription("Rejected by permission check security policy. " + pkg + " does not have permission " + permission);
            }
        }
        return Status.OK;
    }

    private static SecurityPolicy anyPackageWithUidSatisfies(final Context applicationContext, final Predicate<String> condition, final String errorMessageForNoPackages, final String errorMessageForDenied) {
        return new SecurityPolicy(){

            @Override
            public Status checkAuthorization(int uid) {
                String[] packages = applicationContext.getPackageManager().getPackagesForUid(uid);
                if (packages == null || packages.length == 0) {
                    return Status.UNAUTHENTICATED.withDescription(errorMessageForNoPackages);
                }
                for (String pkg : packages) {
                    if (!condition.apply((Object)pkg)) continue;
                    return Status.OK;
                }
                return Status.PERMISSION_DENIED.withDescription(errorMessageForDenied);
            }
        };
    }

    private static boolean checkSignatureSha256HashesMatch(Signature signature, List<byte[]> expectedSignatureSha256Hashes) {
        byte[] signatureHash = SecurityPolicies.getSha256Hash(signature);
        for (byte[] hash : expectedSignatureSha256Hashes) {
            if (!Arrays.equals(hash, signatureHash)) continue;
            return true;
        }
        return false;
    }

    private static byte[] getSha256Hash(Signature signature) {
        return Hashing.sha256().hashBytes(signature.toByteArray()).asBytes();
    }
}

