/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.rls;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Ticker;
import io.grpc.rls.LruCache;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import javax.annotation.CheckReturnValue;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;

@ThreadSafe
abstract class LinkedHashLruCache<K, V>
implements LruCache<K, V> {
    private final Object lock;
    @GuardedBy(value="lock")
    private final LinkedHashMap<K, SizedValue> delegate;
    private final PeriodicCleaner periodicCleaner;
    private final Ticker ticker;
    private final LruCache.EvictionListener<K, SizedValue> evictionListener;
    private final AtomicLong estimatedSizeBytes = new AtomicLong();
    private long estimatedMaxSizeBytes;

    LinkedHashLruCache(long estimatedMaxSizeBytes, @Nullable LruCache.EvictionListener<K, V> evictionListener, int cleaningInterval, TimeUnit cleaningIntervalUnit, ScheduledExecutorService ses, final Ticker ticker, Object lock) {
        Preconditions.checkState((estimatedMaxSizeBytes > 0L ? 1 : 0) != 0, (Object)"max estimated cache size should be positive");
        this.estimatedMaxSizeBytes = estimatedMaxSizeBytes;
        this.lock = Preconditions.checkNotNull((Object)lock, (Object)"lock");
        this.evictionListener = new SizeHandlingEvictionListener(evictionListener);
        this.ticker = (Ticker)Preconditions.checkNotNull((Object)ticker, (Object)"ticker");
        this.delegate = new LinkedHashMap<K, SizedValue>(Math.max((int)(estimatedMaxSizeBytes / 1000L), 16), 0.75f, true){

            @Override
            protected boolean removeEldestEntry(Map.Entry<K, SizedValue> eldest) {
                boolean shouldRemove;
                if (LinkedHashLruCache.this.estimatedSizeBytes.get() <= LinkedHashLruCache.this.estimatedMaxSizeBytes) {
                    return false;
                }
                boolean removed = LinkedHashLruCache.this.cleanupExpiredEntries(1, ticker.read());
                boolean bl = shouldRemove = !removed && LinkedHashLruCache.this.shouldInvalidateEldestEntry(eldest.getKey(), eldest.getValue().value);
                if (shouldRemove) {
                    LinkedHashLruCache.this.invalidate(eldest.getKey(), LruCache.EvictionType.SIZE);
                }
                return false;
            }
        };
        this.periodicCleaner = new PeriodicCleaner(ses, cleaningInterval, cleaningIntervalUnit).start();
    }

    protected boolean shouldInvalidateEldestEntry(K eldestKey, V eldestValue) {
        return true;
    }

    protected abstract boolean isExpired(K var1, V var2, long var3);

    protected int estimateSizeOf(K key, V value) {
        return 1;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateEntrySize(K key) {
        Object object = this.lock;
        synchronized (object) {
            int newSize;
            SizedValue entry = this.readInternal(key);
            if (entry == null) {
                return;
            }
            int prevSize = entry.size;
            entry.size = newSize = this.estimateSizeOf(key, entry.value);
            this.estimatedSizeBytes.addAndGet(newSize - prevSize);
        }
    }

    public long estimatedSizeBytes() {
        return this.estimatedSizeBytes.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    @Nullable
    public final V cache(K key, V value) {
        SizedValue existing;
        Preconditions.checkNotNull(key, (Object)"key");
        Preconditions.checkNotNull(value, (Object)"value");
        int size = this.estimateSizeOf(key, value);
        Object object = this.lock;
        synchronized (object) {
            this.estimatedSizeBytes.addAndGet(size);
            existing = this.delegate.put(key, new SizedValue(size, value));
            if (existing != null) {
                this.evictionListener.onEviction(key, existing, LruCache.EvictionType.REPLACED);
            }
        }
        return existing == null ? null : (V)existing.value;
    }

    @Override
    @Nullable
    @CheckReturnValue
    public final V read(K key) {
        SizedValue entry = this.readInternal(key);
        if (entry != null) {
            return entry.value;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    @CheckReturnValue
    private SizedValue readInternal(K key) {
        Preconditions.checkNotNull(key, (Object)"key");
        Object object = this.lock;
        synchronized (object) {
            SizedValue existing = this.delegate.get(key);
            if (existing != null && this.isExpired(key, existing.value, this.ticker.read())) {
                this.invalidate(key, LruCache.EvictionType.EXPIRED);
                return null;
            }
            return existing;
        }
    }

    @Override
    @Nullable
    public final V invalidate(K key) {
        return this.invalidate(key, LruCache.EvictionType.EXPLICIT);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    private V invalidate(K key, LruCache.EvictionType cause) {
        Preconditions.checkNotNull(key, (Object)"key");
        Preconditions.checkNotNull((Object)((Object)cause), (Object)"cause");
        Object object = this.lock;
        synchronized (object) {
            SizedValue existing = (SizedValue)this.delegate.remove(key);
            if (existing != null) {
                this.evictionListener.onEviction(key, existing, cause);
            }
            return existing == null ? null : (V)existing.value;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final void invalidateAll() {
        Object object = this.lock;
        synchronized (object) {
            Iterator<Map.Entry<K, SizedValue>> iterator = this.delegate.entrySet().iterator();
            while (iterator.hasNext()) {
                Map.Entry<K, SizedValue> entry = iterator.next();
                if (entry.getValue() != null) {
                    this.evictionListener.onEviction(entry.getKey(), entry.getValue(), LruCache.EvictionType.EXPLICIT);
                }
                iterator.remove();
            }
        }
    }

    @Override
    @CheckReturnValue
    public final boolean hasCacheEntry(K key) {
        return this.readInternal(key) != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final List<V> values() {
        Object object = this.lock;
        synchronized (object) {
            ArrayList list = new ArrayList(this.delegate.size());
            for (SizedValue value : this.delegate.values()) {
                list.add(value.value);
            }
            return Collections.unmodifiableList(list);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public final void resize(int newSizeBytes) {
        long now = this.ticker.read();
        Object object = this.lock;
        synchronized (object) {
            this.estimatedMaxSizeBytes = newSizeBytes;
            if (this.estimatedSizeBytes.get() <= (long)newSizeBytes) {
                return;
            }
            this.cleanupExpiredEntries(now);
            Iterator<Map.Entry<K, SizedValue>> lruIter = this.delegate.entrySet().iterator();
            while (lruIter.hasNext() && this.estimatedMaxSizeBytes < this.estimatedSizeBytes.get()) {
                Map.Entry<K, SizedValue> entry = lruIter.next();
                lruIter.remove();
                this.evictionListener.onEviction(entry.getKey(), entry.getValue(), LruCache.EvictionType.SIZE);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    @CheckReturnValue
    public final int estimatedSize() {
        Object object = this.lock;
        synchronized (object) {
            return this.delegate.size();
        }
    }

    private boolean cleanupExpiredEntries(long now) {
        return this.cleanupExpiredEntries(Integer.MAX_VALUE, now);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean cleanupExpiredEntries(int maxExpiredEntries, long now) {
        Preconditions.checkArgument((maxExpiredEntries > 0 ? 1 : 0) != 0, (Object)"maxExpiredEntries must be positive");
        boolean removedAny = false;
        Object object = this.lock;
        synchronized (object) {
            Iterator<Map.Entry<K, SizedValue>> lruIter = this.delegate.entrySet().iterator();
            while (lruIter.hasNext() && maxExpiredEntries > 0) {
                Map.Entry<K, SizedValue> entry = lruIter.next();
                if (!this.isExpired(entry.getKey(), entry.getValue().value, now)) continue;
                lruIter.remove();
                this.evictionListener.onEviction(entry.getKey(), entry.getValue(), LruCache.EvictionType.EXPIRED);
                removedAny = true;
                --maxExpiredEntries;
            }
        }
        return removedAny;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public final void close() {
        Object object = this.lock;
        synchronized (object) {
            this.periodicCleaner.stop();
            this.invalidateAll();
        }
    }

    private final class SizedValue {
        volatile int size;
        final V value;

        SizedValue(int size, V value) {
            this.size = size;
            this.value = value;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            SizedValue that = (SizedValue)o;
            return Objects.equals(this.value, that.value);
        }

        public int hashCode() {
            return Objects.hash(this.value);
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("size", this.size).add("value", this.value).toString();
        }
    }

    private final class SizeHandlingEvictionListener
    implements LruCache.EvictionListener<K, SizedValue> {
        private final LruCache.EvictionListener<K, V> delegate;

        SizeHandlingEvictionListener(LruCache.EvictionListener<K, V> delegate) {
            this.delegate = delegate;
        }

        @Override
        public void onEviction(K key, SizedValue value, LruCache.EvictionType cause) {
            LinkedHashLruCache.this.estimatedSizeBytes.addAndGet(-1 * value.size);
            if (this.delegate != null) {
                this.delegate.onEviction(key, value.value, cause);
            }
        }
    }

    private final class PeriodicCleaner {
        private final ScheduledExecutorService ses;
        private final int interval;
        private final TimeUnit intervalUnit;
        private ScheduledFuture<?> scheduledFuture;

        PeriodicCleaner(ScheduledExecutorService ses, int interval, TimeUnit intervalUnit) {
            this.ses = (ScheduledExecutorService)Preconditions.checkNotNull((Object)ses, (Object)"ses");
            Preconditions.checkState((interval > 0 ? 1 : 0) != 0, (Object)"interval must be positive");
            this.interval = interval;
            this.intervalUnit = (TimeUnit)((Object)Preconditions.checkNotNull((Object)((Object)intervalUnit), (Object)"intervalUnit"));
        }

        PeriodicCleaner start() {
            Preconditions.checkState((this.scheduledFuture == null ? 1 : 0) != 0, (Object)"cleaning task can be started only once");
            this.scheduledFuture = this.ses.scheduleAtFixedRate(new CleaningTask(), this.interval, this.interval, this.intervalUnit);
            return this;
        }

        void stop() {
            if (this.scheduledFuture != null) {
                this.scheduledFuture.cancel(false);
                this.scheduledFuture = null;
            }
        }

        private class CleaningTask
        implements Runnable {
            private CleaningTask() {
            }

            @Override
            public void run() {
                LinkedHashLruCache.this.cleanupExpiredEntries(LinkedHashLruCache.this.ticker.read());
            }
        }
    }
}

