/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.services;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.primitives.Bytes;
import com.google.protobuf.ByteString;
import io.grpc.InternalMetadata;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.binarylog.Message;
import io.grpc.binarylog.Metadata;
import io.grpc.binarylog.MetadataEntry;
import io.grpc.binarylog.Peer;
import io.grpc.binarylog.Uint128;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nullable;

final class BinaryLog {
    private static final Logger logger = Logger.getLogger(BinaryLog.class.getName());
    private static final int IP_PORT_BYTES = 2;
    private static final int IP_PORT_UPPER_MASK = 65280;
    private static final int IP_PORT_LOWER_MASK = 255;
    private final int maxHeaderBytes;
    private final int maxMessageBytes;
    private static final Factory DEFAULT_FACTORY;
    private static final Factory NULL_FACTORY;

    @VisibleForTesting
    BinaryLog(int maxHeaderBytes, int maxMessageBytes) {
        this.maxHeaderBytes = maxHeaderBytes;
        this.maxMessageBytes = maxMessageBytes;
    }

    public boolean equals(Object o) {
        if (!(o instanceof BinaryLog)) {
            return false;
        }
        BinaryLog that = (BinaryLog)o;
        return this.maxHeaderBytes == that.maxHeaderBytes && this.maxMessageBytes == that.maxMessageBytes;
    }

    public int hashCode() {
        return Objects.hashCode((Object[])new Object[]{this.maxHeaderBytes, this.maxMessageBytes});
    }

    public String toString() {
        return this.getClass().getSimpleName() + '[' + "maxHeaderBytes=" + this.maxHeaderBytes + ", maxMessageBytes=" + this.maxMessageBytes + "]";
    }

    static BinaryLog getLog(String fullMethodName) {
        return DEFAULT_FACTORY.getLog(fullMethodName);
    }

    static Uint128 callIdToProto(byte[] bytes) {
        Preconditions.checkArgument((bytes.length == 16 ? 1 : 0) != 0, (Object)String.format("can only convert from 16 byte input, actual length = %d", bytes.length));
        ByteBuffer bb = ByteBuffer.wrap(bytes);
        long high = bb.getLong();
        long low = bb.getLong();
        return Uint128.newBuilder().setHigh(high).setLow(low).build();
    }

    @VisibleForTesting
    static Peer socketToProto(SocketAddress address) {
        Peer.PeerType peerType = Peer.PeerType.UNKNOWN_PEERTYPE;
        byte[] peerAddress = null;
        if (address instanceof InetSocketAddress) {
            InetAddress inetAddress = ((InetSocketAddress)address).getAddress();
            if (inetAddress instanceof Inet4Address) {
                peerType = Peer.PeerType.PEER_IPV4;
            } else if (inetAddress instanceof Inet6Address) {
                peerType = Peer.PeerType.PEER_IPV6;
            } else {
                logger.log(Level.SEVERE, "unknown type of InetSocketAddress: {}", address);
            }
            int port = ((InetSocketAddress)address).getPort();
            byte[] portBytes = new byte[]{(byte)((port & 0xFF00) >> 8), (byte)(port & 0xFF)};
            peerAddress = Bytes.concat((byte[][])new byte[][]{inetAddress.getAddress(), portBytes});
        } else if (address.getClass().getName().equals("io.netty.channel.unix.DomainSocketAddress")) {
            peerType = Peer.PeerType.PEER_UNIX;
        }
        if (peerAddress == null) {
            peerAddress = address.toString().getBytes(Charset.defaultCharset());
        }
        return Peer.newBuilder().setPeerType(peerType).setPeer(ByteString.copyFrom((byte[])peerAddress)).build();
    }

    @VisibleForTesting
    static io.grpc.binarylog.Metadata metadataToProto(Metadata metadata, int maxHeaderBytes) {
        Preconditions.checkState((maxHeaderBytes >= 0 ? 1 : 0) != 0);
        Metadata.Builder builder = io.grpc.binarylog.Metadata.newBuilder();
        if (maxHeaderBytes > 0) {
            byte[][] serialized = InternalMetadata.serialize((Metadata)metadata);
            int written = 0;
            for (int i = 0; i < serialized.length && written < maxHeaderBytes; i += 2) {
                byte[] key = serialized[i];
                byte[] value = serialized[i + 1];
                if (written + key.length + value.length > maxHeaderBytes) continue;
                builder.addEntry(MetadataEntry.newBuilder().setKey(ByteString.copyFrom((byte[])key)).setValue(ByteString.copyFrom((byte[])value)).build());
                written += key.length;
                written += value.length;
            }
        }
        return builder.build();
    }

    @VisibleForTesting
    static Message messageToProto(byte[] message, boolean compressed, int maxMessageBytes) {
        Preconditions.checkState((maxMessageBytes >= 0 ? 1 : 0) != 0);
        Message.Builder builder = Message.newBuilder().setFlags(BinaryLog.flagsForMessage(compressed)).setLength(message.length);
        if (maxMessageBytes > 0) {
            int limit = Math.min(maxMessageBytes, message.length);
            builder.setData(ByteString.copyFrom((byte[])message, (int)0, (int)limit));
        }
        return builder.build();
    }

    @VisibleForTesting
    static int flagsForMessage(boolean compressed) {
        return compressed ? 1 : 0;
    }

    static {
        Factory defaultFactory = NULL_FACTORY = new NullFactory();
        try {
            String configStr = System.getenv("GRPC_BINARY_LOG_CONFIG");
            if (configStr != null && configStr.length() > 0) {
                defaultFactory = new FactoryImpl(configStr);
            }
        }
        catch (Throwable t) {
            logger.log(Level.SEVERE, "Failed to initialize binary log. Disabling binary log.", t);
            defaultFactory = NULL_FACTORY;
        }
        DEFAULT_FACTORY = defaultFactory;
    }

    private static final class NullFactory
    implements Factory {
        private NullFactory() {
        }

        @Override
        public BinaryLog getLog(String fullMethodName) {
            return null;
        }
    }

    static final class FactoryImpl
    implements Factory {
        private static final Pattern logPatternRe = Pattern.compile("[^{]+");
        private static final Pattern logOptionsRe = Pattern.compile("\\{[^}]+}");
        private static final Pattern configRe = Pattern.compile(String.format("^(%s)(%s)?$", logPatternRe.pattern(), logOptionsRe.pattern()));
        private static final Pattern msgRe = Pattern.compile("\\{m(?::(\\d+))?}");
        private static final Pattern headerRe = Pattern.compile("\\{h(?::(\\d+))?}");
        private static final Pattern bothRe = Pattern.compile("\\{h(?::(\\d+))?;m(?::(\\d+))?}");
        private final BinaryLog globalLog;
        private final Map<String, BinaryLog> perServiceLogs;
        private final Map<String, BinaryLog> perMethodLogs;

        @VisibleForTesting
        FactoryImpl(String configurationString) {
            Preconditions.checkState((configurationString != null && configurationString.length() > 0 ? 1 : 0) != 0);
            BinaryLog globalLog = null;
            HashMap<String, BinaryLog> perServiceLogs = new HashMap<String, BinaryLog>();
            HashMap<String, BinaryLog> perMethodLogs = new HashMap<String, BinaryLog>();
            for (String configuration : Splitter.on((char)',').split((CharSequence)configurationString)) {
                Matcher configMatcher = configRe.matcher(configuration);
                if (!configMatcher.matches()) {
                    throw new IllegalArgumentException("Bad input: " + configuration);
                }
                String methodOrSvc = configMatcher.group(1);
                String binlogOptionStr = configMatcher.group(2);
                BinaryLog binLog = FactoryImpl.createBinaryLog(binlogOptionStr);
                if (binLog == null) continue;
                if (methodOrSvc.equals("*")) {
                    if (globalLog != null) {
                        logger.log(Level.SEVERE, "Ignoring duplicate entry: " + configuration);
                        continue;
                    }
                    globalLog = binLog;
                    logger.info("Global binlog: " + globalLog);
                    continue;
                }
                if (FactoryImpl.isServiceGlob(methodOrSvc)) {
                    String service = MethodDescriptor.extractFullServiceName((String)methodOrSvc);
                    if (perServiceLogs.containsKey(service)) {
                        logger.log(Level.SEVERE, "Ignoring duplicate entry: " + configuration);
                        continue;
                    }
                    perServiceLogs.put(service, binLog);
                    logger.info(String.format("Service binlog: service=%s log=%s", service, binLog));
                    continue;
                }
                if (perMethodLogs.containsKey(methodOrSvc)) {
                    logger.log(Level.SEVERE, "Ignoring duplicate entry: " + configuration);
                    continue;
                }
                perMethodLogs.put(methodOrSvc, binLog);
                logger.info(String.format("Method binlog: method=%s log=%s", methodOrSvc, binLog));
            }
            this.globalLog = globalLog;
            this.perServiceLogs = Collections.unmodifiableMap(perServiceLogs);
            this.perMethodLogs = Collections.unmodifiableMap(perMethodLogs);
        }

        @Override
        public BinaryLog getLog(String fullMethodName) {
            BinaryLog methodLog = this.perMethodLogs.get(fullMethodName);
            if (methodLog != null) {
                return methodLog;
            }
            BinaryLog serviceLog = this.perServiceLogs.get(MethodDescriptor.extractFullServiceName((String)fullMethodName));
            if (serviceLog != null) {
                return serviceLog;
            }
            return this.globalLog;
        }

        @Nullable
        @VisibleForTesting
        static BinaryLog createBinaryLog(@Nullable String logConfig) {
            if (logConfig == null) {
                return new BinaryLog(Integer.MAX_VALUE, Integer.MAX_VALUE);
            }
            try {
                int maxMsgBytes;
                int maxHeaderBytes;
                Matcher headerMatcher = headerRe.matcher(logConfig);
                if (headerMatcher.matches()) {
                    String maxHeaderStr = headerMatcher.group(1);
                    maxHeaderBytes = maxHeaderStr != null ? Integer.parseInt(maxHeaderStr) : Integer.MAX_VALUE;
                    maxMsgBytes = 0;
                } else {
                    Matcher msgMatcher = msgRe.matcher(logConfig);
                    if (msgMatcher.matches()) {
                        maxHeaderBytes = 0;
                        String maxMsgStr = msgMatcher.group(1);
                        maxMsgBytes = maxMsgStr != null ? Integer.parseInt(maxMsgStr) : Integer.MAX_VALUE;
                    } else {
                        Matcher bothMatcher = bothRe.matcher(logConfig);
                        if (bothMatcher.matches()) {
                            String maxHeaderStr = bothMatcher.group(1);
                            String maxMsgStr = bothMatcher.group(2);
                            maxHeaderBytes = maxHeaderStr != null ? Integer.parseInt(maxHeaderStr) : Integer.MAX_VALUE;
                            maxMsgBytes = maxMsgStr != null ? Integer.parseInt(maxMsgStr) : Integer.MAX_VALUE;
                        } else {
                            logger.log(Level.SEVERE, "Illegal log config pattern: " + logConfig);
                            return null;
                        }
                    }
                }
                return new BinaryLog(maxHeaderBytes, maxMsgBytes);
            }
            catch (NumberFormatException e) {
                logger.log(Level.SEVERE, "Illegal log config pattern: " + logConfig);
                return null;
            }
        }

        static boolean isServiceGlob(String input) {
            return input.endsWith("/*");
        }
    }

    static interface Factory {
        @Nullable
        public BinaryLog getLog(String var1);
    }
}

