/*
 * Copyright 2016 Game Server Services, Inc. or its affiliates. All Rights
 * Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License").
 * You may not use this file except in compliance with the License.
 * A copy of the License is located at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * or in the "license" file accompanying this file. This file is distributed
 * on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

package io.gs2.gateway;

import java.util.ArrayList;
import java.util.List;

import io.gs2.model.Region;
import io.gs2.util.EncodingUtil;
import org.apache.http.NameValuePair;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.utils.URLEncodedUtils;
import org.apache.http.message.BasicNameValuePair;

import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.JsonNode;
import org.json.JSONObject;

import io.gs2.AbstractGs2Client;
import io.gs2.Gs2Constant;
import io.gs2.model.IGs2Credential;
import io.gs2.gateway.request.*;
import io.gs2.gateway.result.*;
import io.gs2.gateway.model.*;

/**
 * GS2 Gateway API クライアント
 *
 * @author Game Server Services, Inc.
 *
 */
public class Gs2GatewayRestClient extends AbstractGs2Client<Gs2GatewayRestClient> {

	public static String ENDPOINT = "gateway";

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 */
	public Gs2GatewayRestClient(IGs2Credential credential) {
		super(credential);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2GatewayRestClient(IGs2Credential credential, Region region) {
		super(credential, region);
	}

	/**
	 * コンストラクタ。
	 *
	 * @param credential 認証情報
	 * @param region リージョン
	 */
	public Gs2GatewayRestClient(IGs2Credential credential, String region) {
		super(credential, region);

	}

	/**
	 * ネームスペースの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeNamespacesResult describeNamespaces(DescribeNamespacesRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.describeNamespaces";
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DescribeNamespacesResult.class);
    }

	/**
	 * ネームスペースを新規作成<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public CreateNamespaceResult createNamespace(CreateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.createNamespace";
        String name = null;
        if(request.getName() != null) {
            name = request.getName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String firebaseSecret = null;
        if(request.getFirebaseSecret() != null) {
            firebaseSecret = request.getFirebaseSecret();
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(name != null) {
            _body.put("name", name);
        }
        if(description != null) {
            _body.put("description", description);
        }
        if(firebaseSecret != null) {
            _body.put("firebaseSecret", firebaseSecret);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, CreateNamespaceResult.class);
    }

	/**
	 * ネームスペースを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceStatusResult getNamespaceStatus(GetNamespaceStatusRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.getNamespaceStatus";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceStatusResult.class);
    }

	/**
	 * ネームスペースを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetNamespaceResult getNamespace(GetNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.getNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetNamespaceResult.class);
    }

	/**
	 * ネームスペースを更新<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public UpdateNamespaceResult updateNamespace(UpdateNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.updateNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String description = null;
        if(request.getDescription() != null) {
            description = request.getDescription();
        }
        String firebaseSecret = null;
        if(request.getFirebaseSecret() != null) {
            firebaseSecret = request.getFirebaseSecret();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(description != null) {
            _body.put("description", description);
        }
        if(firebaseSecret != null) {
            _body.put("firebaseSecret", firebaseSecret);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, UpdateNamespaceResult.class);
    }

	/**
	 * ネームスペースを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteNamespaceResult deleteNamespace(DeleteNamespaceRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FNamespaceFunctionHandler.deleteNamespace";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, DeleteNamespaceResult.class);
    }

	/**
	 * Websocketセッションの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeWebSocketSessionsResult describeWebSocketSessions(DescribeWebSocketSessionsRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.describeWebSocketSessions";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeWebSocketSessionsResult.class);
    }

	/**
	 * ユーザIDを指定してWebsocketセッションの一覧を取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DescribeWebSocketSessionsByUserIdResult describeWebSocketSessionsByUserId(DescribeWebSocketSessionsByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.describeWebSocketSessionsByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String pageToken = String.valueOf(request.getPageToken());
        String limit = String.valueOf(request.getLimit());
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(pageToken != null) {
            _queryString.add(new BasicNameValuePair("pageToken", String.valueOf(pageToken)));
        }
        if(limit != null) {
            _queryString.add(new BasicNameValuePair("limit", String.valueOf(limit)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DescribeWebSocketSessionsByUserIdResult.class);
    }

	/**
	 * WebsocketセッションにユーザIDを設定<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SetUserIdResult setUserId(SetUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.setUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String connectionId = null;
        if(request.getConnectionId() != null) {
            connectionId = request.getConnectionId();
        }
        Boolean allowConcurrentAccess = null;
        if(request.getAllowConcurrentAccess() != null) {
            allowConcurrentAccess = request.getAllowConcurrentAccess();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(allowConcurrentAccess != null) {
            _body.put("allowConcurrentAccess", allowConcurrentAccess);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SetUserIdResult.class);
    }

	/**
	 * WebsocketセッションにユーザIDを設定<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SetUserIdByUserIdResult setUserIdByUserId(SetUserIdByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.setUserIdByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String connectionId = null;
        if(request.getConnectionId() != null) {
            connectionId = request.getConnectionId();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        Boolean allowConcurrentAccess = null;
        if(request.getAllowConcurrentAccess() != null) {
            allowConcurrentAccess = request.getAllowConcurrentAccess();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(allowConcurrentAccess != null) {
            _body.put("allowConcurrentAccess", allowConcurrentAccess);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SetUserIdByUserIdResult.class);
    }

	/**
	 * Websocketセッションを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetWebSocketSessionResult getWebSocketSession(GetWebSocketSessionRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.getWebSocketSession";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String connectionId = null;
        if(request.getConnectionId() != null) {
            connectionId = request.getConnectionId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetWebSocketSessionResult.class);
    }

	/**
	 * ユーザIDを指定してWebsocketセッションを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetWebSocketSessionByConnectionIdResult getWebSocketSessionByConnectionId(GetWebSocketSessionByConnectionIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.getWebSocketSessionByConnectionId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String connectionId = null;
        if(request.getConnectionId() != null) {
            connectionId = request.getConnectionId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(connectionId != null) {
            _queryString.add(new BasicNameValuePair("connectionId", String.valueOf(connectionId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
		return doRequest(http, GetWebSocketSessionByConnectionIdResult.class);
    }

	/**
	 * 通知を送信<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SendNotificationResult sendNotification(SendNotificationRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FWebSocketSessionFunctionHandler.sendNotification";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String issuer = null;
        if(request.getIssuer() != null) {
            issuer = request.getIssuer();
        }
        String subject = null;
        if(request.getSubject() != null) {
            subject = request.getSubject();
        }
        String payload = null;
        if(request.getPayload() != null) {
            payload = request.getPayload();
        }
        Boolean enableTransferMobileNotification = null;
        if(request.getEnableTransferMobileNotification() != null) {
            enableTransferMobileNotification = request.getEnableTransferMobileNotification();
        }
        String sound = null;
        if(request.getSound() != null) {
            sound = request.getSound();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(subject != null) {
            _body.put("subject", subject);
        }
        if(payload != null) {
            _body.put("payload", payload);
        }
        if(enableTransferMobileNotification != null) {
            _body.put("enableTransferMobileNotification", enableTransferMobileNotification);
        }
        if(sound != null) {
            _body.put("sound", sound);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SendNotificationResult.class);
    }

	/**
	 * デバイストークンを設定<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SetFirebaseTokenResult setFirebaseToken(SetFirebaseTokenRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.setFirebaseToken";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String token = null;
        if(request.getToken() != null) {
            token = request.getToken();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(token != null) {
            _body.put("token", token);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SetFirebaseTokenResult.class);
    }

	/**
	 * ユーザIDを指定してデバイストークンを設定<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SetFirebaseTokenByUserIdResult setFirebaseTokenByUserId(SetFirebaseTokenByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.setFirebaseTokenByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String token = null;
        if(request.getToken() != null) {
            token = request.getToken();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(token != null) {
            _body.put("token", token);
        }
		HttpPut http = createHttpPut(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SetFirebaseTokenByUserIdResult.class);
    }

	/**
	 * Firebaseデバイストークンを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetFirebaseTokenResult getFirebaseToken(GetFirebaseTokenRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.getFirebaseToken";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetFirebaseTokenResult.class);
    }

	/**
	 * ユーザIDを指定してFirebaseデバイストークンを取得<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public GetFirebaseTokenByUserIdResult getFirebaseTokenByUserId(GetFirebaseTokenByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.getFirebaseTokenByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpGet http = createHttpGet(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, GetFirebaseTokenByUserIdResult.class);
    }

	/**
	 * Firebaseデバイストークンを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteFirebaseTokenResult deleteFirebaseToken(DeleteFirebaseTokenRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.deleteFirebaseToken";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        http.setHeader("X-GS2-ACCESS-TOKEN", request.getAccessToken());
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteFirebaseTokenResult.class);
    }

	/**
	 * ユーザIDを指定してFirebaseデバイストークンを削除<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public DeleteFirebaseTokenByUserIdResult deleteFirebaseTokenByUserId(DeleteFirebaseTokenByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.deleteFirebaseTokenByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
		HttpDelete http = createHttpDelete(
				url,
				ENDPOINT
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, DeleteFirebaseTokenByUserIdResult.class);
    }

	/**
	 * モバイルプッシュ通知を送信<br>
	 *
	 * @param request リクエストパラメータ
	 * @return 結果
	 */
	public SendMobileNotificationByUserIdResult sendMobileNotificationByUserId(SendMobileNotificationByUserIdRequest request) {
	    String url = Gs2Constant.ENDPOINT_HOST + "/gateway-handler?handler=gs2_gateway%2Fhandler%2FFirebaseTokenFunctionHandler.sendMobileNotificationByUserId";
        String namespaceName = null;
        if(request.getNamespaceName() != null) {
            namespaceName = request.getNamespaceName();
        }
        String userId = null;
        if(request.getUserId() != null) {
            userId = request.getUserId();
        }
        String subject = null;
        if(request.getSubject() != null) {
            subject = request.getSubject();
        }
        String payload = null;
        if(request.getPayload() != null) {
            payload = request.getPayload();
        }
        String sound = null;
        if(request.getSound() != null) {
            sound = request.getSound();
        }
        List<NameValuePair> _queryString = new ArrayList<>();
        if(namespaceName != null) {
            _queryString.add(new BasicNameValuePair("namespaceName", String.valueOf(namespaceName)));
        }
        if(userId != null) {
            _queryString.add(new BasicNameValuePair("userId", String.valueOf(userId)));
        }
        if(_queryString.size() > 0) {
            url += "&" + URLEncodedUtils.format(_queryString, "UTF-8");
        }

		ObjectNode _body = JsonNodeFactory.instance.objectNode();
        if(subject != null) {
            _body.put("subject", subject);
        }
        if(payload != null) {
            _body.put("payload", payload);
        }
        if(sound != null) {
            _body.put("sound", sound);
        }
		HttpPost http = createHttpPost(
				url,
				ENDPOINT,
				_body.toString()
        );
        if(request.getRequestId() != null) {
            http.setHeader("X-GS2-REQUEST-ID", request.getRequestId());
        }
        if(request.getDuplicationAvoider() != null) {
            http.setHeader("X-GS2-DUPLICATION-AVOIDER", request.getDuplicationAvoider());
        }
		return doRequest(http, SendMobileNotificationByUserIdResult.class);
    }
}