/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.build.common;

import io.helidon.build.common.OSType;
import io.helidon.build.common.Requirements;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.CopyOption;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermission;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Base64;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.Random;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiPredicate;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public final class FileUtils {
    private static final Map<String, String> FS_ENV = Map.of("create", "true", "enablePosixFileAttributes", "true");
    private static final boolean IS_WINDOWS = OSType.currentOS() == OSType.Windows;
    private static final Path TMPDIR = Path.of(System.getProperty("java.io.tmpdir"), new String[0]);
    private static final Random RANDOM = new Random();
    public static final Path WORKING_DIR = FileUtils.requiredDirectoryFromProperty("user.dir", false);
    public static final Path USER_HOME_DIR = FileUtils.requiredDirectoryFromProperty("user.home", false);
    private static final OSType OS = OSType.currentOS();
    private static final String JAVA_BINARY_NAME = OS.javaExecutable();
    private static final String JAVA_HOME_VAR = "JAVA_HOME";
    private static final String PATH_VAR = "PATH";
    private static final String BIN_DIR_NAME = "bin";
    private static final long KB = 1024L;
    private static final long MB = 0x100000L;
    private static final long GB = 0x40000000L;
    private static final long TB = 0x10000000000L;
    private static final DecimalFormat DF = new DecimalFormat("0.00");

    public static Path requiredDirectoryFromProperty(String systemPropertyName, boolean createIfRequired) {
        String path = Requirements.requireNonNull(System.getProperty(systemPropertyName), "Required system property %s not set", systemPropertyName);
        return FileUtils.requiredDirectory(path, createIfRequired);
    }

    public static Path requiredDirectory(String path, boolean createIfRequired) {
        Path dir = Path.of(Objects.requireNonNull(path, "valid path required"), new String[0]);
        return createIfRequired ? FileUtils.ensureDirectory(dir, new FileAttribute[0]) : FileUtils.requireDirectory(dir);
    }

    public static Path fromWorking(Path path) {
        try {
            Path relativePath = WORKING_DIR.relativize(path);
            if (relativePath.getName(0).toString().equals("..")) {
                return path;
            }
            return relativePath;
        }
        catch (IllegalArgumentException e) {
            return path;
        }
    }

    public static Path ensureDirectory(Path path, FileAttribute<?> ... attrs) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            return FileUtils.requireDirectory(path);
        }
        try {
            return Files.createDirectories(path, attrs);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Path copyDirectory(Path source, Path destination) {
        FileUtils.requireNonExistent(destination);
        try (Stream<Path> stream = Files.walk(source, new FileVisitOption[0]);){
            stream.forEach(src -> {
                try {
                    Path dst = destination.resolve(source.relativize((Path)src));
                    if (Files.isDirectory(src, new LinkOption[0])) {
                        Files.createDirectory(dst, new FileAttribute[0]);
                    } else {
                        Files.copy(src, dst, StandardCopyOption.COPY_ATTRIBUTES);
                    }
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            });
        }
        catch (UncheckedIOException e) {
            throw e;
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
        return destination.toAbsolutePath().normalize();
    }

    public static List<Path> listFiles(Path directory, Predicate<String> fileNameFilter) {
        return FileUtils.listFiles(directory, fileNameFilter, 1);
    }

    public static List<Path> listFiles(Path directory, BiPredicate<Path, BasicFileAttributes> pathFilter) {
        return FileUtils.listFiles(directory, pathFilter, 1);
    }

    public static List<Path> listFiles(Path directory, Predicate<String> fileNameFilter, int maxDepth) {
        return FileUtils.listFiles(directory, (Path path, BasicFileAttributes attrs) -> fileNameFilter.test(path.getFileName().toString()), maxDepth);
    }

    public static List<Path> listFiles(Path directory, BiPredicate<Path, BasicFileAttributes> pathFilter, int maxDepth) {
        List<Path> list;
        block8: {
            Stream<Path> pathStream = Files.find(FileUtils.requireDirectory(directory), maxDepth, pathFilter, new FileVisitOption[0]);
            try {
                list = pathStream.collect(Collectors.toList());
                if (pathStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (pathStream != null) {
                        try {
                            pathStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            pathStream.close();
        }
        return list;
    }

    public static List<Path> list(Path directory) {
        return FileUtils.list(directory, 1);
    }

    public static List<Path> list(Path directory, int maxDepth) {
        List<Path> list;
        block8: {
            Stream<Path> pathStream = Files.find(FileUtils.requireDirectory(directory), maxDepth, (path, attrs) -> true, new FileVisitOption[0]);
            try {
                list = pathStream.collect(Collectors.toList());
                if (pathStream == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (pathStream != null) {
                        try {
                            pathStream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            pathStream.close();
        }
        return list;
    }

    public static Path requireDirectory(Path directory) {
        Path result = FileUtils.requireExistent(directory);
        if (Files.isDirectory(result, new LinkOption[0])) {
            return result;
        }
        throw new IllegalArgumentException(directory + " is not a directory");
    }

    public static Path requireFile(Path file) {
        Path result = FileUtils.requireExistent(file);
        if (Files.isRegularFile(result, new LinkOption[0])) {
            return result;
        }
        throw new IllegalArgumentException(file + " is not a file");
    }

    public static Path requireExistent(Path path) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            return path.toAbsolutePath().normalize();
        }
        throw new IllegalArgumentException(path + " does not exist");
    }

    public static Path requireNonExistent(Path path) {
        if (Files.exists(Objects.requireNonNull(path), new LinkOption[0])) {
            throw new IllegalArgumentException(path + " exists");
        }
        return path.toAbsolutePath().normalize();
    }

    public static Path delete(Path fileOrDirectory) {
        if (Files.exists(fileOrDirectory, new LinkOption[0])) {
            if (Files.isRegularFile(fileOrDirectory, new LinkOption[0])) {
                try {
                    Files.delete(fileOrDirectory);
                }
                catch (IOException ioe) {
                    throw new UncheckedIOException(ioe);
                }
            } else {
                FileUtils.deleteDirectory(fileOrDirectory);
            }
        }
        return fileOrDirectory;
    }

    public static Path deleteDirectory(Path directory) {
        block10: {
            if (Files.exists(directory, new LinkOption[0])) {
                if (Files.isDirectory(directory, new LinkOption[0])) {
                    try (Stream<Path> stream = Files.walk(directory, new FileVisitOption[0]);){
                        stream.sorted(Comparator.reverseOrder()).forEach(file -> {
                            try {
                                Files.delete(file);
                            }
                            catch (IOException ioe) {
                                throw new UncheckedIOException(ioe);
                            }
                        });
                        break block10;
                    }
                    catch (IOException ioe) {
                        throw new UncheckedIOException(ioe);
                    }
                }
                throw new IllegalArgumentException(directory + " is not a directory");
            }
        }
        return directory;
    }

    public static Path deleteDirectoryContent(Path directory) throws IOException {
        if (Files.exists(directory, new LinkOption[0])) {
            if (Files.isDirectory(directory, new LinkOption[0])) {
                try (Stream<Path> stream = Files.walk(directory, new FileVisitOption[0]);){
                    stream.sorted(Comparator.reverseOrder()).filter(file -> !file.equals(directory)).forEach(file -> {
                        try {
                            Files.delete(file);
                        }
                        catch (IOException e) {
                            throw new UncheckedIOException(e);
                        }
                    });
                }
            } else {
                throw new IllegalArgumentException(directory + " is not a directory");
            }
        }
        return directory;
    }

    public static long sizeOf(Path path) {
        try {
            if (Files.isRegularFile(path, new LinkOption[0])) {
                return Files.size(path);
            }
            final AtomicLong size = new AtomicLong();
            Files.walkFileTree(path, (FileVisitor<? super Path>)new FileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    size.addAndGet(attrs.size());
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult visitFileFailed(Path file, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
                    return FileVisitResult.CONTINUE;
                }
            });
            return size.get();
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static long lastModifiedSeconds(Path file) {
        if (Files.exists(file, new LinkOption[0])) {
            return FileUtils.lastModifiedTime(file).to(TimeUnit.SECONDS);
        }
        return 0L;
    }

    public static long lastModifiedMillis(Path file) {
        if (Files.exists(file, new LinkOption[0])) {
            return FileUtils.lastModifiedTime(file).to(TimeUnit.MILLISECONDS);
        }
        return 0L;
    }

    public static FileTime lastModifiedTime(Path file) {
        try {
            return Files.getLastModifiedTime(file, new LinkOption[0]);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Optional<FileTime> newerThan(Path file, FileTime baseTime) {
        FileTime modTime = FileUtils.lastModifiedTime(file);
        if (FileUtils.newerThan(modTime, baseTime)) {
            return Optional.of(modTime);
        }
        return Optional.empty();
    }

    public static Optional<FileTime> olderThan(Path file, FileTime baseTime) {
        FileTime modTime = FileUtils.lastModifiedTime(file);
        if (FileUtils.olderThan(modTime, baseTime)) {
            return Optional.of(modTime);
        }
        return Optional.empty();
    }

    public static boolean newerThan(FileTime changeTime, FileTime baseTime) {
        return baseTime == null || changeTime.compareTo(baseTime) > 0;
    }

    public static boolean olderThan(FileTime changeTime, FileTime baseTime) {
        return baseTime == null || changeTime.compareTo(baseTime) < 0;
    }

    public static String fileName(Path file) {
        return Objects.requireNonNull(file.getFileName()).toString();
    }

    public static Optional<Path> findExecutableInPath(String executableName) {
        return Arrays.stream(Objects.requireNonNull(System.getenv(PATH_VAR)).split(File.pathSeparator)).map(x$0 -> Paths.get(x$0, new String[0])).map(path -> path.resolve(executableName)).filter(Files::isExecutable).findFirst();
    }

    public static Optional<Path> javaExecutable() {
        Optional<Path> path = FileUtils.javaExecutableInPath();
        if (path.isPresent()) {
            return path;
        }
        return FileUtils.javaExecutableInJavaHome();
    }

    public static Path requireJavaExecutable() {
        return FileUtils.javaExecutable().orElseThrow(() -> new IllegalStateException(JAVA_BINARY_NAME + " not found. Please add it to your PATH or set the JAVA_HOME or variable."));
    }

    public static Optional<Path> javaExecutableInPath() {
        return FileUtils.findExecutableInPath(JAVA_BINARY_NAME);
    }

    public static Optional<Path> javaExecutableInJavaHome() {
        String javaHomePath = System.getenv(JAVA_HOME_VAR);
        if (javaHomePath != null) {
            Path javaHome = Paths.get(javaHomePath, new String[0]);
            Path binary = javaHome.resolve(BIN_DIR_NAME).resolve(JAVA_BINARY_NAME);
            if (Files.isExecutable(binary)) {
                return Optional.of(binary);
            }
            throw new IllegalStateException(JAVA_BINARY_NAME + " not found in JAVA_HOME path: " + javaHomePath);
        }
        return Optional.empty();
    }

    public static Path ensureFile(Path file) {
        if (!Files.exists(file, new LinkOption[0])) {
            try {
                Files.createFile(file, new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return file;
    }

    public static Path touch(Path file) {
        if (Files.exists(file, new LinkOption[0])) {
            long currentTime = System.currentTimeMillis();
            long lastModified = FileUtils.lastModifiedSeconds(file);
            long lastModifiedPlusOneSecond = lastModified + 1000L;
            long newTime = Math.max(currentTime, lastModifiedPlusOneSecond);
            try {
                Files.setLastModifiedTime(file, FileTime.fromMillis(newTime));
                return file;
            }
            catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        }
        return FileUtils.ensureFile(file);
    }

    public static String javaHome() {
        String javaHome = System.getProperty("java.home");
        if (javaHome == null) {
            javaHome = FileUtils.requireJavaExecutable().getParent().getParent().toString();
        }
        return javaHome;
    }

    public static Path unique(Path directory, String name, String suffix) {
        Path path = directory.resolve(name + suffix);
        int i = 1;
        while (Files.exists(path, new LinkOption[0])) {
            path = directory.resolve(name + "-" + i + suffix);
            ++i;
        }
        return path;
    }

    public static Path unique(Path directory, String name) {
        return FileUtils.unique(directory, name, "");
    }

    public static String toBase64(Path path) {
        try {
            byte[] byteCode = Files.readAllBytes(path);
            return new String(Base64.getEncoder().encode(byteCode), StandardCharsets.UTF_8);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
    }

    public static FileSystem newZipFileSystem(Path zip) {
        Object uriPrefix = "jar:file:";
        if (IS_WINDOWS) {
            uriPrefix = (String)uriPrefix + "/";
        }
        URI uri = URI.create((String)uriPrefix + zip.toString().replace("\\", "/"));
        try {
            return FileSystems.newFileSystem(uri, FS_ENV);
        }
        catch (IOException ioe) {
            throw new UncheckedIOException(ioe);
        }
    }

    public static Path zip(Path zip, Path directory) {
        return FileUtils.zip(zip, directory, path -> {});
    }

    public static Path zip(Path zip, Path directory, Consumer<Path> fileConsumer) {
        Path path;
        block15: {
            FileUtils.ensureDirectory(zip.getParent(), new FileAttribute[0]);
            FileSystem fs = FileUtils.newZipFileSystem(zip);
            try {
                try (Stream<Path> entries = Files.walk(directory, new FileVisitOption[0]);){
                    entries.sorted(Comparator.reverseOrder()).filter(p -> Files.isRegularFile(p, new LinkOption[0]) && !p.equals(zip)).map(p -> {
                        try {
                            Path target = fs.getPath(directory.relativize((Path)p).toString(), new String[0]);
                            Path parent = target.getParent();
                            if (parent != null) {
                                Files.createDirectories(parent, new FileAttribute[0]);
                            }
                            Files.copy(p, target, StandardCopyOption.REPLACE_EXISTING);
                            return target;
                        }
                        catch (IOException ioe) {
                            throw new UncheckedIOException(ioe);
                        }
                    }).forEach(fileConsumer);
                }
                path = zip;
                if (fs == null) break block15;
            }
            catch (Throwable throwable) {
                try {
                    if (fs != null) {
                        try {
                            fs.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException ioe) {
                    throw new UncheckedIOException(ioe);
                }
            }
            fs.close();
        }
        return path;
    }

    public static boolean isPosix(Path path) {
        return path.getFileSystem().supportedFileAttributeViews().contains("posix");
    }

    public static void unzip(Path zip, Path directory) {
        try (FileSystem fs = FileUtils.newZipFileSystem(zip);){
            FileUtils.ensureDirectory(directory, new FileAttribute[0]);
            boolean posix = FileUtils.isPosix(directory);
            Path root = fs.getRootDirectories().iterator().next();
            try (Stream<Path> entries = Files.walk(root, new FileVisitOption[0]);){
                entries.filter(p -> !p.equals(root)).forEach(file -> {
                    Path filePath = directory.resolve(Path.of(file.toString().substring(1), new String[0]));
                    try {
                        if (Files.isDirectory(file, new LinkOption[0])) {
                            Files.createDirectories(filePath, new FileAttribute[0]);
                        } else {
                            Files.copy(file, filePath, new CopyOption[0]);
                        }
                        if (posix) {
                            Set<PosixFilePermission> perms = Files.getPosixFilePermissions(file, new LinkOption[0]);
                            Files.setPosixFilePermissions(filePath, perms);
                        }
                    }
                    catch (IOException ioe) {
                        throw new UncheckedIOException(ioe);
                    }
                });
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static Path pathOf(URI uri) {
        return FileUtils.pathOf(uri, FileUtils.class.getClassLoader());
    }

    public static Path pathOf(URI uri, ClassLoader classLoader) {
        FileSystem fileSystem;
        if ("file".equals(uri.getScheme())) {
            return FileSystems.getDefault().provider().getPath(uri);
        }
        try {
            fileSystem = FileSystems.newFileSystem(uri, FS_ENV, classLoader);
        }
        catch (FileSystemAlreadyExistsException ex) {
            fileSystem = FileSystems.getFileSystem(uri);
        }
        catch (IOException ex) {
            throw new UncheckedIOException(ex);
        }
        String spec = uri.getSchemeSpecificPart();
        int index = spec.indexOf("!/");
        if (index == -1) {
            return fileSystem.getPath("/", new String[0]);
        }
        return fileSystem.getPath(spec.substring(index + 1), new String[0]);
    }

    public static Path pathOf(URL url) {
        try {
            return FileUtils.pathOf(url.toURI());
        }
        catch (URISyntaxException e) {
            throw new IllegalStateException(e);
        }
    }

    public static Path pathOf(URL url, ClassLoader classLoader) {
        try {
            return FileUtils.pathOf(url.toURI(), classLoader);
        }
        catch (URISyntaxException e) {
            throw new IllegalStateException(e);
        }
    }

    private FileUtils() {
    }

    public static Path resourceAsPath(String path, Class<?> clazz) throws IllegalArgumentException {
        URL templatesDirURL = clazz.getResource(path);
        if (templatesDirURL == null) {
            throw new IllegalArgumentException("resource not found: " + path);
        }
        try {
            return FileUtils.pathOf(templatesDirURL.toURI());
        }
        catch (URISyntaxException ex) {
            throw new RuntimeException(ex);
        }
    }

    public static String fileExt(Path file) {
        String filename = file.getFileName().toString();
        int index = filename.lastIndexOf(".");
        return index < 0 ? null : filename.substring(index + 1);
    }

    public static Path randomPath(String prefix, String suffix) {
        return FileUtils.randomPath(TMPDIR, prefix, suffix);
    }

    public static Path randomPath(Path dir, String prefix, String suffix) {
        Object filename = prefix == null ? "" : prefix;
        filename = (String)filename + Long.toUnsignedString(RANDOM.nextLong());
        if (suffix != null) {
            filename = (String)filename + suffix;
        }
        return dir.resolve((String)filename);
    }

    public static String measuredSize(long size) {
        if (size < 1024L) {
            return size + " B";
        }
        if (size < 0x100000L) {
            return size / 1024L + " KB";
        }
        if (size < 0x40000000L) {
            return DF.format((double)size / 1048576.0) + " MB";
        }
        if (size < 0x10000000000L) {
            return DF.format((double)size / 1.073741824E9) + " GB";
        }
        return DF.format((double)size / 1.099511627776E12) + " GB";
    }

    public static List<String> readAllLines(URI fileUri) throws IOException, URISyntaxException {
        Path path = FileUtils.pathOf(fileUri);
        return Files.readAllLines(path);
    }

    public static Properties loadProperties(Path filePath) {
        Properties properties;
        block8: {
            InputStream input = Files.newInputStream(filePath, new OpenOption[0]);
            try {
                Properties props = new Properties();
                props.load(input);
                properties = props;
                if (input == null) break block8;
            }
            catch (Throwable throwable) {
                try {
                    if (input != null) {
                        try {
                            input.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new UncheckedIOException(e);
                }
            }
            input.close();
        }
        return properties;
    }

    public static void saveToPropertiesFile(Map<String, String> values, Path filePath) {
        try (OutputStream output = Files.newOutputStream(filePath, new OpenOption[0]);){
            Properties props = new Properties();
            props.putAll(values);
            props.store(output, null);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    public static URL urlOf(Path path) {
        try {
            return path.toUri().toURL();
        }
        catch (MalformedURLException e) {
            throw new RuntimeException(e);
        }
    }
}

