/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.build.common;

import io.helidon.build.common.ConsoleRecorder;
import io.helidon.build.common.PrintStreams;
import io.helidon.build.common.logging.Log;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.LockSupport;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

public final class ProcessMonitor {
    private static final String EOL = System.getProperty("line.separator");
    private static final int GRACEFUL_STOP_TIMEOUT = 3;
    private static final int FORCEFUL_STOP_TIMEOUT = 2;
    private static final MonitorThread MONITOR_THREAD = new MonitorThread();
    private final ProcessBuilder builder;
    private final String description;
    private final Consumer<String> monitorOut;
    private final ProcessBuilder.Redirect stdIn;
    private final ConsoleRecorder recorder;
    private final boolean capturing;
    private final CompletableFuture<Void> exitFuture;
    private final AtomicBoolean shutdown;
    private final Runnable beforeShutdown;
    private final Runnable afterShutdown;
    private volatile Process process;

    public static Builder builder() {
        return new Builder();
    }

    private ProcessMonitor(Builder builder) {
        this.builder = builder.builder;
        this.description = builder.description;
        this.capturing = builder.capture;
        this.monitorOut = builder.monitorOut;
        this.stdIn = builder.stdIn;
        this.recorder = new ConsoleRecorder(builder.stdOut, builder.stdErr, builder.filter, builder.transform, builder.capture);
        this.shutdown = new AtomicBoolean();
        this.beforeShutdown = builder.beforeShutdown;
        this.afterShutdown = builder.afterShutdown;
        this.exitFuture = new CompletableFuture();
    }

    public ProcessMonitor execute(long timeout, TimeUnit unit) throws IOException, ProcessTimeoutException, ProcessFailedException, InterruptedException {
        return this.start().waitForCompletion(timeout, unit);
    }

    public ProcessMonitor start() throws IOException {
        if (this.process != null) {
            throw new IllegalStateException("already started");
        }
        if (this.description != null) {
            this.monitorOut.accept(this.description);
        }
        if (this.stdIn != null) {
            this.builder.redirectInput(this.stdIn);
        }
        Log.debug("Executing command: %s", String.join((CharSequence)" ", this.builder.command()));
        this.process = this.builder.start();
        this.recorder.start(this.process.getInputStream(), this.process.getErrorStream());
        Log.debug("Process ID: %d", this.process.pid());
        MONITOR_THREAD.register(this);
        return this;
    }

    public ProcessMonitor stop() {
        long pid = this.process.toHandle().pid();
        this.process.destroy();
        try {
            try {
                this.exitFuture.get(3L, TimeUnit.SECONDS);
            }
            catch (TimeoutException e) {
                this.process.destroyForcibly();
                try {
                    this.exitFuture.get(2L, TimeUnit.SECONDS);
                }
                catch (TimeoutException ex) {
                    throw new IllegalStateException(String.format("Failed to stop process %d: %s", pid, "timeout expired"));
                }
            }
        }
        catch (InterruptedException e) {
            throw new IllegalStateException(String.format("Failed to stop process %d: %s", pid, e.getMessage()));
        }
        catch (ExecutionException e) {
            throw new IllegalStateException(String.format("Failed to stop process %d: %s", pid, e.getCause().getMessage()));
        }
        return this;
    }

    public ProcessMonitor waitForCompletion(long timeout, TimeUnit unit) throws ProcessTimeoutException, ProcessFailedException, InterruptedException {
        if (this.process == null) {
            throw new IllegalStateException("not started");
        }
        if (this.process.isAlive()) {
            Log.debug("Waiting for completion, pid=%d, timeout=%d, unit=%s", new Object[]{this.process.pid(), timeout, unit});
            try {
                this.exitFuture.get(timeout, unit);
                try {
                    if (this.process.exitValue() != 0 && !this.shutdown.get()) {
                        throw new ProcessFailedException();
                    }
                }
                catch (IllegalThreadStateException ex) {
                    throw new ProcessFailedException();
                }
            }
            catch (ExecutionException ex) {
                throw new IllegalStateException(ex);
            }
            catch (TimeoutException e) {
                throw new ProcessTimeoutException();
            }
        }
        return this;
    }

    public boolean isAlive() {
        Process process = this.process;
        if (process == null) {
            return false;
        }
        return process.isAlive();
    }

    public String output() {
        return this.recorder.capturedOutput();
    }

    public String stdOut() {
        return this.recorder.capturedStdOut();
    }

    public String stdErr() {
        return this.recorder.capturedStdErr();
    }

    public static final class Builder {
        private ProcessBuilder builder;
        private String description;
        private boolean capture;
        private Consumer<String> monitorOut;
        private ProcessBuilder.Redirect stdIn;
        private PrintStream stdOut;
        private PrintStream stdErr;
        private Predicate<String> filter = line -> true;
        private Function<String, String> transform = Function.identity();
        private Runnable beforeShutdown = () -> {};
        private Runnable afterShutdown = () -> {};

        private Builder() {
        }

        public Builder processBuilder(ProcessBuilder processBuilder) {
            this.builder = processBuilder;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder capture(boolean capture) {
            this.capture = capture;
            return this;
        }

        public Builder stdIn(File stdIn) {
            if (stdIn != null) {
                return this.stdIn(ProcessBuilder.Redirect.from(stdIn));
            }
            return this;
        }

        public Builder stdIn(ProcessBuilder.Redirect stdIn) {
            this.stdIn = stdIn;
            return this;
        }

        public Builder stdOut(PrintStream stdOut) {
            this.stdOut = stdOut;
            this.monitorOut = stdOut::println;
            return this;
        }

        public Builder stdErr(PrintStream stdErr) {
            this.stdErr = stdErr;
            return this;
        }

        public Builder filter(Predicate<String> filter) {
            this.filter = filter;
            return this;
        }

        public Builder transform(Function<String, String> transform) {
            this.transform = transform;
            return this;
        }

        public Builder beforeShutdown(Runnable beforeShutdown) {
            this.beforeShutdown = beforeShutdown;
            return this;
        }

        public Builder afterShutdown(Runnable afterShutdown) {
            this.afterShutdown = afterShutdown;
            return this;
        }

        public ProcessMonitor build() {
            if (this.builder == null) {
                throw new IllegalStateException("processBuilder required");
            }
            if (this.stdOut == null) {
                this.capture = true;
                this.stdOut = PrintStreams.DEVNULL;
                this.monitorOut = x$0 -> Log.info(x$0, new Object[0]);
            } else {
                this.monitorOut = this.stdOut::println;
            }
            if (this.stdErr == null) {
                this.capture = true;
                this.stdErr = PrintStreams.DEVNULL;
            }
            if (this.filter == null) {
                this.filter = line -> true;
            }
            if (this.transform == null) {
                this.transform = Function.identity();
            }
            return new ProcessMonitor(this);
        }
    }

    private static final class MonitorThread
    extends Thread {
        private final List<ProcessMonitor> processes = new ArrayList<ProcessMonitor>();
        private int backoff = 0;
        private Iterator<ProcessMonitor> iterator;

        private MonitorThread() {
            this.start();
            Runtime.getRuntime().addShutdownHook(new Thread(this::shutdown));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void register(ProcessMonitor process) {
            List<ProcessMonitor> list = this.processes;
            synchronized (list) {
                this.processes.add(process);
                if (this.processes.size() == 1) {
                    LockSupport.unpark(this);
                }
            }
        }

        @Override
        public void run() {
            while (true) {
                if (this.processes.isEmpty()) {
                    LockSupport.park();
                }
                this.iterator = this.processes.iterator();
                boolean ticked = true;
                while (this.iterator.hasNext()) {
                    if (this.tick(this.iterator.next())) continue;
                    ticked = false;
                }
                if (this.processes.isEmpty()) continue;
                this.backoff = ticked ? 0 : (this.backoff < 5 ? this.backoff + 1 : this.backoff);
                try {
                    Thread.sleep(50L / (long)this.processes.size() * (long)this.backoff);
                }
                catch (InterruptedException interruptedException) {
                }
            }
        }

        private void shutdown() {
            CompletableFuture<Void> exitFuture = CompletableFuture.allOf((CompletableFuture[])new ArrayList<ProcessMonitor>(this.processes).stream().map(p -> {
                p.recorder.stop();
                p.beforeShutdown.run();
                p.shutdown.set(true);
                p.process.destroy();
                return p.exitFuture.thenRun(p.afterShutdown);
            }).toArray(CompletableFuture[]::new));
            try {
                exitFuture.get();
            }
            catch (InterruptedException | ExecutionException exception) {
                // empty catch block
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private boolean tick(ProcessMonitor process) {
            try {
                boolean bl = process.recorder.tick();
                return bl;
            }
            catch (Throwable ex) {
                boolean bl = false;
                return bl;
            }
            finally {
                if (!process.isAlive()) {
                    process.recorder.drain();
                    process.exitFuture.complete(null);
                    this.iterator.remove();
                }
            }
        }
    }

    public final class ProcessFailedException
    extends ProcessException {
        private ProcessFailedException() {
            super(" failed with exit code " + ProcessMonitor.this.process.exitValue());
        }
    }

    public final class ProcessTimeoutException
    extends ProcessException {
        private ProcessTimeoutException() {
            super("timed out");
        }
    }

    public abstract class ProcessException
    extends Exception {
        private final String reason;

        private ProcessException(String reason) {
            this.reason = reason;
        }

        @Override
        public String getMessage() {
            StringBuilder message = new StringBuilder().append(Objects.requireNonNullElseGet(ProcessMonitor.this.description, () -> String.join((CharSequence)" ", ProcessMonitor.this.builder.command()))).append(" ").append(this.reason);
            if (ProcessMonitor.this.capturing) {
                message.append(EOL);
                for (String line : ProcessMonitor.this.output().split("\\R")) {
                    message.append("    ").append(line).append(EOL);
                }
            }
            return message.toString();
        }

        public ProcessMonitor monitor() {
            return ProcessMonitor.this;
        }
    }
}

