/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.build.maven.enforcer;

import io.helidon.build.common.logging.Log;
import io.helidon.build.maven.enforcer.EnforcerException;
import io.helidon.build.maven.enforcer.FileMatcher;
import io.helidon.build.maven.enforcer.FileRequest;
import io.helidon.build.maven.enforcer.FileSystem;
import io.helidon.build.maven.enforcer.FoundFiles;
import io.helidon.build.maven.enforcer.GitCommands;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

public class FileFinder {
    private static final DateTimeFormatter YEAR_FORMATTER = DateTimeFormatter.ofPattern("yyyy");
    private final String currentYear = YEAR_FORMATTER.format(ZonedDateTime.now());
    private final Path repositoryRoot;
    private final boolean useGit;
    private final boolean honorGitIgnore;

    private FileFinder(Builder builder) {
        this.useGit = builder.useGit;
        this.honorGitIgnore = builder.honorGitIgnore;
        this.repositoryRoot = builder.repositoryRoot;
    }

    public static Builder builder() {
        return new Builder();
    }

    public FoundFiles findFiles(Path basePath) {
        Set<FileRequest> foundFiles;
        Set<FileRequest> locallyModified;
        Path gitRepoDir = this.repositoryRoot == null ? GitCommands.repositoryRoot(basePath) : this.repositoryRoot;
        ArrayList<FileMatcher> excludes = new ArrayList<FileMatcher>();
        if (this.useGit && this.honorGitIgnore) {
            this.addGitIgnore(gitRepoDir, excludes);
        }
        if (this.useGit) {
            locallyModified = GitCommands.locallyModified(gitRepoDir, basePath, this.currentYear);
            foundFiles = new HashSet<FileRequest>(locallyModified);
            foundFiles.addAll(GitCommands.gitTracked(gitRepoDir, basePath));
        } else {
            foundFiles = this.findAllFiles(gitRepoDir, basePath);
            locallyModified = foundFiles;
        }
        List<FileRequest> fileRequests = foundFiles.stream().filter(file -> this.isValid((FileRequest)file, (List<FileMatcher>)excludes)).collect(Collectors.toList());
        Set<String> filteredLocallyModified = this.exclude(excludes, locallyModified);
        fileRequests.sort(FileRequest::compareTo);
        return FoundFiles.create(gitRepoDir, fileRequests, filteredLocallyModified, this.useGit);
    }

    public String toString() {
        return "FileConfig{repositoryRoot=" + this.repositoryRoot + ", useGit=" + this.useGit + ", honorGitIgnore=" + this.honorGitIgnore + "}";
    }

    private void addGitIgnore(Path gitRepoDir, List<FileMatcher> excludes) {
        Path gitIgnore = gitRepoDir.resolve(".gitignore");
        excludes.addAll(FileMatcher.create(".git/"));
        List lines = FileSystem.toLines(gitIgnore).stream().filter(it -> !it.startsWith("#")).filter(it -> !it.isBlank()).collect(Collectors.toList());
        for (String line : lines) {
            if (line.contains("*")) {
                if (line.startsWith("*.")) {
                    excludes.addAll(FileMatcher.create(line.substring(1)));
                    continue;
                }
                if (line.startsWith("*")) {
                    excludes.add(new NameEndExclude(line.substring(1)));
                    continue;
                }
                if (line.endsWith("*")) {
                    excludes.add(new NameStartExclude(line.substring(line.length() - 1)));
                    continue;
                }
                Log.warn((String)("$(YELLOW .gitignore) matches not supported: " + line), (Object[])new Object[0]);
                continue;
            }
            excludes.addAll(FileMatcher.create(line));
        }
    }

    private Set<FileRequest> findAllFiles(final Path gitRepoDir, Path basePath) {
        final HashSet<FileRequest> result = new HashSet<FileRequest>();
        try {
            Files.walkFileTree(basePath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
                    result.add(FileRequest.create(gitRepoDir, gitRepoDir.relativize(file).toString(), FileFinder.this.lastModifiedYear(file)));
                    return FileVisitResult.CONTINUE;
                }
            });
        }
        catch (IOException e) {
            throw new EnforcerException("Failed to list files", e);
        }
        return result;
    }

    private String lastModifiedYear(Path file) {
        try {
            return YEAR_FORMATTER.format(ZonedDateTime.ofInstant(Files.getLastModifiedTime(file, new LinkOption[0]).toInstant(), ZoneId.of("GMT")));
        }
        catch (IOException e) {
            throw new EnforcerException("Failed to parse last modified year from local file: " + file, e);
        }
    }

    private Set<String> exclude(List<FileMatcher> excludes, Set<FileRequest> locallyModified) {
        return locallyModified.stream().filter(it -> {
            for (FileMatcher exclude : excludes) {
                if (!exclude.matches((FileRequest)it)) continue;
                return false;
            }
            return true;
        }).map(FileRequest::relativePath).collect(Collectors.toSet());
    }

    private boolean isValid(FileRequest file, List<FileMatcher> excludes) {
        if (!Files.exists(file.path(), new LinkOption[0])) {
            Log.debug((String)("File " + file.relativePath() + " does not exist, ignoring."), (Object[])new Object[0]);
            return false;
        }
        for (FileMatcher exclude : excludes) {
            if (!exclude.matches(file)) continue;
            Log.debug((String)("Excluding " + file.relativePath()), (Object[])new Object[0]);
            return false;
        }
        return true;
    }

    public static final class Builder {
        private Path repositoryRoot;
        private boolean useGit = true;
        private boolean honorGitIgnore = true;

        private Builder() {
        }

        public Builder useGit(boolean useGit) {
            this.useGit = useGit;
            return this;
        }

        public Builder honorGitIgnore(boolean honorGitIgnore) {
            this.honorGitIgnore = honorGitIgnore;
            return this;
        }

        public Builder repositoryRoot(Path repositoryRoot) {
            this.repositoryRoot = repositoryRoot;
            return this;
        }

        public FileFinder build() {
            return new FileFinder(this);
        }
    }

    private static final class NameEndExclude
    implements FileMatcher {
        private final String end;

        private NameEndExclude(String end) {
            this.end = end;
        }

        @Override
        public boolean matches(FileRequest file) {
            return file.fileName().endsWith(this.end);
        }
    }

    private static final class NameStartExclude
    implements FileMatcher {
        private final String start;

        private NameStartExclude(String start) {
            this.start = start;
        }

        @Override
        public boolean matches(FileRequest file) {
            return file.fileName().startsWith(this.start);
        }
    }
}

