/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.linker;

import io.helidon.build.util.FileUtils;
import io.helidon.build.util.Log;
import io.helidon.build.util.LogFormatter;
import io.helidon.build.util.PrintStreams;
import io.helidon.build.util.ProcessMonitor;
import io.helidon.build.util.StyleFunction;
import io.helidon.linker.Application;
import io.helidon.linker.ClassDataSharing;
import io.helidon.linker.Configuration;
import io.helidon.linker.StartScript;
import io.helidon.linker.util.Constants;
import io.helidon.linker.util.JavaRuntime;
import java.io.File;
import java.io.PrintStream;
import java.io.UncheckedIOException;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.spi.ToolProvider;

public final class Linker {
    private static final String JLINK_TOOL_NAME = "jlink";
    private static final String JLINK_DEBUG_PROPERTY = "jlink.debug";
    private static final float BYTES_PER_MEGABYTE = 1048576.0f;
    private final ToolProvider jlink = ToolProvider.findFirst("jlink").orElseThrow(() -> new IllegalStateException("jlink not found"));
    private final List<String> jlinkArgs = new ArrayList<String>();
    private final Configuration config;
    private final String imageName;
    private long startTime;
    private Application application;
    private String exitOnStarted;
    private Set<String> javaDependencies;
    private JavaRuntime jri;
    private Path jriMainJar;
    private long cdsArchiveSize;
    private StartScript startScript;
    private List<String> startCommand;
    private float appSize;
    private float jdkSize;
    private float jriSize;
    private float cdsSize;
    private float jriAppSize;
    private String initialSize;
    private String imageSize;
    private String percent;

    public static void main(String ... args) throws Exception {
        Linker.linker(Configuration.builder().commandLine(args).build()).link();
    }

    public static Linker linker(Configuration config) {
        return new Linker(config);
    }

    private Linker(Configuration config) {
        this.config = config;
        this.imageName = FileUtils.fileName((Path)config.jriDirectory());
        if (config.verbose()) {
            System.setProperty(JLINK_DEBUG_PROPERTY, "true");
        }
    }

    public Path link() {
        this.begin();
        this.buildApplication();
        this.collectJavaDependencies();
        this.buildJlinkArguments();
        this.buildJri();
        this.installJars();
        this.installCdsArchive();
        this.installStartScript();
        this.testImage();
        this.displayStartScriptHelp();
        this.computeSizes();
        this.end();
        return this.config.jriDirectory();
    }

    public Configuration config() {
        return this.config;
    }

    private void begin() {
        Log.info();
        this.startTime = System.currentTimeMillis();
    }

    private void buildApplication() {
        this.application = Application.create(this.config.mainJar());
        this.exitOnStarted = this.application.exitOnStartedValue();
        String version = this.application.helidonVersion();
        Log.info((String)"Creating Java Runtime Image %s from %s and %s, built with Helidon %s", (Object[])new Object[]{StyleFunction.Cyan.apply((Object)this.imageName), StyleFunction.Cyan.apply((Object)("JDK " + this.config.jdk().version())), StyleFunction.Cyan.apply((Object)this.config.mainJar().getFileName()), StyleFunction.Cyan.apply((Object)version)});
    }

    private void collectJavaDependencies() {
        Log.info((String)"Collecting Java module dependencies", (Object[])new Object[0]);
        this.javaDependencies = this.application.javaDependencies(this.config.jdk());
        this.javaDependencies.addAll(this.config.additionalModules());
        ArrayList<String> sorted = new ArrayList<String>(this.javaDependencies);
        sorted.sort(null);
        Log.info((String)"Including %d Java dependencies: %s", (Object[])new Object[]{sorted.size(), String.join((CharSequence)", ", sorted)});
        if (this.config.stripDebug()) {
            Log.info((String)"Excluding debug support: %s", (Object[])new Object[]{"jdk.jdwp.agent"});
        } else {
            this.javaDependencies.add("jdk.jdwp.agent");
            Log.info((String)"Including debug support: %s", (Object[])new Object[]{"jdk.jdwp.agent"});
        }
    }

    private void buildJlinkArguments() {
        this.jlinkArgs.add("--module-path");
        this.jlinkArgs.add(this.config.jdk().jmodsDir().normalize().toString());
        this.jlinkArgs.add("--add-modules");
        this.jlinkArgs.add(String.join((CharSequence)",", this.javaDependencies));
        this.jlinkArgs.add("--output");
        this.jlinkArgs.add(this.config.jriDirectory().normalize().toString());
        if (this.config.stripDebug()) {
            this.jlinkArgs.add("--strip-debug");
        }
        this.jlinkArgs.add("--no-header-files");
        this.jlinkArgs.add("--no-man-pages");
        this.jlinkArgs.add("--compress");
        this.jlinkArgs.add("2");
        this.jlinkArgs.addAll(this.config.additionalJlinkArgs());
    }

    private void buildJri() {
        Log.info((String)"Creating base image: %s", (Object[])new Object[]{this.jriDirectory()});
        int result = this.jlink.run(System.out, System.err, this.jlinkArgs.toArray(new String[0]));
        if (result != 0) {
            throw new Error("JRI creation failed.");
        }
        this.jri = JavaRuntime.jri(this.config.jriDirectory(), this.config.jdk().version());
    }

    private void installJars() {
        boolean stripDebug = this.config.stripDebug();
        Path appDir = this.jriDirectory().resolve(Application.APP_DIR);
        String message = stripDebug ? ", stripping debug information from all classes" : "";
        Log.info((String)"Installing %d application jars in %s%s", (Object[])new Object[]{this.application.size(), appDir, message});
        this.jriMainJar = this.application.install(this.jri, stripDebug);
    }

    private void installCdsArchive() {
        if (this.config.cds()) {
            try {
                ClassDataSharing cds = ClassDataSharing.builder().jri(this.jri.path()).applicationJar(this.jriMainJar).jvmOptions(this.config.defaultJvmOptions()).args(this.config.defaultArgs()).archiveFile(this.application.archivePath()).exitOnStartedValue(this.exitOnStarted).maxWaitSeconds(this.config.maxAppStartSeconds()).logOutput(this.config.verbose()).build();
                this.cdsArchiveSize = FileUtils.sizeOf((Path)this.jri.path().resolve(this.application.archivePath()));
                JavaRuntime jdk = this.config.jdk();
                Application app = this.application;
                int jdkCount = 0;
                int appCount = 0;
                for (String name : cds.classList()) {
                    String resourcePath = name + ".class";
                    if (jdk.containsResource(resourcePath)) {
                        ++jdkCount;
                        continue;
                    }
                    if (!app.containsResource(resourcePath)) continue;
                    ++appCount;
                }
                String cdsSize = StyleFunction.BoldBlue.format("%.1fM", new Object[]{Float.valueOf(Linker.mb(this.cdsArchiveSize))});
                String jdkSize = StyleFunction.BoldBlue.format("%d", new Object[]{jdkCount});
                String appSize = StyleFunction.BoldBlue.format("%d", new Object[]{appCount});
                if (appCount == 0) {
                    if (!Constants.CDS_REQUIRES_UNLOCK_OPTION) {
                        Log.warn((String)"CDS archive does not contain any application classes, but should!", (Object[])new Object[0]);
                    }
                    Log.info((String)"CDS archive is %s for %s JDK classes", (Object[])new Object[]{cdsSize, jdkSize});
                } else {
                    String total = StyleFunction.BoldBlue.format("%d", new Object[]{jdkCount + appCount});
                    Log.info((String)"CDS archive is %s for %s classes: %s JDK and %s application", (Object[])new Object[]{cdsSize, total, jdkSize, appSize});
                }
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }
    }

    private void installStartScript() {
        try {
            this.startScript = StartScript.builder().installHomeDirectory(this.config.jriDirectory()).defaultJvmOptions(this.config.defaultJvmOptions()).defaultDebugOptions(this.config.defaultDebugOptions()).mainJar(this.jriMainJar).defaultArgs(this.config.defaultArgs()).cdsInstalled(this.config.cds()).debugInstalled(!this.config.stripDebug()).exitOnStartedValue(this.exitOnStarted).build();
            Log.info((String)"Installing start script in %s", (Object[])new Object[]{this.startScript.installDirectory()});
            this.startScript.install();
            this.startCommand = List.of(this.imageName + Constants.DIR_SEP + "bin" + Constants.DIR_SEP + this.startScript.scriptFile().getFileName());
        }
        catch (StartScript.PlatformNotSupportedError e) {
            if (this.config.cds()) {
                Log.warn((String)"Start script cannot be created for this platform; for CDS to function, the jar path %s be relative as shown below.", (Object[])new Object[]{StyleFunction.BoldYellow.apply((Object)"must")});
            } else {
                Log.warn((String)"Start script cannot be created for this platform.", (Object[])new Object[0]);
            }
            this.startCommand = e.command();
        }
    }

    private String startCommand() {
        return String.join((CharSequence)" ", this.startCommand);
    }

    private void testImage() {
        if (this.config.test()) {
            if (this.startScript != null) {
                this.executeStartScript("--test");
            } else {
                Log.info();
                Log.info((String)"Executing %s", (Object[])new Object[]{StyleFunction.Cyan.apply((Object)this.startCommand())});
                Log.info();
                ArrayList<String> command = new ArrayList<String>(this.startCommand);
                command.add(command.indexOf("-jar"), "-Dexit.on.started=!");
                File root = this.config.jriDirectory().toFile();
                try {
                    ProcessMonitor.builder().processBuilder(new ProcessBuilder(new String[0]).command(command).directory(root)).stdOut(PrintStreams.apply((PrintStream)PrintStreams.STDOUT, (Function)LogFormatter.of((Log.Level)Log.Level.INFO))).stdErr(PrintStreams.apply((PrintStream)PrintStreams.STDERR, (Function)LogFormatter.of((Log.Level)Log.Level.WARN))).transform(Constants.INDENT).capture(false).build().execute((long)this.config.maxAppStartSeconds(), TimeUnit.SECONDS);
                }
                catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    private void displayStartScriptHelp() {
        this.executeStartScript("--help");
    }

    private void executeStartScript(String option) {
        if (this.startScript != null) {
            Log.info();
            Log.info((String)"Executing %s", (Object[])new Object[]{StyleFunction.Cyan.apply((Object)(this.startCommand() + " " + option))});
            Log.info();
            this.startScript.execute(Constants.INDENT, option);
        }
    }

    private void computeSizes() {
        try {
            long app = this.application.diskSize();
            long jdk = this.config.jdk().diskSize();
            long jri = this.jri.diskSize();
            long cds = this.cdsArchiveSize;
            long jriApp = this.config.stripDebug() ? this.application.installedSize(this.jri) : app;
            long jriOnly = jri - cds - jriApp;
            long initial = app + jdk;
            float reduction = (1.0f - (float)jri / (float)initial) * 100.0f;
            this.appSize = Linker.mb(app);
            this.jdkSize = Linker.mb(jdk);
            this.jriSize = Linker.mb(jriOnly);
            this.cdsSize = this.config.cds() ? Linker.mb(cds) : 0.0f;
            this.jriAppSize = Linker.mb(jriApp);
            this.initialSize = StyleFunction.BoldBlue.format("%5.1fM", new Object[]{Float.valueOf(Linker.mb(initial))});
            this.imageSize = StyleFunction.BoldBrightGreen.format("%5.1fM", new Object[]{Float.valueOf(Linker.mb(jri))});
            this.percent = StyleFunction.BoldBrightGreen.format("%5.1f%%", new Object[]{Float.valueOf(reduction)});
        }
        catch (UncheckedIOException e) {
            Log.debug((String)"Could not compute disk size: %s", (Object[])new Object[]{e.getMessage()});
        }
    }

    private void end() {
        long elapsed = System.currentTimeMillis() - this.startTime;
        float startSeconds = (float)elapsed / 1000.0f;
        Log.info();
        Log.info((String)"Java Runtime Image %s completed in %.1f seconds", (Object[])new Object[]{StyleFunction.Cyan.apply((Object)this.imageName), Float.valueOf(startSeconds)});
        Log.info();
        Log.info((String)"     initial size: %s  (%.1f JDK + %.1f application)", (Object[])new Object[]{this.initialSize, Float.valueOf(this.jdkSize), Float.valueOf(this.appSize)});
        if (this.config.cds()) {
            Log.info((String)"       image size: %s  (%5.1f JDK + %.1f application + %.1f CDS)", (Object[])new Object[]{this.imageSize, Float.valueOf(this.jriSize), Float.valueOf(this.jriAppSize), Float.valueOf(this.cdsSize)});
        } else {
            Log.info((String)"       image size: %s  (%5.1f JDK + %.1f application)", (Object[])new Object[]{this.imageSize, Float.valueOf(this.jriSize), Float.valueOf(this.jriAppSize)});
        }
        Log.info((String)"        reduction: %s", (Object[])new Object[]{this.percent});
        Log.info();
    }

    private static float mb(long bytes) {
        return (float)bytes / 1048576.0f;
    }

    private Path jriDirectory() {
        return FileUtils.fromWorking((Path)this.config.jriDirectory());
    }
}

