/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.concurrency.limits;

import io.helidon.common.concurrency.limits.AimdLimitConfig;
import io.helidon.common.concurrency.limits.IgnoreTaskException;
import io.helidon.common.concurrency.limits.LimitAlgorithm;
import io.helidon.common.concurrency.limits.LimitException;
import io.helidon.common.config.ConfigException;
import java.util.Optional;
import java.util.concurrent.Callable;
import java.util.concurrent.Semaphore;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

class AimdLimitImpl {
    private final double backoffRatio;
    private final long timeoutInNanos;
    private final int minLimit;
    private final int maxLimit;
    private final Supplier<Long> clock;
    private final AtomicInteger concurrentRequests;
    private final AdjustableSemaphore semaphore;
    private final AtomicInteger limit;
    private final Lock limitLock = new ReentrantLock();

    AimdLimitImpl(AimdLimitConfig config) {
        int initialLimit = config.initialLimit();
        this.backoffRatio = config.backoffRatio();
        this.timeoutInNanos = config.timeout().toNanos();
        this.minLimit = config.minLimit();
        this.maxLimit = config.maxLimit();
        this.clock = config.clock().orElseGet(() -> System::nanoTime);
        this.concurrentRequests = new AtomicInteger();
        this.semaphore = new AdjustableSemaphore(initialLimit);
        this.limit = new AtomicInteger(initialLimit);
        if (!(this.backoffRatio < 1.0) || !(this.backoffRatio >= 0.5)) {
            throw new ConfigException("Backoff ratio must be within [0.5, 1.0)");
        }
        if (this.maxLimit < this.minLimit) {
            throw new ConfigException("Max limit must be higher than min limit, or equal to it");
        }
        if (initialLimit > this.maxLimit) {
            throw new ConfigException("Initial limit must be lower than max limit, or equal to it");
        }
        if (initialLimit < this.minLimit) {
            throw new ConfigException("Initial limit must be higher than minimum limit, or equal to it");
        }
    }

    Semaphore semaphore() {
        return this.semaphore;
    }

    int currentLimit() {
        return this.limit.get();
    }

    Optional<LimitAlgorithm.Token> tryAcquire() {
        if (!this.semaphore.tryAcquire()) {
            return Optional.empty();
        }
        return Optional.of(new AimdToken(this.clock, this.concurrentRequests));
    }

    void invoke(Runnable runnable) throws Exception {
        this.invoke(() -> {
            runnable.run();
            return null;
        });
    }

    <T> T invoke(Callable<T> callable) throws Exception {
        long startTime = this.clock.get();
        int currentRequests = this.concurrentRequests.incrementAndGet();
        if (this.semaphore.tryAcquire()) {
            try {
                T response = callable.call();
                this.updateWithSample(startTime, this.clock.get(), currentRequests, true);
                T t = response;
                return t;
            }
            catch (IgnoreTaskException e) {
                Object t = e.handle();
                return t;
            }
            catch (Throwable e) {
                this.updateWithSample(startTime, this.clock.get(), currentRequests, false);
                throw e;
            }
            finally {
                this.concurrentRequests.decrementAndGet();
                this.semaphore.release();
            }
        }
        throw new LimitException("No more permits available for the semaphore");
    }

    void updateWithSample(long startTime, long endTime, int currentRequests, boolean success) {
        long rtt = endTime - startTime;
        int currentLimit = this.limit.get();
        if (rtt > this.timeoutInNanos || !success) {
            currentLimit = (int)((double)currentLimit * this.backoffRatio);
        } else if (currentRequests * 2 >= currentLimit) {
            ++currentLimit;
        }
        this.setLimit(Math.min(this.maxLimit, Math.max(this.minLimit, currentLimit)));
    }

    private void setLimit(int newLimit) {
        if (newLimit == this.limit.get()) {
            return;
        }
        this.limitLock.lock();
        try {
            int oldLimit = this.limit.get();
            if (oldLimit == newLimit) {
                return;
            }
            this.limit.set(newLimit);
            if (newLimit > oldLimit) {
                this.semaphore.release(newLimit - oldLimit);
            } else {
                this.semaphore.reducePermits(oldLimit - newLimit);
            }
        }
        finally {
            this.limitLock.unlock();
        }
    }

    private static final class AdjustableSemaphore
    extends Semaphore {
        private static final long serialVersionUID = 114L;

        private AdjustableSemaphore(int permits) {
            super(permits);
        }

        @Override
        protected void reducePermits(int reduction) {
            super.reducePermits(reduction);
        }
    }

    private class AimdToken
    implements LimitAlgorithm.Token {
        private final long startTime;
        private final int currentRequests;

        private AimdToken(Supplier<Long> clock, AtomicInteger concurrentRequests) {
            this.startTime = clock.get();
            this.currentRequests = concurrentRequests.incrementAndGet();
        }

        @Override
        public void dropped() {
            try {
                AimdLimitImpl.this.updateWithSample(this.startTime, AimdLimitImpl.this.clock.get(), this.currentRequests, false);
            }
            finally {
                AimdLimitImpl.this.semaphore.release();
            }
        }

        @Override
        public void ignore() {
            AimdLimitImpl.this.concurrentRequests.decrementAndGet();
            AimdLimitImpl.this.semaphore.release();
        }

        @Override
        public void success() {
            try {
                AimdLimitImpl.this.updateWithSample(this.startTime, AimdLimitImpl.this.clock.get(), this.currentRequests, true);
                AimdLimitImpl.this.concurrentRequests.decrementAndGet();
            }
            finally {
                AimdLimitImpl.this.semaphore.release();
            }
        }
    }
}

