/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.configurable;

import io.helidon.common.context.ContextAwareExecutorService;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.management.NotificationEmitter;

public class ThreadPool
extends ThreadPoolExecutor {
    private static final Logger LOGGER = Logger.getLogger(ThreadPool.class.getName());
    private static final int MAX_GROWTH_RATE = 100;
    private final String name;
    private final WorkQueue queue;
    private final RejectionHandler rejectionHandler;
    private final AtomicInteger activeThreads;
    private final LongAdder totalActiveThreads;
    private final AtomicInteger completedTasks;
    private final AtomicInteger failedTasks;
    private final int growthThreshold;
    private final int growthRate;

    public static Optional<ThreadPool> asThreadPool(ExecutorService executor) {
        if (executor instanceof ThreadPool) {
            return Optional.of((ThreadPool)executor);
        }
        if (executor instanceof ContextAwareExecutorService) {
            return ThreadPool.asThreadPool(((ContextAwareExecutorService)executor).unwrap());
        }
        return Optional.empty();
    }

    static ThreadPool create(String name, int corePoolSize, int maxPoolSize, int growthThreshold, int growthRate, long keepAliveTime, TimeUnit keepAliveTimeUnits, int workQueueCapacity, String threadNamePrefix, boolean useDaemonThreads, RejectionHandler rejectionHandler) {
        if (name == null || name.isEmpty()) {
            throw new IllegalArgumentException("name is null or empty");
        }
        if (corePoolSize < 0) {
            throw new IllegalArgumentException("corePoolSize < 0");
        }
        if (maxPoolSize < 0) {
            throw new IllegalArgumentException("maxPoolSize < 0");
        }
        if (maxPoolSize < corePoolSize) {
            throw new IllegalArgumentException("maxPoolSize < corePoolSize");
        }
        if (growthThreshold < 0) {
            throw new IllegalArgumentException("growthThreshold < 0");
        }
        if (growthRate < 0) {
            throw new IllegalArgumentException("growthRate < 0");
        }
        if (growthRate > 100) {
            throw new IllegalArgumentException("growthRate > 100");
        }
        if (keepAliveTime < 1L) {
            throw new IllegalArgumentException("keepAliveTime < 1");
        }
        if (workQueueCapacity < 1) {
            throw new IllegalArgumentException("workQueueCapacity < 1");
        }
        if (threadNamePrefix == null || threadNamePrefix.isEmpty()) {
            throw new IllegalArgumentException("threadNamePrefix is null or empty");
        }
        if (rejectionHandler == null) {
            throw new IllegalArgumentException("rejectionPolicy is null");
        }
        WorkQueue queue = ThreadPool.createQueue(workQueueCapacity, corePoolSize, maxPoolSize, growthThreshold, growthRate);
        GroupedThreadFactory threadFactory = new GroupedThreadFactory(name, threadNamePrefix, useDaemonThreads);
        return new ThreadPool(name, corePoolSize, maxPoolSize, growthThreshold, growthRate, keepAliveTime, keepAliveTimeUnits, threadFactory, queue, rejectionHandler);
    }

    private ThreadPool(String name, int corePoolSize, int maximumPoolSize, int growthThreshold, int growthRate, long keepAliveTime, TimeUnit keepAliveTimeUnit, ThreadFactory threadFactory, WorkQueue queue, RejectionHandler rejectionHandler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, keepAliveTimeUnit, queue, threadFactory, rejectionHandler);
        this.name = name;
        this.queue = queue;
        this.growthThreshold = growthThreshold;
        this.activeThreads = new AtomicInteger();
        this.totalActiveThreads = new LongAdder();
        this.completedTasks = new AtomicInteger();
        this.failedTasks = new AtomicInteger();
        this.growthRate = growthRate;
        this.rejectionHandler = rejectionHandler;
        queue.setPool(this);
        if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.fine(this.toString());
        }
    }

    public String getName() {
        return this.name;
    }

    public int getQueueCapacity() {
        return this.queue.getCapacity();
    }

    int getGrowthThreshold() {
        return this.growthThreshold;
    }

    int getGrowthRate() {
        return this.growthRate;
    }

    public float getAverageQueueSize() {
        return this.queue.getAverageSize();
    }

    public int getPeakQueueSize() {
        return this.queue.getPeakSize();
    }

    public int getCompletedTasks() {
        return this.completedTasks.get();
    }

    public int getFailedTasks() {
        return this.failedTasks.get();
    }

    public int getTotalTasks() {
        return this.completedTasks.get() + this.failedTasks.get();
    }

    public int getActiveThreads() {
        return this.activeThreads.get();
    }

    public float getAverageActiveThreads() {
        float totalActive = this.totalActiveThreads.sum();
        if (totalActive == 0.0f) {
            return 0.0f;
        }
        return totalActive / (float)this.getTotalTasks();
    }

    public int getRejectionCount() {
        return this.rejectionHandler.getRejectionCount();
    }

    public boolean isFixedSize() {
        return this.getMaximumPoolSize() == this.getCorePoolSize();
    }

    public WorkQueue getQueue() {
        return this.queue;
    }

    public int getQueueSize() {
        return this.queue.size();
    }

    @Override
    public void setRejectedExecutionHandler(RejectedExecutionHandler handler) {
        if (!(handler instanceof RejectionHandler)) {
            throw new IllegalArgumentException(handler.getClass() + " must be an instance of " + RejectionHandler.class);
        }
        super.setRejectedExecutionHandler(handler);
    }

    @Override
    public RejectionHandler getRejectedExecutionHandler() {
        return (RejectionHandler)super.getRejectedExecutionHandler();
    }

    @Override
    public void setMaximumPoolSize(int maximumPoolSize) {
        if (maximumPoolSize != this.getMaximumPoolSize()) {
            LOGGER.warning("Maximum pool size cannot be changed in " + this);
        }
    }

    @Override
    public String toString() {
        boolean fixedSize = this.isFixedSize();
        return "ThreadPool '" + this.getName() + "' {corePoolSize=" + this.getCorePoolSize() + ", maxPoolSize=" + this.getMaximumPoolSize() + ", queueCapacity=" + this.getQueueCapacity() + (String)(fixedSize ? "" : ", growthThreshold=" + this.getGrowthThreshold()) + (String)(fixedSize ? "" : ", growthRate=" + this.getGrowthRate() + "%") + String.format(", averageQueueSize=%.2f", Float.valueOf(this.getAverageQueueSize())) + ", peakQueueSize=" + this.getPeakQueueSize() + String.format(", averageActiveThreads=%.2f", Float.valueOf(this.getAverageActiveThreads())) + (String)(fixedSize ? "" : ", peakPoolSize=" + this.getLargestPoolSize()) + ", currentPoolSize=" + this.getPoolSize() + ", completedTasks=" + this.getCompletedTasks() + ", failedTasks=" + this.getFailedTasks() + ", rejectedTasks=" + this.getRejectionCount() + "}";
    }

    @Override
    protected void beforeExecute(Thread t, Runnable r) {
        this.activeThreads.incrementAndGet();
    }

    @Override
    protected void afterExecute(Runnable r, Throwable t) {
        this.completedTasks.incrementAndGet();
        this.totalActiveThreads.add(this.activeThreads.getAndDecrement());
    }

    @Override
    public void shutdown() {
        Event.write();
        super.shutdown();
    }

    private static WorkQueue createQueue(int capacity, int corePoolSize, int maxPoolSize, int growthThreshold, int growthRate) {
        if (maxPoolSize == corePoolSize || growthRate == 0) {
            return new WorkQueue(capacity);
        }
        RateLimitGrowth growthPolicy = new RateLimitGrowth(growthThreshold, growthRate);
        return new DynamicPoolWorkQueue(growthPolicy, capacity, maxPoolSize);
    }

    private static class Event
    implements Comparable<Event> {
        private static final int MAX_EVENTS = Event.getIntProperty("thread.pool.events", 0);
        private static final int DELAY_SECONDS = Event.getIntProperty("thread.pool.events.delay", 0);
        private static final List<Event> EVENTS = MAX_EVENTS == 0 ? Collections.emptyList() : new ArrayList(MAX_EVENTS);
        private static final String EVENTS_FILE_NAME = "thread-pool-events.csv";
        private static final String FILE_HEADER = "Elapsed Seconds,Completed Tasks,Event,Threads,Active Threads,Queue Size%n";
        private static final AtomicBoolean STARTED = new AtomicBoolean();
        private static final AtomicBoolean WRITTEN = new AtomicBoolean();
        private static final long START_TIME = ManagementFactory.getRuntimeMXBean().getStartTime();
        private final long time = System.currentTimeMillis();
        private final Type type;
        private final int threads;
        private final int activeThreads;
        private final int queueSize;
        private final int completedTasks;

        private Event(Type type, ThreadPool pool, WorkQueue queue) {
            this.type = type;
            this.threads = pool.getPoolSize();
            this.activeThreads = pool.getActiveThreads();
            this.queueSize = queue.size();
            this.completedTasks = pool.getCompletedTasks();
        }

        @Override
        public int compareTo(Event o) {
            return Long.compare(this.time, o.time);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            Event event = (Event)o;
            return this.time == event.time && this.threads == event.threads && this.activeThreads == event.activeThreads && this.queueSize == event.queueSize && this.completedTasks == event.completedTasks && this.type == event.type;
        }

        public int hashCode() {
            return Objects.hash(new Object[]{this.time, this.type, this.threads, this.activeThreads, this.queueSize, this.completedTasks});
        }

        private String toCsv() {
            float elapsedMillis = this.time - START_TIME;
            float elapsedSeconds = elapsedMillis / 1000.0f;
            return String.format("%.4f,%d,%s,%d,%d,%d%n", new Object[]{Float.valueOf(elapsedSeconds), this.completedTasks, this.type, this.threads, this.activeThreads, this.queueSize});
        }

        private static void add(Type type, ThreadPool pool, WorkQueue queue) {
            if (Event.shouldAdd()) {
                if (!STARTED.getAndSet(true)) {
                    LOGGER.info("Recording up to " + MAX_EVENTS + " thread pool events");
                    for (GarbageCollectorMXBean bean : ManagementFactory.getGarbageCollectorMXBeans()) {
                        NotificationEmitter emitter = (NotificationEmitter)((Object)bean);
                        emitter.addNotificationListener((notification, handback) -> {
                            if (notification.getType().equals("com.sun.management.gc.notification") && !WRITTEN.get()) {
                                Event.add(Type.GC, pool, queue);
                            }
                        }, null, null);
                    }
                    Runtime.getRuntime().addShutdownHook(new Thread(Event::write));
                }
                EVENTS.add(new Event(type, pool, queue));
            }
        }

        private static boolean shouldAdd() {
            if (EVENTS.size() < MAX_EVENTS) {
                if (DELAY_SECONDS == 0) {
                    return true;
                }
                long elapsedMillis = System.currentTimeMillis() - START_TIME;
                return elapsedMillis / 1000L >= (long)DELAY_SECONDS;
            }
            return false;
        }

        private static void write() {
            if (!EVENTS.isEmpty() && !WRITTEN.getAndSet(true)) {
                Path file = Paths.get(EVENTS_FILE_NAME, new String[0]).toAbsolutePath();
                LOGGER.info("Writing thread pool events to " + file);
                EVENTS.sort(null);
                try (OutputStream out = Files.newOutputStream(file, StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);){
                    out.write(FILE_HEADER.getBytes(StandardCharsets.UTF_8));
                    for (Event event : EVENTS) {
                        out.write(event.toCsv().getBytes(StandardCharsets.UTF_8));
                    }
                    LOGGER.info("Finished writing thread pool events");
                }
                catch (Throwable e) {
                    LOGGER.warning("failed to write thread pool events" + e);
                }
            }
        }

        private static int getIntProperty(String propertyName, int defaultValue) {
            String value = System.getProperty(propertyName);
            return value == null ? defaultValue : Integer.parseInt(value);
        }

        static enum Type {
            IDLE,
            MAX,
            BELOW,
            ADD,
            WAIT,
            GC;

        }
    }

    private static class RateLimitGrowth
    implements Predicate<ThreadPool> {
        private static final int ALWAYS_THRESHOLD_MULTIPLIER = 8;
        private final int queueThreshold;
        private final int alwaysThreshold;
        private final boolean alwaysAdd;
        private final float rate;

        RateLimitGrowth(int queueThreshold, int growthRate) {
            this.queueThreshold = queueThreshold;
            this.alwaysThreshold = queueThreshold * 8;
            this.alwaysAdd = growthRate == 100;
            this.rate = (float)growthRate / 100.0f;
        }

        @Override
        public boolean test(ThreadPool pool) {
            WorkQueue queue = pool.getQueue();
            int queueSize = queue.size();
            if (queueSize > this.queueThreshold) {
                if (this.alwaysAdd || queueSize > this.alwaysThreshold || ThreadLocalRandom.current().nextFloat() < this.rate) {
                    Event.add(Event.Type.ADD, pool, queue);
                    return true;
                }
                Event.add(Event.Type.WAIT, pool, queue);
                return false;
            }
            Event.add(Event.Type.BELOW, pool, queue);
            return false;
        }
    }

    static final class DynamicPoolWorkQueue
    extends WorkQueue {
        private final Predicate<ThreadPool> growthPolicy;
        private final int maxPoolSize;
        private ThreadPool pool;

        DynamicPoolWorkQueue(Predicate<ThreadPool> growthPolicy, int capacity, int maxPoolSize) {
            super(capacity);
            this.maxPoolSize = maxPoolSize;
            this.growthPolicy = growthPolicy;
        }

        @Override
        void setPool(ThreadPool pool) {
            this.pool = pool;
        }

        @Override
        public boolean offer(Runnable task) {
            int currentSize = this.pool.getPoolSize();
            if (currentSize >= this.maxPoolSize) {
                Event.add(Event.Type.MAX, this.pool, this);
                return this.enqueue(task);
            }
            if (this.pool.getActiveThreads() < currentSize) {
                Event.add(Event.Type.IDLE, this.pool, this);
                return this.enqueue(task);
            }
            if (this.growthPolicy.test(this.pool)) {
                if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.fine("Adding a thread, pool size = " + this.pool.getPoolSize() + ", queue size = " + this.size());
                }
                return false;
            }
            return this.enqueue(task);
        }

        private void writeObject(ObjectOutputStream stream) throws IOException {
            throw new UnsupportedOperationException("cannot serialize");
        }

        private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException {
            throw new UnsupportedOperationException("cannot deserialize");
        }
    }

    static class WorkQueue
    extends LinkedBlockingQueue<Runnable> {
        private final int capacity;
        private final LongAdder totalSize;
        private final AtomicInteger totalTasks;
        private final AtomicInteger peakSize;

        WorkQueue(int capacity) {
            super(capacity);
            this.capacity = capacity;
            this.totalSize = new LongAdder();
            this.totalTasks = new AtomicInteger();
            this.peakSize = new AtomicInteger();
        }

        void setPool(ThreadPool pool) {
        }

        @Override
        public boolean offer(Runnable task) {
            return this.enqueue(task);
        }

        boolean enqueue(Runnable task) {
            if (super.offer(task)) {
                int queueSize = this.size();
                if (queueSize > this.peakSize.get()) {
                    this.peakSize.set(queueSize);
                }
                this.totalSize.add(queueSize);
                this.totalTasks.incrementAndGet();
                return true;
            }
            return false;
        }

        public int getCapacity() {
            return this.capacity;
        }

        public float getAverageSize() {
            float totalSize = this.totalSize.sum();
            if (totalSize == 0.0f) {
                return 0.0f;
            }
            return totalSize / (float)this.totalTasks.get();
        }

        public int getPeakSize() {
            return this.peakSize.get();
        }
    }

    private static class GroupedThreadFactory
    implements ThreadFactory {
        private final ThreadGroup group;
        private final String namePrefix;
        private final boolean useDaemonThreads;
        private final AtomicInteger threadCount;

        GroupedThreadFactory(String groupName, String threadNamePrefix, boolean useDaemonThreads) {
            this.group = new ThreadGroup(groupName);
            this.namePrefix = threadNamePrefix;
            this.useDaemonThreads = useDaemonThreads;
            this.threadCount = new AtomicInteger();
        }

        @Override
        public Thread newThread(Runnable runnable) {
            String name = this.namePrefix + this.threadCount.incrementAndGet();
            Thread thread = new Thread(this.group, runnable, name);
            thread.setDaemon(this.useDaemonThreads);
            return thread;
        }
    }

    public static class RejectionHandler
    implements RejectedExecutionHandler {
        private final AtomicInteger rejections = new AtomicInteger();

        @Override
        public void rejectedExecution(Runnable task, ThreadPoolExecutor executor) {
            WorkQueue queue = ((ThreadPool)executor).getQueue();
            if (!queue.enqueue(task)) {
                LOGGER.warning(RejectionHandler.rejectionMessage(executor));
                this.rejections.incrementAndGet();
                this.throwException(executor);
            }
        }

        public int getRejectionCount() {
            return this.rejections.get();
        }

        protected void throwException(ThreadPoolExecutor executor) {
            throw new RejectedExecutionException(RejectionHandler.rejectionMessage(executor));
        }

        private static String rejectionMessage(ThreadPoolExecutor executor) {
            ThreadPool pool = (ThreadPool)executor;
            return "Task rejected by ThreadPool '" + pool.getName() + "': queue is full";
        }
    }
}

