/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.common.crypto;

import io.helidon.common.Base64Value;
import io.helidon.common.LazyValue;
import io.helidon.common.crypto.CommonCipher;
import io.helidon.common.crypto.CryptoException;
import io.helidon.common.crypto.PasswordKeyDerivation;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.security.Key;
import java.security.SecureRandom;
import java.security.spec.AlgorithmParameterSpec;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.spec.ChaCha20ParameterSpec;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;

public class SymmetricCipher
implements CommonCipher {
    public static final String ALGORITHM_AES_CBC = "AES/CBC/PKCS5Padding";
    public static final String ALGORITHM_AES_CTR = "AES/CTR/NoPadding";
    public static final String ALGORITHM_AES_GCM = "AES/GCM/NoPadding";
    public static final String ALGORITHM_CHA_CHA = "ChaCha20";
    public static final String ALGORITHM_CHA_CHA_POLY1305 = "ChaCha20-Poly1305";
    private static final LazyValue<SecureRandom> SECURE_RANDOM = LazyValue.create(SecureRandom::new);
    private static final Pattern PATTERN_ALGORITHM = Pattern.compile("^(\\S+)/\\S+/\\S+$");
    private static final int SALT_LENGTH = 16;
    private final String algorithm;
    private final String provider;
    private final char[] password;
    private final int keySize;
    private final int numberOfIterations;

    private SymmetricCipher(Builder builder) {
        this.algorithm = builder.algorithm;
        this.provider = builder.provider;
        this.password = builder.password;
        this.keySize = builder.keySize;
        this.numberOfIterations = builder.numberOfIterations;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static SymmetricCipher create(char[] password) {
        return new Builder().password(password).build();
    }

    public static Base64Value encrypt(String algorithm, byte[] key, byte[] iv, Base64Value plain) {
        return SymmetricCipher.encrypt(algorithm, null, key, iv, plain);
    }

    public static Base64Value encrypt(String algorithm, String provider, byte[] key, byte[] iv, Base64Value plain) {
        Objects.requireNonNull(algorithm, "Algorithm cannot be null");
        Objects.requireNonNull(iv, "Initialization vector cannot be null");
        return SymmetricCipher.encrypt(algorithm, provider, key, SymmetricCipher.createAlgorithmParameter(algorithm, iv), plain);
    }

    public static Base64Value encrypt(String algorithm, String provider, byte[] key, AlgorithmParameterSpec params, Base64Value plain) {
        Objects.requireNonNull(algorithm, "Algorithm cannot be null");
        Objects.requireNonNull(key, "Key cannot be null");
        Objects.requireNonNull(params, "Algorithm parameters cannot be null");
        Objects.requireNonNull(plain, "Plain content cannot be null");
        Cipher cipher = SymmetricCipher.cipher(algorithm, provider, key, params, 1);
        try {
            return Base64Value.create((byte[])cipher.doFinal(plain.toBytes()));
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new CryptoException("Failed to encrypt the message", e);
        }
    }

    public static Base64Value decrypt(String algorithm, byte[] key, byte[] iv, Base64Value encrypted) {
        return SymmetricCipher.decrypt(algorithm, null, key, iv, encrypted);
    }

    public static Base64Value decrypt(String algorithm, String provider, byte[] key, byte[] iv, Base64Value encrypted) {
        Objects.requireNonNull(algorithm, "Algorithm cannot be null");
        Objects.requireNonNull(iv, "Initialization vector cannot be null");
        return SymmetricCipher.decrypt(algorithm, provider, key, SymmetricCipher.createAlgorithmParameter(algorithm, iv), encrypted);
    }

    public static Base64Value decrypt(String algorithm, String provider, byte[] key, AlgorithmParameterSpec params, Base64Value encrypted) {
        Objects.requireNonNull(algorithm, "Algorithm cannot be null");
        Objects.requireNonNull(key, "Key cannot be null");
        Objects.requireNonNull(encrypted, "Encrypted content cannot be null");
        Cipher cipher = SymmetricCipher.cipher(algorithm, provider, key, params, 2);
        try {
            return Base64Value.create((byte[])cipher.doFinal(encrypted.toBytes()));
        }
        catch (BadPaddingException | IllegalBlockSizeException e) {
            throw new CryptoException("Failed to decrypt the message", e);
        }
    }

    private static Cipher cipher(String algorithm, String provider, byte[] key, AlgorithmParameterSpec parameterSpec, int cipherMode) {
        try {
            Matcher matcher = PATTERN_ALGORITHM.matcher(algorithm);
            String keySpecAlg = matcher.matches() ? matcher.group(1) : algorithm;
            SecretKeySpec spec = new SecretKeySpec(key, keySpecAlg);
            Cipher cipher = provider == null ? Cipher.getInstance(algorithm) : Cipher.getInstance(algorithm, provider);
            if (parameterSpec == null) {
                cipher.init(cipherMode, spec);
            } else {
                cipher.init(cipherMode, (Key)spec, parameterSpec);
            }
            return cipher;
        }
        catch (Exception e) {
            throw new CryptoException("Failed to prepare a cipher instance", e);
        }
    }

    private static AlgorithmParameterSpec createAlgorithmParameter(String algorithm, byte[] iv) {
        switch (algorithm) {
            case "AES/GCM/NoPadding": {
                return new GCMParameterSpec(128, iv);
            }
            case "ChaCha20": {
                return new ChaCha20ParameterSpec(iv, 1);
            }
        }
        return new IvParameterSpec(iv);
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public Base64Value encrypt(Base64Value message) {
        Objects.requireNonNull(message, "Plain content cannot be null");
        try (ByteArrayOutputStream outputStream = new ByteArrayOutputStream();){
            Base64Value base64Value;
            try (DataOutputStream dataOutputStream = new DataOutputStream(outputStream);){
                byte[] salt = new byte[16];
                ((SecureRandom)SECURE_RANDOM.get()).nextBytes(salt);
                byte[] key = PasswordKeyDerivation.deriveKey(this.password, salt, this.numberOfIterations, this.keySize);
                Cipher cipher = SymmetricCipher.cipher(this.algorithm, this.provider, key, null, 1);
                byte[] iv = cipher.getIV();
                outputStream.writeBytes(salt);
                dataOutputStream.writeInt(iv.length);
                outputStream.writeBytes(iv);
                outputStream.writeBytes(cipher.doFinal(message.toBytes()));
                base64Value = Base64Value.create((byte[])outputStream.toByteArray());
            }
            return base64Value;
        }
        catch (IOException | BadPaddingException | IllegalBlockSizeException e) {
            throw new CryptoException("An error occurred while message encryption", e);
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public Base64Value decrypt(Base64Value encrypted) {
        Objects.requireNonNull(encrypted, "Encrypted content cannot be null");
        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(encrypted.toBytes());){
            Base64Value base64Value;
            try (DataInputStream dataInputStream = new DataInputStream(inputStream);){
                byte[] salt = inputStream.readNBytes(16);
                int ivSize = dataInputStream.readInt();
                byte[] iv = inputStream.readNBytes(ivSize);
                byte[] key = PasswordKeyDerivation.deriveKey(this.password, salt, this.numberOfIterations, this.keySize);
                base64Value = SymmetricCipher.decrypt(this.algorithm, this.provider, key, iv, Base64Value.create((byte[])inputStream.readAllBytes()));
            }
            return base64Value;
        }
        catch (EOFException e) {
            throw new CryptoException("Encrypted value is not valid", e);
        }
        catch (IOException e) {
            throw new CryptoException("An error occurred while message decryption", e);
        }
    }

    public static class Builder
    implements io.helidon.common.Builder<Builder, SymmetricCipher> {
        private String algorithm = "AES/GCM/NoPadding";
        private String provider = null;
        private Integer numberOfIterations = 10000;
        private Integer keySize = 256;
        private char[] password;

        private Builder() {
        }

        public Builder algorithm(String algorithm) {
            this.algorithm = Objects.requireNonNull(algorithm, "Algorithm cannot be null");
            return this;
        }

        public Builder provider(String provider) {
            this.provider = provider;
            return this;
        }

        public Builder password(char[] password) {
            Objects.requireNonNull(password, "Password cannot be null");
            this.password = (char[])password.clone();
            return this;
        }

        public Builder keySize(int keySize) {
            this.keySize = keySize;
            return this;
        }

        public Builder numberOfIterations(int numberOfIterations) {
            this.numberOfIterations = numberOfIterations;
            return this;
        }

        public SymmetricCipher build() {
            if (this.password == null) {
                throw new CryptoException("Password has to be specified.");
            }
            return new SymmetricCipher(this);
        }
    }
}

