/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.common.tls;

import java.net.URI;
import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Certificate revocation configuration.
 * This configuration determines whether client certificate validation should include checking if
 * it is still considered valid by the certificate authority.
 * <br>
 * Types of certificate validation checks:
 * <ul>
 *     <li>CRL - shortcut name for Certificate Revocation List. It is a list of certificates that have
 *     been revoked by a certificate authority before their expiration date</li>
 *     <li>OCSP - shortcut name for Online Certificate Status Protocol. It  is a real-time protocol used
 *     to check the status of a certificate, providing immediate verification of its validity</li>
 * </ul>
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.common.tls.RevocationConfigBlueprint")
public interface RevocationConfig extends RevocationConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static RevocationConfig.Builder builder() {
        return new RevocationConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static RevocationConfig.Builder builder(RevocationConfig instance) {
        return RevocationConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static RevocationConfig create(Config config) {
        return RevocationConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static RevocationConfig create() {
        return RevocationConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link RevocationConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends RevocationConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends RevocationConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private boolean checkOnlyEndEntity = false;
        private boolean enabled = false;
        private boolean fallbackEnabled = true;
        private boolean preferCrlOverOcsp = false;
        private boolean softFailEnabled = false;
        private Config config;
        private URI ocspResponderUri;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(RevocationConfig prototype) {
            enabled(prototype.enabled());
            preferCrlOverOcsp(prototype.preferCrlOverOcsp());
            checkOnlyEndEntity(prototype.checkOnlyEndEntity());
            fallbackEnabled(prototype.fallbackEnabled());
            softFailEnabled(prototype.softFailEnabled());
            ocspResponderUri(prototype.ocspResponderUri());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(RevocationConfig.BuilderBase<?, ?> builder) {
            enabled(builder.enabled());
            preferCrlOverOcsp(builder.preferCrlOverOcsp());
            checkOnlyEndEntity(builder.checkOnlyEndEntity());
            fallbackEnabled(builder.fallbackEnabled());
            softFailEnabled(builder.softFailEnabled());
            builder.ocspResponderUri().ifPresent(this::ocspResponderUri);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("enabled").as(Boolean.class).ifPresent(this::enabled);
            config.get("prefer-crl-over-ocsp").as(Boolean.class).ifPresent(this::preferCrlOverOcsp);
            config.get("check-only-end-entity").as(Boolean.class).ifPresent(this::checkOnlyEndEntity);
            config.get("fallback-enabled").as(Boolean.class).ifPresent(this::fallbackEnabled);
            config.get("soft-fail-enabled").as(Boolean.class).ifPresent(this::softFailEnabled);
            config.get("ocsp-responder-uri").as(URI.class).ifPresent(this::ocspResponderUri);
            return self();
        }

        /**
         * Flag indicating whether this revocation config is enabled.
         *
         * @param enabled enabled flag
         * @return updated builder instance
         * @see #enabled()
         */
        public BUILDER enabled(boolean enabled) {
            this.enabled = enabled;
            return self();
        }

        /**
         * Prefer CRL over OCSP.
         * Default value is {@code false}. OCSP is preferred over the CRL by default.
         *
         * @param preferCrlOverOcsp whether to prefer CRL over OCSP
         * @return updated builder instance
         * @see #preferCrlOverOcsp()
         */
        public BUILDER preferCrlOverOcsp(boolean preferCrlOverOcsp) {
            this.preferCrlOverOcsp = preferCrlOverOcsp;
            return self();
        }

        /**
         * Only check the revocation status of end-entity certificates.
         * Default value is {@code false}.
         *
         * @param checkOnlyEndEntity whether to check only end-entity certificates
         * @return updated builder instance
         * @see #checkOnlyEndEntity()
         */
        public BUILDER checkOnlyEndEntity(boolean checkOnlyEndEntity) {
            this.checkOnlyEndEntity = checkOnlyEndEntity;
            return self();
        }

        /**
         * Enable fallback to the less preferred checking option.
         * <br>
         * If the primary method for revocation checking fails to verify the revocation status of a certificate
         * (such as using a CRL or OCSP), the checker will attempt alternative methods. This option ensures
         * whether revocation checking is performed strictly according to the specified method, or should fallback
         * to the one less preferred. OCSP is preferred over the CRL by default.
         *
         * @param fallbackEnabled whether to allow fallback to the less preferred checking option
         * @return updated builder instance
         * @see #fallbackEnabled()
         */
        public BUILDER fallbackEnabled(boolean fallbackEnabled) {
            this.fallbackEnabled = fallbackEnabled;
            return self();
        }

        /**
         * Allow revocation check to succeed if the revocation status cannot be
         * determined for one of the following reasons:
         * <ul>
         *  <li>The CRL or OCSP response cannot be obtained because of a
         *      network error.
         *  <li>The OCSP responder returns one of the following errors
         *      specified in section 2.3 of RFC 2560: internalError or tryLater.
         * </ul>
         *
         * @param softFailEnabled whether soft fail is enabled
         * @return updated builder instance
         * @see #softFailEnabled()
         */
        public BUILDER softFailEnabled(boolean softFailEnabled) {
            this.softFailEnabled = softFailEnabled;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #ocspResponderUri()
         */
        public BUILDER clearOcspResponderUri() {
            this.ocspResponderUri = null;
            return self();
        }

        /**
         * The URI that identifies the location of the OCSP responder. This
         * overrides the {@code ocsp.responderURL} security property and any
         * responder specified in a certificate's Authority Information Access
         * Extension, as defined in RFC 5280.
         *
         * @param ocspResponderUri OCSP responder URI
         * @return updated builder instance
         * @see #ocspResponderUri()
         */
        public BUILDER ocspResponderUri(URI ocspResponderUri) {
            Objects.requireNonNull(ocspResponderUri);
            this.ocspResponderUri = ocspResponderUri;
            return self();
        }

        /**
         * Flag indicating whether this revocation config is enabled.
         *
         * @return the enabled
         */
        public boolean enabled() {
            return enabled;
        }

        /**
         * Prefer CRL over OCSP.
         * Default value is {@code false}. OCSP is preferred over the CRL by default.
         *
         * @return the prefer crl over ocsp
         */
        public boolean preferCrlOverOcsp() {
            return preferCrlOverOcsp;
        }

        /**
         * Only check the revocation status of end-entity certificates.
         * Default value is {@code false}.
         *
         * @return the check only end entity
         */
        public boolean checkOnlyEndEntity() {
            return checkOnlyEndEntity;
        }

        /**
         * Enable fallback to the less preferred checking option.
         * <br>
         * If the primary method for revocation checking fails to verify the revocation status of a certificate
         * (such as using a CRL or OCSP), the checker will attempt alternative methods. This option ensures
         * whether revocation checking is performed strictly according to the specified method, or should fallback
         * to the one less preferred. OCSP is preferred over the CRL by default.
         *
         * @return the fallback enabled
         */
        public boolean fallbackEnabled() {
            return fallbackEnabled;
        }

        /**
         * Allow revocation check to succeed if the revocation status cannot be
         * determined for one of the following reasons:
         * <ul>
         *  <li>The CRL or OCSP response cannot be obtained because of a
         *      network error.
         *  <li>The OCSP responder returns one of the following errors
         *      specified in section 2.3 of RFC 2560: internalError or tryLater.
         * </ul>
         *
         * @return the soft fail enabled
         */
        public boolean softFailEnabled() {
            return softFailEnabled;
        }

        /**
         * The URI that identifies the location of the OCSP responder. This
         * overrides the {@code ocsp.responderURL} security property and any
         * responder specified in a certificate's Authority Information Access
         * Extension, as defined in RFC 5280.
         *
         * @return the ocsp responder uri
         */
        public Optional<URI> ocspResponderUri() {
            return Optional.ofNullable(ocspResponderUri);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "RevocationConfigBuilder{"
                    + "enabled=" + enabled + ","
                    + "preferCrlOverOcsp=" + preferCrlOverOcsp + ","
                    + "checkOnlyEndEntity=" + checkOnlyEndEntity + ","
                    + "fallbackEnabled=" + fallbackEnabled + ","
                    + "softFailEnabled=" + softFailEnabled + ","
                    + "ocspResponderUri=" + ocspResponderUri
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * The URI that identifies the location of the OCSP responder. This
         * overrides the {@code ocsp.responderURL} security property and any
         * responder specified in a certificate's Authority Information Access
         * Extension, as defined in RFC 5280.
         *
         * @param ocspResponderUri OCSP responder URI
         * @return updated builder instance
         * @see #ocspResponderUri()
         */
        BUILDER ocspResponderUri(Optional<? extends URI> ocspResponderUri) {
            Objects.requireNonNull(ocspResponderUri);
            this.ocspResponderUri = ocspResponderUri.map(java.net.URI.class::cast).orElse(this.ocspResponderUri);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class RevocationConfigImpl implements RevocationConfig {

            private final boolean checkOnlyEndEntity;
            private final boolean enabled;
            private final boolean fallbackEnabled;
            private final boolean preferCrlOverOcsp;
            private final boolean softFailEnabled;
            private final Optional<URI> ocspResponderUri;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected RevocationConfigImpl(RevocationConfig.BuilderBase<?, ?> builder) {
                this.enabled = builder.enabled();
                this.preferCrlOverOcsp = builder.preferCrlOverOcsp();
                this.checkOnlyEndEntity = builder.checkOnlyEndEntity();
                this.fallbackEnabled = builder.fallbackEnabled();
                this.softFailEnabled = builder.softFailEnabled();
                this.ocspResponderUri = builder.ocspResponderUri();
            }

            @Override
            public boolean enabled() {
                return enabled;
            }

            @Override
            public boolean preferCrlOverOcsp() {
                return preferCrlOverOcsp;
            }

            @Override
            public boolean checkOnlyEndEntity() {
                return checkOnlyEndEntity;
            }

            @Override
            public boolean fallbackEnabled() {
                return fallbackEnabled;
            }

            @Override
            public boolean softFailEnabled() {
                return softFailEnabled;
            }

            @Override
            public Optional<URI> ocspResponderUri() {
                return ocspResponderUri;
            }

            @Override
            public String toString() {
                return "RevocationConfig{"
                        + "enabled=" + enabled + ","
                        + "preferCrlOverOcsp=" + preferCrlOverOcsp + ","
                        + "checkOnlyEndEntity=" + checkOnlyEndEntity + ","
                        + "fallbackEnabled=" + fallbackEnabled + ","
                        + "softFailEnabled=" + softFailEnabled + ","
                        + "ocspResponderUri=" + ocspResponderUri
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof RevocationConfig other)) {
                    return false;
                }
                return enabled == other.enabled()
                    && preferCrlOverOcsp == other.preferCrlOverOcsp()
                    && checkOnlyEndEntity == other.checkOnlyEndEntity()
                    && fallbackEnabled == other.fallbackEnabled()
                    && softFailEnabled == other.softFailEnabled()
                    && Objects.equals(ocspResponderUri, other.ocspResponderUri());
            }

            @Override
            public int hashCode() {
                return Objects.hash(enabled, preferCrlOverOcsp, checkOnlyEndEntity, fallbackEnabled, softFailEnabled, ocspResponderUri);
            }

        }

    }

    /**
     * Fluent API builder for {@link RevocationConfig}.
     */
    class Builder extends RevocationConfig.BuilderBase<RevocationConfig.Builder, RevocationConfig> implements io.helidon.common.Builder<RevocationConfig.Builder, RevocationConfig> {

        private Builder() {
        }

        @Override
        public RevocationConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new RevocationConfigImpl(this);
        }

        @Override
        public RevocationConfig build() {
            return buildPrototype();
        }

    }

}
