/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.common.tls;

import java.security.PrivateKey;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.ServiceLoader;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLParameters;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.HelidonServiceLoader;
import io.helidon.common.config.Config;
import io.helidon.common.pki.Keys;
import io.helidon.common.tls.spi.TlsManagerProvider;

/**
 * Interface generated from definition. Please add javadoc to the definition interface.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.codegen.BuilderCodegen", trigger = "io.helidon.common.tls.TlsConfigBlueprint")
public interface TlsConfig extends TlsConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static TlsConfig.Builder builder() {
        return new TlsConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static TlsConfig.Builder builder(TlsConfig instance) {
        return TlsConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static TlsConfig create(Config config) {
        return TlsConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static TlsConfig create() {
        return TlsConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Tls}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends TlsConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends TlsConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<String> applicationProtocols = new ArrayList<>();
        private final List<String> enabledCipherSuites = new ArrayList<>();
        private final List<String> enabledProtocols = new ArrayList<>();
        private final List<X509Certificate> privateKeyCertChain = new ArrayList<>();
        private final List<X509Certificate> trust = new ArrayList<>();
        private boolean enabled = true;
        private boolean isApplicationProtocolsMutated;
        private boolean isEnabledCipherSuitesMutated;
        private boolean isEnabledProtocolsMutated;
        private boolean isPrivateKeyCertChainMutated;
        private boolean isTrustMutated;
        private boolean managerDiscoverServices = false;
        private boolean trustAll = false;
        private Config config;
        private Duration sessionTimeout = Duration.parse("PT24H");
        private int sessionCacheSize = 20480;
        private PrivateKey privateKey;
        private RevocationConfig revocation;
        private SecureRandom secureRandom;
        private SSLContext sslContext;
        private SSLParameters sslParameters;
        private String endpointIdentificationAlgorithm = "HTTPS";
        private String internalKeystoreProvider;
        private String internalKeystoreType;
        private String keyManagerFactoryAlgorithm;
        private String keyManagerFactoryProvider;
        private String protocol = "TLS";
        private String provider;
        private String secureRandomAlgorithm;
        private String secureRandomProvider;
        private String trustManagerFactoryAlgorithm;
        private String trustManagerFactoryProvider;
        private TlsClientAuth clientAuth = TlsClientAuth.NONE;
        private TlsManager manager;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance. This method disables automatic service discovery.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(TlsConfig prototype) {
            sslContext(prototype.sslContext());
            privateKey(prototype.privateKey());
            if (!isPrivateKeyCertChainMutated) {
                privateKeyCertChain.clear();
            }
            addPrivateKeyCertChain(prototype.privateKeyCertChain());
            if (!isTrustMutated) {
                trust.clear();
            }
            addTrust(prototype.trust());
            manager(prototype.manager());
            managerDiscoverServices = false;
            secureRandom(prototype.secureRandom());
            sslParameters(prototype.sslParameters());
            secureRandomProvider(prototype.secureRandomProvider());
            secureRandomAlgorithm(prototype.secureRandomAlgorithm());
            keyManagerFactoryAlgorithm(prototype.keyManagerFactoryAlgorithm());
            keyManagerFactoryProvider(prototype.keyManagerFactoryProvider());
            trustManagerFactoryAlgorithm(prototype.trustManagerFactoryAlgorithm());
            trustManagerFactoryProvider(prototype.trustManagerFactoryProvider());
            if (!isApplicationProtocolsMutated) {
                applicationProtocols.clear();
            }
            addApplicationProtocols(prototype.applicationProtocols());
            endpointIdentificationAlgorithm(prototype.endpointIdentificationAlgorithm());
            enabled(prototype.enabled());
            trustAll(prototype.trustAll());
            clientAuth(prototype.clientAuth());
            protocol(prototype.protocol());
            provider(prototype.provider());
            if (!isEnabledCipherSuitesMutated) {
                enabledCipherSuites.clear();
            }
            addEnabledCipherSuites(prototype.enabledCipherSuites());
            if (!isEnabledProtocolsMutated) {
                enabledProtocols.clear();
            }
            addEnabledProtocols(prototype.enabledProtocols());
            sessionCacheSize(prototype.sessionCacheSize());
            sessionTimeout(prototype.sessionTimeout());
            internalKeystoreType(prototype.internalKeystoreType());
            internalKeystoreProvider(prototype.internalKeystoreProvider());
            revocation(prototype.revocation());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(TlsConfig.BuilderBase<?, ?> builder) {
            builder.sslContext().ifPresent(this::sslContext);
            builder.privateKey().ifPresent(this::privateKey);
            if (isPrivateKeyCertChainMutated) {
                if (builder.isPrivateKeyCertChainMutated) {
                    addPrivateKeyCertChain(builder.privateKeyCertChain);
                }
            } else {
                privateKeyCertChain.clear();
                addPrivateKeyCertChain(builder.privateKeyCertChain);
            }
            if (isTrustMutated) {
                if (builder.isTrustMutated) {
                    addTrust(builder.trust);
                }
            } else {
                trust.clear();
                addTrust(builder.trust);
            }
            builder.manager().ifPresent(this::manager);
            managerDiscoverServices = builder.managerDiscoverServices;
            builder.secureRandom().ifPresent(this::secureRandom);
            builder.sslParameters().ifPresent(this::sslParameters);
            builder.secureRandomProvider().ifPresent(this::secureRandomProvider);
            builder.secureRandomAlgorithm().ifPresent(this::secureRandomAlgorithm);
            builder.keyManagerFactoryAlgorithm().ifPresent(this::keyManagerFactoryAlgorithm);
            builder.keyManagerFactoryProvider().ifPresent(this::keyManagerFactoryProvider);
            builder.trustManagerFactoryAlgorithm().ifPresent(this::trustManagerFactoryAlgorithm);
            builder.trustManagerFactoryProvider().ifPresent(this::trustManagerFactoryProvider);
            if (isApplicationProtocolsMutated) {
                if (builder.isApplicationProtocolsMutated) {
                    addApplicationProtocols(builder.applicationProtocols);
                }
            } else {
                applicationProtocols.clear();
                addApplicationProtocols(builder.applicationProtocols);
            }
            endpointIdentificationAlgorithm(builder.endpointIdentificationAlgorithm());
            enabled(builder.enabled());
            trustAll(builder.trustAll());
            clientAuth(builder.clientAuth());
            protocol(builder.protocol());
            builder.provider().ifPresent(this::provider);
            if (isEnabledCipherSuitesMutated) {
                if (builder.isEnabledCipherSuitesMutated) {
                    addEnabledCipherSuites(builder.enabledCipherSuites);
                }
            } else {
                enabledCipherSuites.clear();
                addEnabledCipherSuites(builder.enabledCipherSuites);
            }
            if (isEnabledProtocolsMutated) {
                if (builder.isEnabledProtocolsMutated) {
                    addEnabledProtocols(builder.enabledProtocols);
                }
            } else {
                enabledProtocols.clear();
                addEnabledProtocols(builder.enabledProtocols);
            }
            sessionCacheSize(builder.sessionCacheSize());
            sessionTimeout(builder.sessionTimeout());
            builder.internalKeystoreType().ifPresent(this::internalKeystoreType);
            builder.internalKeystoreProvider().ifPresent(this::internalKeystoreProvider);
            builder.revocation().ifPresent(this::revocation);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("private-key").map(Keys::create).ifPresent(this::privateKey);
            config.get("private-key").map(Keys::create).ifPresent(this::privateKeyCertChain);
            config.get("trust").map(Keys::create).ifPresent(this::trust);
            config.get("secure-random-provider").as(String.class).ifPresent(this::secureRandomProvider);
            config.get("secure-random-algorithm").as(String.class).ifPresent(this::secureRandomAlgorithm);
            config.get("key-manager-factory-algorithm").as(String.class).ifPresent(this::keyManagerFactoryAlgorithm);
            config.get("trust-manager-factory-algorithm").as(String.class).ifPresent(this::trustManagerFactoryAlgorithm);
            config.get("endpoint-identification-algorithm").as(String.class).ifPresent(this::endpointIdentificationAlgorithm);
            config.get("enabled").as(Boolean.class).ifPresent(this::enabled);
            config.get("trust-all").as(Boolean.class).ifPresent(this::trustAll);
            config.get("client-auth").as(TlsClientAuth.class).ifPresent(this::clientAuth);
            config.get("protocol").as(String.class).ifPresent(this::protocol);
            config.get("provider").as(String.class).ifPresent(this::provider);
            config.get("cipher-suite").asList(String.class).ifPresent(this::enabledCipherSuites);
            config.get("protocols").asList(String.class).ifPresent(this::enabledProtocols);
            config.get("session-cache-size").as(Integer.class).ifPresent(this::sessionCacheSize);
            config.get("session-timeout").as(Duration.class).ifPresent(this::sessionTimeout);
            config.get("internal-keystore-type").as(String.class).ifPresent(this::internalKeystoreType);
            config.get("internal-keystore-provider").as(String.class).ifPresent(this::internalKeystoreProvider);
            config.get("revocation").map(RevocationConfig::create).ifPresent(this::revocation);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #sslContext()
         */
        public BUILDER clearSslContext() {
            this.sslContext = null;
            return self();
        }

        /**
         * Provide a fully configured {@link javax.net.ssl.SSLContext}. If defined, context related configuration
         * is ignored.
         *
         * @param sslContext SSL context to use
         * @return updated builder instance
         * @see #sslContext()
         */
        public BUILDER sslContext(SSLContext sslContext) {
            Objects.requireNonNull(sslContext);
            this.sslContext = sslContext;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #privateKey()
         */
        public BUILDER clearPrivateKey() {
            this.privateKey = null;
            return self();
        }

        /**
         * Private key to use. For server side TLS, this is required.
         * For client side TLS, this is optional (used when mutual TLS is enabled).
         *
         * @param privateKey private key to use
         * @return updated builder instance
         * @see #privateKey()
         */
        public BUILDER privateKey(PrivateKey privateKey) {
            Objects.requireNonNull(privateKey);
            this.privateKey = privateKey;
            return self();
        }

        /**
         * Private key to use. For server side TLS, this is required.
         * For client side TLS, this is optional (used when mutual TLS is enabled).
         *
         * @param privateKeyConfig private key to use
         * @return updated builder instance
         * @see #privateKey()
         */
        public BUILDER privateKey(Keys privateKeyConfig) {
            Objects.requireNonNull(privateKeyConfig);
            this.privateKey = TlsConfigBlueprint.createPrivateKey(privateKeyConfig).orElse(null);
            return self();
        }

        /**
         * Private key to use. For server side TLS, this is required.
         * For client side TLS, this is optional (used when mutual TLS is enabled).
         *
         * @param consumer private key to use
         * @return updated builder instance
         * @see #privateKey()
         */
        public BUILDER privateKey(Consumer<Keys.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Keys.builder();
            consumer.accept(builder);
            this.privateKey(builder.build());
            return self();
        }

        /**
         * Certificate chain of the private key.
         *
         * @param privateKeyCertChain private key certificate chain, only used when private key is configured
         * @return updated builder instance
         * @see #privateKeyCertChain()
         */
        public BUILDER privateKeyCertChain(List<? extends X509Certificate> privateKeyCertChain) {
            Objects.requireNonNull(privateKeyCertChain);
            isPrivateKeyCertChainMutated = true;
            this.privateKeyCertChain.clear();
            this.privateKeyCertChain.addAll(privateKeyCertChain);
            return self();
        }

        /**
         * Certificate chain of the private key.
         *
         * @param privateKeyCertChain private key certificate chain, only used when private key is configured
         * @return updated builder instance
         * @see #privateKeyCertChain()
         */
        public BUILDER addPrivateKeyCertChain(List<? extends X509Certificate> privateKeyCertChain) {
            Objects.requireNonNull(privateKeyCertChain);
            isPrivateKeyCertChainMutated = true;
            this.privateKeyCertChain.addAll(privateKeyCertChain);
            return self();
        }

        /**
         * Certificate chain of the private key.
         *
         * @param privateKeyCertChainConfig private key certificate chain, only used when private key is configured
         * @return updated builder instance
         * @see #privateKeyCertChain()
         */
        public BUILDER privateKeyCertChain(Keys privateKeyCertChainConfig) {
            Objects.requireNonNull(privateKeyCertChainConfig);
            this.privateKeyCertChain.clear();
            this.privateKeyCertChain.addAll(TlsConfigBlueprint.createPrivateKeyCertChain(privateKeyCertChainConfig));
            return self();
        }

        /**
         * Certificate chain of the private key.
         *
         * @param privateKeyCertChain private key certificate chain, only used when private key is configured
         * @return updated builder instance
         * @see #privateKeyCertChain()
         */
        public BUILDER addPrivateKeyCertChain(X509Certificate privateKeyCertChain) {
            Objects.requireNonNull(privateKeyCertChain);
            this.privateKeyCertChain.add(privateKeyCertChain);
            isPrivateKeyCertChainMutated = true;
            return self();
        }

        /**
         * Certificate chain of the private key.
         *
         * @param consumer private key certificate chain, only used when private key is configured
         * @return updated builder instance
         * @see #privateKeyCertChain()
         */
        public BUILDER privateKeyCertChain(Consumer<Keys.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Keys.builder();
            consumer.accept(builder);
            this.privateKeyCertChain(builder.build());
            return self();
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @param trust certificates to be trusted
         * @return updated builder instance
         * @see #trust()
         */
        public BUILDER trust(List<? extends X509Certificate> trust) {
            Objects.requireNonNull(trust);
            isTrustMutated = true;
            this.trust.clear();
            this.trust.addAll(trust);
            return self();
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @param trust certificates to be trusted
         * @return updated builder instance
         * @see #trust()
         */
        public BUILDER addTrust(List<? extends X509Certificate> trust) {
            Objects.requireNonNull(trust);
            isTrustMutated = true;
            this.trust.addAll(trust);
            return self();
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @param trustConfig certificates to be trusted
         * @return updated builder instance
         * @see #trust()
         */
        public BUILDER trust(Keys trustConfig) {
            Objects.requireNonNull(trustConfig);
            this.trust.clear();
            this.trust.addAll(TlsConfigBlueprint.createTrust(trustConfig));
            return self();
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @param trust certificates to be trusted
         * @return updated builder instance
         * @see #trust()
         */
        public BUILDER addTrust(X509Certificate trust) {
            Objects.requireNonNull(trust);
            this.trust.add(trust);
            isTrustMutated = true;
            return self();
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @param consumer certificates to be trusted
         * @return updated builder instance
         * @see #trust()
         */
        public BUILDER trust(Consumer<Keys.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Keys.builder();
            consumer.accept(builder);
            this.trust(builder.build());
            return self();
        }

        /**
         * The Tls manager. If one is not explicitly defined in the config then a default manager will be created.
         *
         * @param manager the tls manager of the tls instance
         * @return updated builder instance
         * @see ConfiguredTlsManager
         * @see #manager()
         */
        public BUILDER manager(TlsManager manager) {
            Objects.requireNonNull(manager);
            this.manager = manager;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #secureRandom()
         */
        public BUILDER clearSecureRandom() {
            this.secureRandom = null;
            return self();
        }

        /**
         * Explicit secure random to use.
         *
         * @param secureRandom secure random to use
         * @return updated builder instance
         * @see #secureRandom()
         */
        public BUILDER secureRandom(SecureRandom secureRandom) {
            Objects.requireNonNull(secureRandom);
            this.secureRandom = secureRandom;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #sslParameters()
         */
        public BUILDER clearSslParameters() {
            this.sslParameters = null;
            return self();
        }

        /**
         * Configure SSL parameters.
         * This will always have a value, as we compute ssl parameters in a builder interceptor from configured options.
         *
         * @param sslParameters SSL parameters to use
         * @return updated builder instance
         * @see #sslParameters()
         */
        public BUILDER sslParameters(SSLParameters sslParameters) {
            Objects.requireNonNull(sslParameters);
            this.sslParameters = sslParameters;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #secureRandomProvider()
         */
        public BUILDER clearSecureRandomProvider() {
            this.secureRandomProvider = null;
            return self();
        }

        /**
         * Provider to use when creating a new secure random.
         * When defined, {@link #secureRandomAlgorithm()} must be defined as well.
         *
         * @param secureRandomProvider provider to use, by default no provider is specified
         * @return updated builder instance
         * @see #secureRandomProvider()
         */
        public BUILDER secureRandomProvider(String secureRandomProvider) {
            Objects.requireNonNull(secureRandomProvider);
            this.secureRandomProvider = secureRandomProvider;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #secureRandomAlgorithm()
         */
        public BUILDER clearSecureRandomAlgorithm() {
            this.secureRandomAlgorithm = null;
            return self();
        }

        /**
         * Algorithm to use when creating a new secure random.
         *
         * @param secureRandomAlgorithm algorithm to use, by default uses {@link java.security.SecureRandom} constructor
         * @return updated builder instance
         * @see #secureRandomAlgorithm()
         */
        public BUILDER secureRandomAlgorithm(String secureRandomAlgorithm) {
            Objects.requireNonNull(secureRandomAlgorithm);
            this.secureRandomAlgorithm = secureRandomAlgorithm;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #keyManagerFactoryAlgorithm()
         */
        public BUILDER clearKeyManagerFactoryAlgorithm() {
            this.keyManagerFactoryAlgorithm = null;
            return self();
        }

        /**
         * Algorithm of the key manager factory used when private key is defined.
         * Defaults to {@link javax.net.ssl.KeyManagerFactory#getDefaultAlgorithm()}.
         *
         * @param keyManagerFactoryAlgorithm algorithm to use
         * @return updated builder instance
         * @see #keyManagerFactoryAlgorithm()
         */
        public BUILDER keyManagerFactoryAlgorithm(String keyManagerFactoryAlgorithm) {
            Objects.requireNonNull(keyManagerFactoryAlgorithm);
            this.keyManagerFactoryAlgorithm = keyManagerFactoryAlgorithm;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #keyManagerFactoryProvider()
         */
        public BUILDER clearKeyManagerFactoryProvider() {
            this.keyManagerFactoryProvider = null;
            return self();
        }

        /**
         * Key manager factory provider.
         *
         * @param keyManagerFactoryProvider provider to use
         * @return updated builder instance
         * @see #keyManagerFactoryProvider()
         */
        public BUILDER keyManagerFactoryProvider(String keyManagerFactoryProvider) {
            Objects.requireNonNull(keyManagerFactoryProvider);
            this.keyManagerFactoryProvider = keyManagerFactoryProvider;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #trustManagerFactoryAlgorithm()
         */
        public BUILDER clearTrustManagerFactoryAlgorithm() {
            this.trustManagerFactoryAlgorithm = null;
            return self();
        }

        /**
         * Trust manager factory algorithm.
         *
         * @param trustManagerFactoryAlgorithm algorithm to use
         * @return updated builder instance
         * @see #trustManagerFactoryAlgorithm()
         */
        public BUILDER trustManagerFactoryAlgorithm(String trustManagerFactoryAlgorithm) {
            Objects.requireNonNull(trustManagerFactoryAlgorithm);
            this.trustManagerFactoryAlgorithm = trustManagerFactoryAlgorithm;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #trustManagerFactoryProvider()
         */
        public BUILDER clearTrustManagerFactoryProvider() {
            this.trustManagerFactoryProvider = null;
            return self();
        }

        /**
         * Trust manager factory provider to use.
         *
         * @param trustManagerFactoryProvider provider to use
         * @return updated builder instance
         * @see #trustManagerFactoryProvider()
         */
        public BUILDER trustManagerFactoryProvider(String trustManagerFactoryProvider) {
            Objects.requireNonNull(trustManagerFactoryProvider);
            this.trustManagerFactoryProvider = trustManagerFactoryProvider;
            return self();
        }

        /**
         * Configure list of supported application protocols (such as {@code h2}) for application layer protocol negotiation (ALPN).
         *
         * @param applicationProtocols application protocols
         * @return updated builder instance
         * @see #applicationProtocols()
         */
        public BUILDER applicationProtocols(List<? extends String> applicationProtocols) {
            Objects.requireNonNull(applicationProtocols);
            isApplicationProtocolsMutated = true;
            this.applicationProtocols.clear();
            this.applicationProtocols.addAll(applicationProtocols);
            return self();
        }

        /**
         * Configure list of supported application protocols (such as {@code h2}) for application layer protocol negotiation (ALPN).
         *
         * @param applicationProtocols application protocols
         * @return updated builder instance
         * @see #applicationProtocols()
         */
        public BUILDER addApplicationProtocols(List<? extends String> applicationProtocols) {
            Objects.requireNonNull(applicationProtocols);
            isApplicationProtocolsMutated = true;
            this.applicationProtocols.addAll(applicationProtocols);
            return self();
        }

        /**
         * Configure list of supported application protocols (such as {@code h2}) for application layer protocol negotiation (ALPN).
         *
         * @param applicationProtocol application protocols
         * @return updated builder instance
         * @see #applicationProtocols()
         */
        public BUILDER addApplicationProtocol(String applicationProtocol) {
            Objects.requireNonNull(applicationProtocol);
            this.applicationProtocols.add(applicationProtocol);
            isApplicationProtocolsMutated = true;
            return self();
        }

        /**
         * Identification algorithm for SSL endpoints.
         *
         * @param endpointIdentificationAlgorithm configure endpoint identification algorithm, or set to {@code NONE}
         *                                        to disable endpoint identification (equivalent to hostname verification).
         *                                        Defaults to {@value Tls#ENDPOINT_IDENTIFICATION_HTTPS}
         * @return updated builder instance
         * @see #endpointIdentificationAlgorithm()
         */
        public BUILDER endpointIdentificationAlgorithm(String endpointIdentificationAlgorithm) {
            Objects.requireNonNull(endpointIdentificationAlgorithm);
            this.endpointIdentificationAlgorithm = endpointIdentificationAlgorithm;
            return self();
        }

        /**
         * Flag indicating whether Tls is enabled.
         *
         * @param enabled enabled flag
         * @return updated builder instance
         * @see #enabled()
         */
        public BUILDER enabled(boolean enabled) {
            this.enabled = enabled;
            return self();
        }

        /**
         * Trust any certificate provided by the other side of communication.
         * <p>
         * <b>This is a dangerous setting: </b> if set to {@code true}, any certificate will be accepted, throwing away
         * most of the security advantages of TLS. <b>NEVER</b> do this in production.
         *
         * @param trustAll whether to trust all certificates, do not use in production
         * @return updated builder instance
         * @see #trustAll()
         */
        public BUILDER trustAll(boolean trustAll) {
            this.trustAll = trustAll;
            return self();
        }

        /**
         * Configure requirement for mutual TLS.
         *
         * @param clientAuth what type of mutual TLS to use, defaults to {@link TlsClientAuth#NONE}
         * @return updated builder instance
         * @see #clientAuth()
         */
        public BUILDER clientAuth(TlsClientAuth clientAuth) {
            Objects.requireNonNull(clientAuth);
            this.clientAuth = clientAuth;
            return self();
        }

        /**
         * Configure the protocol used to obtain an instance of {@link javax.net.ssl.SSLContext}.
         *
         * @param protocol protocol to use, defaults to {@value DEFAULT_PROTOCOL}
         * @return updated builder instance
         * @see #protocol()
         */
        public BUILDER protocol(String protocol) {
            Objects.requireNonNull(protocol);
            this.protocol = protocol;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #provider()
         */
        public BUILDER clearProvider() {
            this.provider = null;
            return self();
        }

        /**
         * Use explicit provider to obtain an instance of {@link javax.net.ssl.SSLContext}.
         *
         * @param provider provider to use, defaults to none (only {@link #protocol()} is used by default)
         * @return updated builder instance
         * @see #provider()
         */
        public BUILDER provider(String provider) {
            Objects.requireNonNull(provider);
            this.provider = provider;
            return self();
        }

        /**
         * Enabled cipher suites for TLS communication.
         *
         * @param enabledCipherSuites cipher suits to enable, by default (or if list is empty), all available cipher suites
         *                            are enabled
         * @return updated builder instance
         * @see #enabledCipherSuites()
         */
        public BUILDER enabledCipherSuites(List<? extends String> enabledCipherSuites) {
            Objects.requireNonNull(enabledCipherSuites);
            isEnabledCipherSuitesMutated = true;
            this.enabledCipherSuites.clear();
            this.enabledCipherSuites.addAll(enabledCipherSuites);
            return self();
        }

        /**
         * Enabled cipher suites for TLS communication.
         *
         * @param enabledCipherSuites cipher suits to enable, by default (or if list is empty), all available cipher suites
         *                            are enabled
         * @return updated builder instance
         * @see #enabledCipherSuites()
         */
        public BUILDER addEnabledCipherSuites(List<? extends String> enabledCipherSuites) {
            Objects.requireNonNull(enabledCipherSuites);
            isEnabledCipherSuitesMutated = true;
            this.enabledCipherSuites.addAll(enabledCipherSuites);
            return self();
        }

        /**
         * Enabled cipher suites for TLS communication.
         *
         * @param enabledCipherSuite cipher suits to enable, by default (or if list is empty), all available cipher suites
         *                           are enabled
         * @return updated builder instance
         * @see #enabledCipherSuites()
         */
        public BUILDER addEnabledCipherSuite(String enabledCipherSuite) {
            Objects.requireNonNull(enabledCipherSuite);
            this.enabledCipherSuites.add(enabledCipherSuite);
            isEnabledCipherSuitesMutated = true;
            return self();
        }

        /**
         * Enabled protocols for TLS communication.
         * Example of valid values for {@code TLS} protocol: {@code TLSv1.3}, {@code TLSv1.2}
         *
         * @param enabledProtocols protocols to enable, by default (or if list is empty), all available protocols are enabled
         * @return updated builder instance
         * @see #enabledProtocols()
         */
        public BUILDER enabledProtocols(List<? extends String> enabledProtocols) {
            Objects.requireNonNull(enabledProtocols);
            isEnabledProtocolsMutated = true;
            this.enabledProtocols.clear();
            this.enabledProtocols.addAll(enabledProtocols);
            return self();
        }

        /**
         * Enabled protocols for TLS communication.
         * Example of valid values for {@code TLS} protocol: {@code TLSv1.3}, {@code TLSv1.2}
         *
         * @param enabledProtocols protocols to enable, by default (or if list is empty), all available protocols are enabled
         * @return updated builder instance
         * @see #enabledProtocols()
         */
        public BUILDER addEnabledProtocols(List<? extends String> enabledProtocols) {
            Objects.requireNonNull(enabledProtocols);
            isEnabledProtocolsMutated = true;
            this.enabledProtocols.addAll(enabledProtocols);
            return self();
        }

        /**
         * Enabled protocols for TLS communication.
         * Example of valid values for {@code TLS} protocol: {@code TLSv1.3}, {@code TLSv1.2}
         *
         * @param enabledProtocol protocols to enable, by default (or if list is empty), all available protocols are enabled
         * @return updated builder instance
         * @see #enabledProtocols()
         */
        public BUILDER addEnabledProtocol(String enabledProtocol) {
            Objects.requireNonNull(enabledProtocol);
            this.enabledProtocols.add(enabledProtocol);
            isEnabledProtocolsMutated = true;
            return self();
        }

        /**
         * SSL session cache size.
         *
         * @param sessionCacheSize session cache size, defaults to {@value DEFAULT_SESSION_CACHE_SIZE}.
         * @return updated builder instance
         * @see #sessionCacheSize()
         */
        public BUILDER sessionCacheSize(int sessionCacheSize) {
            this.sessionCacheSize = sessionCacheSize;
            return self();
        }

        /**
         * SSL session timeout.
         *
         * @param sessionTimeout session timeout, defaults to {@value DEFAULT_SESSION_TIMEOUT}.
         * @return updated builder instance
         * @see #sessionTimeout()
         */
        public BUILDER sessionTimeout(Duration sessionTimeout) {
            Objects.requireNonNull(sessionTimeout);
            this.sessionTimeout = sessionTimeout;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #internalKeystoreType()
         */
        public BUILDER clearInternalKeystoreType() {
            this.internalKeystoreType = null;
            return self();
        }

        /**
         * Type of the key stores used internally to create a key and trust manager factories.
         *
         * @param internalKeystoreType keystore type, defaults to {@link java.security.KeyStore#getDefaultType()}
         * @return updated builder instance
         * @see #internalKeystoreType()
         */
        public BUILDER internalKeystoreType(String internalKeystoreType) {
            Objects.requireNonNull(internalKeystoreType);
            this.internalKeystoreType = internalKeystoreType;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #internalKeystoreProvider()
         */
        public BUILDER clearInternalKeystoreProvider() {
            this.internalKeystoreProvider = null;
            return self();
        }

        /**
         * Provider of the key stores used internally to create a key and trust manager factories.
         *
         * @param internalKeystoreProvider keystore provider, if not defined, provider is not specified
         * @return updated builder instance
         * @see #internalKeystoreProvider()
         */
        public BUILDER internalKeystoreProvider(String internalKeystoreProvider) {
            Objects.requireNonNull(internalKeystoreProvider);
            this.internalKeystoreProvider = internalKeystoreProvider;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #revocation()
         */
        public BUILDER clearRevocation() {
            this.revocation = null;
            return self();
        }

        /**
         * Certificate revocation check configuration.
         *
         * @param revocation certificate revocation configuration
         * @return updated builder instance
         * @see #revocation()
         */
        public BUILDER revocation(RevocationConfig revocation) {
            Objects.requireNonNull(revocation);
            this.revocation = revocation;
            return self();
        }

        /**
         * Certificate revocation check configuration.
         *
         * @param consumer certificate revocation configuration
         * @return updated builder instance
         * @see #revocation()
         */
        public BUILDER revocation(Consumer<RevocationConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = RevocationConfig.builder();
            consumer.accept(builder);
            this.revocation(builder.build());
            return self();
        }

        /**
         * Provide a fully configured {@link javax.net.ssl.SSLContext}. If defined, context related configuration
         * is ignored.
         *
         * @return the ssl context
         */
        public Optional<SSLContext> sslContext() {
            return Optional.ofNullable(sslContext);
        }

        /**
         * Private key to use. For server side TLS, this is required.
         * For client side TLS, this is optional (used when mutual TLS is enabled).
         *
         * @return the private key
         */
        public Optional<PrivateKey> privateKey() {
            return Optional.ofNullable(privateKey);
        }

        /**
         * Certificate chain of the private key.
         *
         * @return the private key cert chain
         */
        public List<X509Certificate> privateKeyCertChain() {
            return privateKeyCertChain;
        }

        /**
         * List of certificates that form the trust manager.
         *
         * @return the trust
         */
        public List<X509Certificate> trust() {
            return trust;
        }

        /**
         * The Tls manager. If one is not explicitly defined in the config then a default manager will be created.
         *
         * @return the manager
         * @see ConfiguredTlsManager
         * @see #manager()
         */
        public Optional<TlsManager> manager() {
            return Optional.ofNullable(manager);
        }

        /**
         * Explicit secure random to use.
         *
         * @return the secure random
         */
        public Optional<SecureRandom> secureRandom() {
            return Optional.ofNullable(secureRandom);
        }

        /**
         * Configure SSL parameters.
         * This will always have a value, as we compute ssl parameters in a builder interceptor from configured options.
         *
         * @return the ssl parameters
         */
        public Optional<SSLParameters> sslParameters() {
            return Optional.ofNullable(sslParameters);
        }

        /**
         * Provider to use when creating a new secure random.
         * When defined, {@link #secureRandomAlgorithm()} must be defined as well.
         *
         * @return the secure random provider
         */
        public Optional<String> secureRandomProvider() {
            return Optional.ofNullable(secureRandomProvider);
        }

        /**
         * Algorithm to use when creating a new secure random.
         *
         * @return the secure random algorithm
         */
        public Optional<String> secureRandomAlgorithm() {
            return Optional.ofNullable(secureRandomAlgorithm);
        }

        /**
         * Algorithm of the key manager factory used when private key is defined.
         * Defaults to {@link javax.net.ssl.KeyManagerFactory#getDefaultAlgorithm()}.
         *
         * @return the key manager factory algorithm
         */
        public Optional<String> keyManagerFactoryAlgorithm() {
            return Optional.ofNullable(keyManagerFactoryAlgorithm);
        }

        /**
         * Key manager factory provider.
         *
         * @return the key manager factory provider
         */
        public Optional<String> keyManagerFactoryProvider() {
            return Optional.ofNullable(keyManagerFactoryProvider);
        }

        /**
         * Trust manager factory algorithm.
         *
         * @return the trust manager factory algorithm
         */
        public Optional<String> trustManagerFactoryAlgorithm() {
            return Optional.ofNullable(trustManagerFactoryAlgorithm);
        }

        /**
         * Trust manager factory provider to use.
         *
         * @return the trust manager factory provider
         */
        public Optional<String> trustManagerFactoryProvider() {
            return Optional.ofNullable(trustManagerFactoryProvider);
        }

        /**
         * Configure list of supported application protocols (such as {@code h2}) for application layer protocol negotiation (ALPN).
         *
         * @return the application protocols
         */
        public List<String> applicationProtocols() {
            return applicationProtocols;
        }

        /**
         * Identification algorithm for SSL endpoints.
         *
         * @return the endpoint identification algorithm
         */
        public String endpointIdentificationAlgorithm() {
            return endpointIdentificationAlgorithm;
        }

        /**
         * Flag indicating whether Tls is enabled.
         *
         * @return the enabled
         */
        public boolean enabled() {
            return enabled;
        }

        /**
         * Trust any certificate provided by the other side of communication.
         * <p>
         * <b>This is a dangerous setting: </b> if set to {@code true}, any certificate will be accepted, throwing away
         * most of the security advantages of TLS. <b>NEVER</b> do this in production.
         *
         * @return the trust all
         */
        public boolean trustAll() {
            return trustAll;
        }

        /**
         * Configure requirement for mutual TLS.
         *
         * @return the client auth
         */
        public TlsClientAuth clientAuth() {
            return clientAuth;
        }

        /**
         * Configure the protocol used to obtain an instance of {@link javax.net.ssl.SSLContext}.
         *
         * @return the protocol
         */
        public String protocol() {
            return protocol;
        }

        /**
         * Use explicit provider to obtain an instance of {@link javax.net.ssl.SSLContext}.
         *
         * @return the provider
         */
        public Optional<String> provider() {
            return Optional.ofNullable(provider);
        }

        /**
         * Enabled cipher suites for TLS communication.
         *
         * @return the enabled cipher suites
         */
        public List<String> enabledCipherSuites() {
            return enabledCipherSuites;
        }

        /**
         * Enabled protocols for TLS communication.
         * Example of valid values for {@code TLS} protocol: {@code TLSv1.3}, {@code TLSv1.2}
         *
         * @return the enabled protocols
         */
        public List<String> enabledProtocols() {
            return enabledProtocols;
        }

        /**
         * SSL session cache size.
         *
         * @return the session cache size
         */
        public int sessionCacheSize() {
            return sessionCacheSize;
        }

        /**
         * SSL session timeout.
         *
         * @return the session timeout
         */
        public Duration sessionTimeout() {
            return sessionTimeout;
        }

        /**
         * Type of the key stores used internally to create a key and trust manager factories.
         *
         * @return the internal keystore type
         */
        public Optional<String> internalKeystoreType() {
            return Optional.ofNullable(internalKeystoreType);
        }

        /**
         * Provider of the key stores used internally to create a key and trust manager factories.
         *
         * @return the internal keystore provider
         */
        public Optional<String> internalKeystoreProvider() {
            return Optional.ofNullable(internalKeystoreProvider);
        }

        /**
         * Certificate revocation check configuration.
         *
         * @return the revocation
         */
        public Optional<RevocationConfig> revocation() {
            return Optional.ofNullable(revocation);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "TlsConfigBuilder{"
                    + "sslContext=" + sslContext + ","
                    + "privateKey=" + privateKey + ","
                    + "privateKeyCertChain=" + privateKeyCertChain + ","
                    + "trust=" + trust + ","
                    + "manager=" + manager + ","
                    + "secureRandom=" + secureRandom + ","
                    + "sslParameters=" + sslParameters + ","
                    + "secureRandomProvider=" + secureRandomProvider + ","
                    + "secureRandomAlgorithm=" + secureRandomAlgorithm + ","
                    + "keyManagerFactoryAlgorithm=" + keyManagerFactoryAlgorithm + ","
                    + "keyManagerFactoryProvider=" + keyManagerFactoryProvider + ","
                    + "trustManagerFactoryAlgorithm=" + trustManagerFactoryAlgorithm + ","
                    + "trustManagerFactoryProvider=" + trustManagerFactoryProvider + ","
                    + "applicationProtocols=" + applicationProtocols + ","
                    + "endpointIdentificationAlgorithm=" + endpointIdentificationAlgorithm + ","
                    + "enabled=" + enabled + ","
                    + "trustAll=" + trustAll + ","
                    + "clientAuth=" + clientAuth + ","
                    + "protocol=" + protocol + ","
                    + "provider=" + provider + ","
                    + "enabledCipherSuites=" + enabledCipherSuites + ","
                    + "enabledProtocols=" + enabledProtocols + ","
                    + "sessionCacheSize=" + sessionCacheSize + ","
                    + "sessionTimeout=" + sessionTimeout + ","
                    + "internalKeystoreType=" + internalKeystoreType + ","
                    + "internalKeystoreProvider=" + internalKeystoreProvider + ","
                    + "revocation=" + revocation
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        @SuppressWarnings("unchecked")
        protected void preBuildPrototype() {
            var config = this.config == null ? Config.empty() : this.config;
            {
                var serviceLoader = HelidonServiceLoader.create(ServiceLoader.load(TlsManagerProvider.class));
                discoverService(config, "manager", serviceLoader, TlsManagerProvider.class, TlsManager.class, managerDiscoverServices, Optional.ofNullable(manager)).ifPresent(this::manager);
            }
            new TlsConfigDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (manager == null) {
                collector.fatal(getClass(), "Property \"manager\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Provide a fully configured {@link javax.net.ssl.SSLContext}. If defined, context related configuration
         * is ignored.
         *
         * @param sslContext SSL context to use
         * @return updated builder instance
         * @see #sslContext()
         */
        BUILDER sslContext(Optional<? extends SSLContext> sslContext) {
            Objects.requireNonNull(sslContext);
            this.sslContext = sslContext.map(javax.net.ssl.SSLContext.class::cast).orElse(this.sslContext);
            return self();
        }

        /**
         * Private key to use. For server side TLS, this is required.
         * For client side TLS, this is optional (used when mutual TLS is enabled).
         *
         * @param privateKey private key to use
         * @return updated builder instance
         * @see #privateKey()
         */
        BUILDER privateKey(Optional<? extends PrivateKey> privateKey) {
            Objects.requireNonNull(privateKey);
            this.privateKey = privateKey.map(java.security.PrivateKey.class::cast).orElse(this.privateKey);
            return self();
        }

        /**
         * Explicit secure random to use.
         *
         * @param secureRandom secure random to use
         * @return updated builder instance
         * @see #secureRandom()
         */
        BUILDER secureRandom(Optional<? extends SecureRandom> secureRandom) {
            Objects.requireNonNull(secureRandom);
            this.secureRandom = secureRandom.map(java.security.SecureRandom.class::cast).orElse(this.secureRandom);
            return self();
        }

        /**
         * Configure SSL parameters.
         * This will always have a value, as we compute ssl parameters in a builder interceptor from configured options.
         *
         * @param sslParameters SSL parameters to use
         * @return updated builder instance
         * @see #sslParameters()
         */
        BUILDER sslParameters(Optional<? extends SSLParameters> sslParameters) {
            Objects.requireNonNull(sslParameters);
            this.sslParameters = sslParameters.map(javax.net.ssl.SSLParameters.class::cast).orElse(this.sslParameters);
            return self();
        }

        /**
         * Provider to use when creating a new secure random.
         * When defined, {@link #secureRandomAlgorithm()} must be defined as well.
         *
         * @param secureRandomProvider provider to use, by default no provider is specified
         * @return updated builder instance
         * @see #secureRandomProvider()
         */
        BUILDER secureRandomProvider(Optional<String> secureRandomProvider) {
            Objects.requireNonNull(secureRandomProvider);
            this.secureRandomProvider = secureRandomProvider.map(java.lang.String.class::cast).orElse(this.secureRandomProvider);
            return self();
        }

        /**
         * Algorithm to use when creating a new secure random.
         *
         * @param secureRandomAlgorithm algorithm to use, by default uses {@link java.security.SecureRandom} constructor
         * @return updated builder instance
         * @see #secureRandomAlgorithm()
         */
        BUILDER secureRandomAlgorithm(Optional<String> secureRandomAlgorithm) {
            Objects.requireNonNull(secureRandomAlgorithm);
            this.secureRandomAlgorithm = secureRandomAlgorithm.map(java.lang.String.class::cast).orElse(this.secureRandomAlgorithm);
            return self();
        }

        /**
         * Algorithm of the key manager factory used when private key is defined.
         * Defaults to {@link javax.net.ssl.KeyManagerFactory#getDefaultAlgorithm()}.
         *
         * @param keyManagerFactoryAlgorithm algorithm to use
         * @return updated builder instance
         * @see #keyManagerFactoryAlgorithm()
         */
        BUILDER keyManagerFactoryAlgorithm(Optional<String> keyManagerFactoryAlgorithm) {
            Objects.requireNonNull(keyManagerFactoryAlgorithm);
            this.keyManagerFactoryAlgorithm = keyManagerFactoryAlgorithm.map(java.lang.String.class::cast).orElse(this.keyManagerFactoryAlgorithm);
            return self();
        }

        /**
         * Key manager factory provider.
         *
         * @param keyManagerFactoryProvider provider to use
         * @return updated builder instance
         * @see #keyManagerFactoryProvider()
         */
        BUILDER keyManagerFactoryProvider(Optional<String> keyManagerFactoryProvider) {
            Objects.requireNonNull(keyManagerFactoryProvider);
            this.keyManagerFactoryProvider = keyManagerFactoryProvider.map(java.lang.String.class::cast).orElse(this.keyManagerFactoryProvider);
            return self();
        }

        /**
         * Trust manager factory algorithm.
         *
         * @param trustManagerFactoryAlgorithm algorithm to use
         * @return updated builder instance
         * @see #trustManagerFactoryAlgorithm()
         */
        BUILDER trustManagerFactoryAlgorithm(Optional<String> trustManagerFactoryAlgorithm) {
            Objects.requireNonNull(trustManagerFactoryAlgorithm);
            this.trustManagerFactoryAlgorithm = trustManagerFactoryAlgorithm.map(java.lang.String.class::cast).orElse(this.trustManagerFactoryAlgorithm);
            return self();
        }

        /**
         * Trust manager factory provider to use.
         *
         * @param trustManagerFactoryProvider provider to use
         * @return updated builder instance
         * @see #trustManagerFactoryProvider()
         */
        BUILDER trustManagerFactoryProvider(Optional<String> trustManagerFactoryProvider) {
            Objects.requireNonNull(trustManagerFactoryProvider);
            this.trustManagerFactoryProvider = trustManagerFactoryProvider.map(java.lang.String.class::cast).orElse(this.trustManagerFactoryProvider);
            return self();
        }

        /**
         * Use explicit provider to obtain an instance of {@link javax.net.ssl.SSLContext}.
         *
         * @param provider provider to use, defaults to none (only {@link #protocol()} is used by default)
         * @return updated builder instance
         * @see #provider()
         */
        BUILDER provider(Optional<String> provider) {
            Objects.requireNonNull(provider);
            this.provider = provider.map(java.lang.String.class::cast).orElse(this.provider);
            return self();
        }

        /**
         * Type of the key stores used internally to create a key and trust manager factories.
         *
         * @param internalKeystoreType keystore type, defaults to {@link java.security.KeyStore#getDefaultType()}
         * @return updated builder instance
         * @see #internalKeystoreType()
         */
        BUILDER internalKeystoreType(Optional<String> internalKeystoreType) {
            Objects.requireNonNull(internalKeystoreType);
            this.internalKeystoreType = internalKeystoreType.map(java.lang.String.class::cast).orElse(this.internalKeystoreType);
            return self();
        }

        /**
         * Provider of the key stores used internally to create a key and trust manager factories.
         *
         * @param internalKeystoreProvider keystore provider, if not defined, provider is not specified
         * @return updated builder instance
         * @see #internalKeystoreProvider()
         */
        BUILDER internalKeystoreProvider(Optional<String> internalKeystoreProvider) {
            Objects.requireNonNull(internalKeystoreProvider);
            this.internalKeystoreProvider = internalKeystoreProvider.map(java.lang.String.class::cast).orElse(this.internalKeystoreProvider);
            return self();
        }

        /**
         * Certificate revocation check configuration.
         *
         * @param revocation certificate revocation configuration
         * @return updated builder instance
         * @see #revocation()
         */
        BUILDER revocation(Optional<? extends RevocationConfig> revocation) {
            Objects.requireNonNull(revocation);
            this.revocation = revocation.map(RevocationConfig.class::cast).orElse(this.revocation);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class TlsConfigImpl implements TlsConfig, Supplier<Tls> {

            private final boolean enabled;
            private final boolean trustAll;
            private final Duration sessionTimeout;
            private final int sessionCacheSize;
            private final List<String> applicationProtocols;
            private final List<String> enabledCipherSuites;
            private final List<String> enabledProtocols;
            private final List<X509Certificate> privateKeyCertChain;
            private final List<X509Certificate> trust;
            private final Optional<RevocationConfig> revocation;
            private final Optional<String> internalKeystoreProvider;
            private final Optional<String> internalKeystoreType;
            private final Optional<String> keyManagerFactoryAlgorithm;
            private final Optional<String> keyManagerFactoryProvider;
            private final Optional<String> provider;
            private final Optional<String> secureRandomAlgorithm;
            private final Optional<String> secureRandomProvider;
            private final Optional<String> trustManagerFactoryAlgorithm;
            private final Optional<String> trustManagerFactoryProvider;
            private final Optional<PrivateKey> privateKey;
            private final Optional<SecureRandom> secureRandom;
            private final Optional<SSLContext> sslContext;
            private final Optional<SSLParameters> sslParameters;
            private final String endpointIdentificationAlgorithm;
            private final String protocol;
            private final TlsClientAuth clientAuth;
            private final TlsManager manager;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected TlsConfigImpl(TlsConfig.BuilderBase<?, ?> builder) {
                this.sslContext = builder.sslContext();
                this.privateKey = builder.privateKey();
                this.privateKeyCertChain = List.copyOf(builder.privateKeyCertChain());
                this.trust = List.copyOf(builder.trust());
                this.manager = builder.manager().get();
                this.secureRandom = builder.secureRandom();
                this.sslParameters = builder.sslParameters();
                this.secureRandomProvider = builder.secureRandomProvider();
                this.secureRandomAlgorithm = builder.secureRandomAlgorithm();
                this.keyManagerFactoryAlgorithm = builder.keyManagerFactoryAlgorithm();
                this.keyManagerFactoryProvider = builder.keyManagerFactoryProvider();
                this.trustManagerFactoryAlgorithm = builder.trustManagerFactoryAlgorithm();
                this.trustManagerFactoryProvider = builder.trustManagerFactoryProvider();
                this.applicationProtocols = List.copyOf(builder.applicationProtocols());
                this.endpointIdentificationAlgorithm = builder.endpointIdentificationAlgorithm();
                this.enabled = builder.enabled();
                this.trustAll = builder.trustAll();
                this.clientAuth = builder.clientAuth();
                this.protocol = builder.protocol();
                this.provider = builder.provider();
                this.enabledCipherSuites = List.copyOf(builder.enabledCipherSuites());
                this.enabledProtocols = List.copyOf(builder.enabledProtocols());
                this.sessionCacheSize = builder.sessionCacheSize();
                this.sessionTimeout = builder.sessionTimeout();
                this.internalKeystoreType = builder.internalKeystoreType();
                this.internalKeystoreProvider = builder.internalKeystoreProvider();
                this.revocation = builder.revocation();
            }

            @Override
            public Tls build() {
                return Tls.create(this);
            }

            @Override
            public Tls get() {
                return build();
            }

            @Override
            public Optional<SSLContext> sslContext() {
                return sslContext;
            }

            @Override
            public Optional<PrivateKey> privateKey() {
                return privateKey;
            }

            @Override
            public List<X509Certificate> privateKeyCertChain() {
                return privateKeyCertChain;
            }

            @Override
            public List<X509Certificate> trust() {
                return trust;
            }

            @Override
            public TlsManager manager() {
                return manager;
            }

            @Override
            public Optional<SecureRandom> secureRandom() {
                return secureRandom;
            }

            @Override
            public Optional<SSLParameters> sslParameters() {
                return sslParameters;
            }

            @Override
            public Optional<String> secureRandomProvider() {
                return secureRandomProvider;
            }

            @Override
            public Optional<String> secureRandomAlgorithm() {
                return secureRandomAlgorithm;
            }

            @Override
            public Optional<String> keyManagerFactoryAlgorithm() {
                return keyManagerFactoryAlgorithm;
            }

            @Override
            public Optional<String> keyManagerFactoryProvider() {
                return keyManagerFactoryProvider;
            }

            @Override
            public Optional<String> trustManagerFactoryAlgorithm() {
                return trustManagerFactoryAlgorithm;
            }

            @Override
            public Optional<String> trustManagerFactoryProvider() {
                return trustManagerFactoryProvider;
            }

            @Override
            public List<String> applicationProtocols() {
                return applicationProtocols;
            }

            @Override
            public String endpointIdentificationAlgorithm() {
                return endpointIdentificationAlgorithm;
            }

            @Override
            public boolean enabled() {
                return enabled;
            }

            @Override
            public boolean trustAll() {
                return trustAll;
            }

            @Override
            public TlsClientAuth clientAuth() {
                return clientAuth;
            }

            @Override
            public String protocol() {
                return protocol;
            }

            @Override
            public Optional<String> provider() {
                return provider;
            }

            @Override
            public List<String> enabledCipherSuites() {
                return enabledCipherSuites;
            }

            @Override
            public List<String> enabledProtocols() {
                return enabledProtocols;
            }

            @Override
            public int sessionCacheSize() {
                return sessionCacheSize;
            }

            @Override
            public Duration sessionTimeout() {
                return sessionTimeout;
            }

            @Override
            public Optional<String> internalKeystoreType() {
                return internalKeystoreType;
            }

            @Override
            public Optional<String> internalKeystoreProvider() {
                return internalKeystoreProvider;
            }

            @Override
            public Optional<RevocationConfig> revocation() {
                return revocation;
            }

            @Override
            public String toString() {
                return "TlsConfig{"
                        + "sslContext=" + sslContext + ","
                        + "privateKey=" + privateKey + ","
                        + "privateKeyCertChain=" + privateKeyCertChain + ","
                        + "trust=" + trust + ","
                        + "manager=" + manager + ","
                        + "secureRandom=" + secureRandom + ","
                        + "sslParameters=" + sslParameters + ","
                        + "secureRandomProvider=" + secureRandomProvider + ","
                        + "secureRandomAlgorithm=" + secureRandomAlgorithm + ","
                        + "keyManagerFactoryAlgorithm=" + keyManagerFactoryAlgorithm + ","
                        + "keyManagerFactoryProvider=" + keyManagerFactoryProvider + ","
                        + "trustManagerFactoryAlgorithm=" + trustManagerFactoryAlgorithm + ","
                        + "trustManagerFactoryProvider=" + trustManagerFactoryProvider + ","
                        + "applicationProtocols=" + applicationProtocols + ","
                        + "endpointIdentificationAlgorithm=" + endpointIdentificationAlgorithm + ","
                        + "enabled=" + enabled + ","
                        + "trustAll=" + trustAll + ","
                        + "clientAuth=" + clientAuth + ","
                        + "protocol=" + protocol + ","
                        + "provider=" + provider + ","
                        + "enabledCipherSuites=" + enabledCipherSuites + ","
                        + "enabledProtocols=" + enabledProtocols + ","
                        + "sessionCacheSize=" + sessionCacheSize + ","
                        + "sessionTimeout=" + sessionTimeout + ","
                        + "internalKeystoreType=" + internalKeystoreType + ","
                        + "internalKeystoreProvider=" + internalKeystoreProvider + ","
                        + "revocation=" + revocation
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof TlsConfig other)) {
                    return false;
                }
                return Objects.equals(sslContext, other.sslContext())
                    && Objects.equals(privateKey, other.privateKey())
                    && Objects.equals(privateKeyCertChain, other.privateKeyCertChain())
                    && Objects.equals(trust, other.trust())
                    && Objects.equals(manager, other.manager())
                    && Objects.equals(secureRandom, other.secureRandom())
                    && Objects.equals(sslParameters, other.sslParameters())
                    && Objects.equals(secureRandomProvider, other.secureRandomProvider())
                    && Objects.equals(secureRandomAlgorithm, other.secureRandomAlgorithm())
                    && Objects.equals(keyManagerFactoryAlgorithm, other.keyManagerFactoryAlgorithm())
                    && Objects.equals(keyManagerFactoryProvider, other.keyManagerFactoryProvider())
                    && Objects.equals(trustManagerFactoryAlgorithm, other.trustManagerFactoryAlgorithm())
                    && Objects.equals(trustManagerFactoryProvider, other.trustManagerFactoryProvider())
                    && Objects.equals(applicationProtocols, other.applicationProtocols())
                    && Objects.equals(endpointIdentificationAlgorithm, other.endpointIdentificationAlgorithm())
                    && enabled == other.enabled()
                    && trustAll == other.trustAll()
                    && Objects.equals(clientAuth, other.clientAuth())
                    && Objects.equals(protocol, other.protocol())
                    && Objects.equals(provider, other.provider())
                    && Objects.equals(enabledCipherSuites, other.enabledCipherSuites())
                    && Objects.equals(enabledProtocols, other.enabledProtocols())
                    && sessionCacheSize == other.sessionCacheSize()
                    && Objects.equals(sessionTimeout, other.sessionTimeout())
                    && Objects.equals(internalKeystoreType, other.internalKeystoreType())
                    && Objects.equals(internalKeystoreProvider, other.internalKeystoreProvider())
                    && Objects.equals(revocation, other.revocation());
            }

            @Override
            public int hashCode() {
                return Objects.hash(sslContext, privateKey, privateKeyCertChain, trust, manager, secureRandom, sslParameters, secureRandomProvider, secureRandomAlgorithm, keyManagerFactoryAlgorithm, keyManagerFactoryProvider, trustManagerFactoryAlgorithm, trustManagerFactoryProvider, applicationProtocols, endpointIdentificationAlgorithm, enabled, trustAll, clientAuth, protocol, provider, enabledCipherSuites, enabledProtocols, sessionCacheSize, sessionTimeout, internalKeystoreType, internalKeystoreProvider, revocation);
            }

        }

    }

    /**
     * Fluent API builder for {@link Tls}.
     */
    class Builder extends TlsConfig.BuilderBase<TlsConfig.Builder, TlsConfig> implements io.helidon.common.Builder<TlsConfig.Builder, Tls> {

        private Builder() {
        }

        @Override
        public TlsConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new TlsConfigImpl(this);
        }

        @Override
        public Tls build() {
            return Tls.create(this.buildPrototype());
        }

    }

}
