/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.config.encryption;

import io.helidon.common.Base64Value;
import io.helidon.common.LazyValue;
import io.helidon.common.configurable.Resource;
import io.helidon.common.crypto.AsymmetricCipher;
import io.helidon.common.crypto.PasswordKeyDerivation;
import io.helidon.common.crypto.SymmetricCipher;
import io.helidon.common.pki.Keys;
import io.helidon.common.pki.KeystoreKeys;
import io.helidon.common.pki.PemKeys;
import io.helidon.config.Config;
import io.helidon.config.ConfigValue;
import io.helidon.config.encryption.ConfigEncryptionException;
import io.helidon.config.mp.MpConfig;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.Objects;
import java.util.Optional;

public final class EncryptionUtil {
    private static final System.Logger LOGGER = System.getLogger(EncryptionUtil.class.getName());
    private static final LazyValue<SecureRandom> SECURE_RANDOM = LazyValue.create(SecureRandom::new);
    private static final int SALT_LENGTH = 16;
    private static final int NONCE_LENGTH = 12;
    private static final int HASH_ITERATIONS = 10000;
    private static final int KEY_LENGTH = 256;

    private EncryptionUtil() {
        throw new IllegalStateException("Utility class");
    }

    public static String decryptRsa(PrivateKey key, String encryptedBase64) throws ConfigEncryptionException {
        Objects.requireNonNull(key, "Key must be provided for decryption");
        Objects.requireNonNull(encryptedBase64, "Encrypted bytes must be provided for decryption (base64 encoded)");
        try {
            Base64Value value = Base64Value.createFromEncoded((String)encryptedBase64);
            return AsymmetricCipher.decrypt((String)"RSA/ECB/OAEPWithSHA-256AndMGF1Padding", null, (PrivateKey)key, (Base64Value)value).toDecodedString();
        }
        catch (ConfigEncryptionException e) {
            throw e;
        }
        catch (Exception e) {
            throw new ConfigEncryptionException("Failed to decrypt value using RSA. Returning clear text value as is: " + encryptedBase64, e);
        }
    }

    public static String encryptRsa(PublicKey key, String secret) throws ConfigEncryptionException {
        Objects.requireNonNull(key, "Key must be provided for encryption");
        Objects.requireNonNull(secret, "Secret message must be provided to be encrypted");
        if (secret.getBytes(StandardCharsets.UTF_8).length > 190) {
            throw new ConfigEncryptionException("Secret value is too large. Maximum of 190 bytes is allowed.");
        }
        try {
            Base64Value value = Base64Value.create((String)secret);
            return AsymmetricCipher.encrypt((String)"RSA/ECB/OAEPWithSHA-256AndMGF1Padding", null, (PublicKey)key, (Base64Value)value).toBase64();
        }
        catch (Exception e) {
            throw new ConfigEncryptionException("Failed to encrypt using RSA key", e);
        }
    }

    public static String encryptAes(char[] masterPassword, String secret) throws ConfigEncryptionException {
        Objects.requireNonNull(secret, "Secret message must be provided to be encrypted");
        return EncryptionUtil.encryptAesBytes(masterPassword, secret.getBytes(StandardCharsets.UTF_8));
    }

    @Deprecated(since="2.2.0")
    public static String encryptAesBytes(char[] masterPassword, byte[] secret) throws ConfigEncryptionException {
        Objects.requireNonNull(masterPassword, "Password must be provided for encryption");
        Objects.requireNonNull(secret, "Secret message must be provided to be encrypted");
        byte[] salt = ((SecureRandom)SECURE_RANDOM.get()).generateSeed(16);
        byte[] nonce = ((SecureRandom)SECURE_RANDOM.get()).generateSeed(12);
        byte[] key = PasswordKeyDerivation.deriveKey((String)"PBKDF2WithHmacSHA256", null, (char[])masterPassword, (byte[])salt, (int)10000, (int)256);
        byte[] encrypted = SymmetricCipher.encrypt((String)"AES/GCM/NoPadding", (byte[])key, (byte[])nonce, (Base64Value)Base64Value.create((byte[])secret)).toBytes();
        byte[] bytesToEncode = new byte[encrypted.length + salt.length + nonce.length];
        System.arraycopy(salt, 0, bytesToEncode, 0, salt.length);
        System.arraycopy(nonce, 0, bytesToEncode, salt.length, nonce.length);
        System.arraycopy(encrypted, 0, bytesToEncode, nonce.length + salt.length, encrypted.length);
        return Base64.getEncoder().encodeToString(bytesToEncode);
    }

    public static String decryptAes(char[] masterPassword, String encryptedBase64) throws ConfigEncryptionException {
        return new String(EncryptionUtil.decryptAesBytes(masterPassword, encryptedBase64), StandardCharsets.UTF_8);
    }

    @Deprecated(since="2.2.0")
    public static byte[] decryptAesBytes(char[] masterPassword, String encryptedBase64) {
        Objects.requireNonNull(masterPassword, "Password must be provided for encryption");
        Objects.requireNonNull(encryptedBase64, "Encrypted bytes must be provided for decryption (base64 encoded)");
        try {
            byte[] decodedBytes = Base64.getDecoder().decode(encryptedBase64);
            byte[] salt = new byte[16];
            byte[] nonce = new byte[12];
            byte[] encryptedBytes = new byte[decodedBytes.length - 16 - 12];
            System.arraycopy(decodedBytes, 0, salt, 0, 16);
            System.arraycopy(decodedBytes, 16, nonce, 0, 12);
            System.arraycopy(decodedBytes, 28, encryptedBytes, 0, encryptedBytes.length);
            byte[] key = PasswordKeyDerivation.deriveKey((String)"PBKDF2WithHmacSHA256", null, (char[])masterPassword, (byte[])salt, (int)10000, (int)256);
            Base64Value encryptedValue = Base64Value.create((byte[])encryptedBytes);
            return SymmetricCipher.decrypt((String)"AES/GCM/NoPadding", (byte[])key, (byte[])nonce, (Base64Value)encryptedValue).toBytes();
        }
        catch (Throwable e) {
            throw new ConfigEncryptionException("Failed to decrypt value using AES. Returning clear text value as is: " + encryptedBase64, e);
        }
    }

    static Optional<char[]> resolveMasterPassword(boolean requireEncryption, org.eclipse.microprofile.config.Config config) {
        Optional<char[]> result = EncryptionUtil.getEnv("SECURE_CONFIG_AES_MASTER_PWD").or(() -> {
            Optional value = config.getOptionalValue("security.config.aes.insecure-passphrase", String.class);
            if (value.isPresent() && requireEncryption) {
                LOGGER.log(System.Logger.Level.WARNING, "Master password is configured as clear text in configuration when encryption is required. This value will be ignored. System property or environment variable expected!!!");
                return Optional.empty();
            }
            return value;
        }).map(String::toCharArray);
        if (result.isEmpty()) {
            LOGGER.log(System.Logger.Level.DEBUG, "Securing properties using master password is not available, as master password is not configured");
        }
        return result;
    }

    static Optional<char[]> resolveMasterPassword(boolean requireEncryption, Config config) {
        Optional<char[]> result = EncryptionUtil.getEnv("SECURE_CONFIG_AES_MASTER_PWD").or(() -> {
            ConfigValue value = config.get("security.config.aes.insecure-passphrase").asString();
            if (value.isPresent() && requireEncryption) {
                LOGGER.log(System.Logger.Level.WARNING, "Master password is configured as clear text in configuration when encryption is required. This value will be ignored. System property or environment variable expected!!!");
                return Optional.empty();
            }
            return value.asOptional();
        }).map(String::toCharArray);
        if (!result.isPresent()) {
            LOGGER.log(System.Logger.Level.DEBUG, "Securing properties using master password is not available, as master password is not configured");
        }
        return result;
    }

    static Optional<PrivateKey> resolvePrivateKey(org.eclipse.microprofile.config.Config config) {
        return EncryptionUtil.resolvePrivateKey(MpConfig.toHelidonConfig((org.eclipse.microprofile.config.Config)config).get("security.config.rsa"));
    }

    static Optional<PrivateKey> resolvePrivateKey(Config config) {
        Keys.Builder builder = Keys.builder();
        builder.config(config);
        builder.pem(pemBuilder -> {
            EncryptionUtil.getEnv("SECURE_CONFIG_RSA_PEM_KEY").map(x$0 -> Paths.get(x$0, new String[0])).ifPresent(path -> pemBuilder.key(Resource.create((Path)path)));
            EncryptionUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((PemKeys.Builder)pemBuilder).keyPassphrase(arg_0));
        });
        EncryptionUtil.getEnv("SECURE_CONFIG_RSA_PRIVATE_KEY").map(x$0 -> Paths.get(x$0, new String[0])).ifPresent(path -> builder.keystore(keystoreBuilder -> {
            keystoreBuilder.keystore(Resource.create((Path)path));
            EncryptionUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_TYPE").ifPresent(arg_0 -> ((KeystoreKeys.Builder)keystoreBuilder).type(arg_0));
            EncryptionUtil.getEnv("SECURE_CONFIG_PRIVATE_KEYSTORE_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((KeystoreKeys.Builder)keystoreBuilder).passphrase(arg_0));
            EncryptionUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_PASSPHRASE").map(String::toCharArray).ifPresent(arg_0 -> ((KeystoreKeys.Builder)keystoreBuilder).keyPassphrase(arg_0));
            EncryptionUtil.getEnv("SECURE_CONFIG_PRIVATE_KEY_ALIAS").ifPresent(arg_0 -> ((KeystoreKeys.Builder)keystoreBuilder).keyAlias(arg_0));
        }));
        Optional result = builder.build().privateKey();
        if (result.isEmpty()) {
            LOGGER.log(System.Logger.Level.DEBUG, "Securing properties using asymmetric cipher is not available, as private key is not configured");
        }
        return result;
    }

    static Optional<String> getEnv(String envVariable) {
        return Optional.ofNullable(System.getenv(envVariable));
    }
}

