/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.config;

import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * Configuration item policy.
 * Contains information about the given item to improve its handling.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.config.ConfigItemBlueprint")
public interface ConfigItem extends ConfigItemBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ConfigItem.Builder builder() {
        return new ConfigItem.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ConfigItem.Builder builder(ConfigItem instance) {
        return ConfigItem.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ConfigItem create() {
        return ConfigItem.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ConfigItem}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ConfigItem.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ConfigItem> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private boolean cacheItem;
        private String item;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ConfigItem prototype) {
            cacheItem(prototype.cacheItem());
            item(prototype.item());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ConfigItem.BuilderBase<?, ?> builder) {
            cacheItem(builder.cacheItem());
            builder.item().ifPresent(this::item);
            return self();
        }

        /**
         * Whether to cache this handled config item or not.
         * If overall caching is disabled, this will not turn it on even if set to true.
         *
         * @param cacheItem whether to cache handled config item
         * @return updated builder instance
         * @see #cacheItem()
         */
        public BUILDER cacheItem(boolean cacheItem) {
            this.cacheItem = cacheItem;
            return self();
        }

        /**
         * Handled config item.
         *
         * @param item config item
         * @return updated builder instance
         * @see #item()
         */
        public BUILDER item(String item) {
            Objects.requireNonNull(item);
            this.item = item;
            return self();
        }

        /**
         * Whether to cache this handled config item or not.
         * If overall caching is disabled, this will not turn it on even if set to true.
         *
         * @return the cache item
         */
        public boolean cacheItem() {
            return cacheItem;
        }

        /**
         * Handled config item.
         *
         * @return the item
         */
        public Optional<String> item() {
            return Optional.ofNullable(item);
        }

        @Override
        public String toString() {
            return "ConfigItemBuilder{"
                    + "cacheItem=" + cacheItem + ","
                    + "item=" + item
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (item == null) {
                collector.fatal(getClass(), "Property \"item\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ConfigItemImpl implements ConfigItem {

            private final boolean cacheItem;
            private final String item;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ConfigItemImpl(ConfigItem.BuilderBase<?, ?> builder) {
                this.cacheItem = builder.cacheItem();
                this.item = builder.item().get();
            }

            @Override
            public boolean cacheItem() {
                return cacheItem;
            }

            @Override
            public String item() {
                return item;
            }

            @Override
            public String toString() {
                return "ConfigItem{"
                        + "cacheItem=" + cacheItem + ","
                        + "item=" + item
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ConfigItem other)) {
                    return false;
                }
                return cacheItem == other.cacheItem()
                        && Objects.equals(item, other.item());
            }

            @Override
            public int hashCode() {
                return Objects.hash(cacheItem, item);
            }

        }

    }

    /**
     * Fluent API builder for {@link ConfigItem}.
     */
    class Builder extends ConfigItem.BuilderBase<ConfigItem.Builder, ConfigItem> implements io.helidon.common.Builder<ConfigItem.Builder, ConfigItem> {

        private Builder() {
        }

        @Override
        public ConfigItem buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ConfigItemImpl(this);
        }

        @Override
        public ConfigItem build() {
            return buildPrototype();
        }

    }

}
