/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.faulttolerance;

import io.helidon.faulttolerance.ErrorChecker;
import io.helidon.faulttolerance.Retry;
import io.helidon.faulttolerance.RetryConfig;
import io.helidon.faulttolerance.RetryTimeoutException;
import io.helidon.faulttolerance.SupplierException;
import io.helidon.faulttolerance.SupplierHelper;
import java.time.Duration;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

class RetryImpl
implements Retry {
    private final ErrorChecker errorChecker;
    private final long maxTimeNanos;
    private final Retry.RetryPolicy retryPolicy;
    private final RetryConfig retryConfig;
    private final AtomicLong retryCounter = new AtomicLong(0L);
    private final String name;

    RetryImpl(RetryConfig retryConfig) {
        this.name = retryConfig.name().orElseGet(() -> "retry-" + System.identityHashCode(retryConfig));
        this.errorChecker = ErrorChecker.create(retryConfig.skipOn(), retryConfig.applyOn());
        this.maxTimeNanos = retryConfig.overallTimeout().toNanos();
        this.retryPolicy = retryConfig.retryPolicy().orElseThrow();
        this.retryConfig = retryConfig;
    }

    public RetryConfig prototype() {
        return this.retryConfig;
    }

    @Override
    public String name() {
        return this.name;
    }

    @Override
    public <T> T invoke(Supplier<? extends T> supplier) {
        RetryContext context = new RetryContext();
        while (true) {
            try {
                return supplier.get();
            }
            catch (Throwable t) {
                Throwable throwable = SupplierHelper.unwrapThrowable(t);
                if (throwable instanceof ExecutionException) {
                    throwable = throwable.getCause();
                }
                context.thrown.add(throwable);
                if (this.errorChecker.shouldSkip(throwable) || throwable instanceof InterruptedException) {
                    return (T)context.throwIt();
                }
                int currentCallIndex = context.count.incrementAndGet();
                Optional<Long> maybeDelay = this.computeDelay(context, currentCallIndex);
                if (maybeDelay.isEmpty()) {
                    return (T)context.throwIt();
                }
                long delayMillis = maybeDelay.get();
                long now = System.nanoTime();
                this.checkTimeout(context, now);
                this.checkTimeout(context, now + TimeUnit.MILLISECONDS.toNanos(delayMillis));
                this.retryCounter.getAndIncrement();
                try {
                    Thread.sleep(Duration.ofMillis(delayMillis));
                }
                catch (InterruptedException e) {
                    e.addSuppressed(context.throwable());
                    throw new RuntimeException("Retries interrupted", e);
                }
                context.lastDelay.set(delayMillis);
                continue;
            }
            break;
        }
    }

    public void checkTimeout(RetryContext<?> context, long nanoTime) {
        if (nanoTime - context.startedNanos > this.maxTimeNanos) {
            RetryTimeoutException te = new RetryTimeoutException("Execution took too long. Already executing for: " + TimeUnit.NANOSECONDS.toMillis(nanoTime - context.startedNanos) + " ms, must be lower than overallTimeout duration of: " + TimeUnit.NANOSECONDS.toMillis(this.maxTimeNanos) + " ms.", context.throwable());
            throw te;
        }
    }

    @Override
    public long retryCounter() {
        return this.retryCounter.get();
    }

    private Optional<Long> computeDelay(RetryContext<?> context, int currentCallIndex) {
        return this.retryPolicy.nextDelayMillis(context.startedMillis, context.lastDelay.get(), currentCallIndex);
    }

    private static class RetryContext<U> {
        private final long startedMillis = System.currentTimeMillis();
        private final long startedNanos = System.nanoTime();
        private final AtomicInteger count = new AtomicInteger();
        private final List<Throwable> thrown = new LinkedList<Throwable>();
        private final AtomicLong lastDelay = new AtomicLong();

        RetryContext() {
        }

        public U throwIt() {
            Throwable t = this.throwable();
            if (t instanceof RuntimeException) {
                RuntimeException r = (RuntimeException)t;
                throw r;
            }
            if (t instanceof Error) {
                Error e = (Error)t;
                throw e;
            }
            throw new SupplierException(t);
        }

        boolean hasThrowable() {
            return !this.thrown.isEmpty();
        }

        Throwable throwable() {
            if (this.thrown.isEmpty()) {
                return new IllegalStateException("Exception list is empty");
            }
            Throwable last = this.thrown.get(this.thrown.size() - 1);
            for (int i = 0; i < this.thrown.size() - 1; ++i) {
                Throwable throwable = this.thrown.get(i);
                if (throwable == last) continue;
                last.addSuppressed(throwable);
            }
            return last;
        }
    }
}

