/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.grpc.client;

import io.grpc.CallOptions;
import io.grpc.Channel;
import io.grpc.ClientCall;
import io.grpc.ClientInterceptor;
import io.grpc.ClientInterceptors;
import io.grpc.ForwardingClientCall;
import io.grpc.ForwardingClientCallListener;
import io.grpc.Metadata;
import io.grpc.MethodDescriptor;
import io.grpc.Status;
import io.helidon.grpc.client.ClientRequestAttribute;
import io.helidon.grpc.core.ContextKeys;
import io.helidon.grpc.core.GrpcTracingContext;
import io.helidon.grpc.core.GrpcTracingName;
import io.helidon.tracing.HeaderConsumer;
import io.helidon.tracing.HeaderProvider;
import io.helidon.tracing.Span;
import io.helidon.tracing.Tracer;
import jakarta.annotation.Priority;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;

@Priority(value=1001)
public class ClientTracingInterceptor
implements ClientInterceptor {
    private final Tracer tracer;
    private final GrpcTracingName operationNameConstructor;
    private final boolean streaming;
    private final boolean verbose;
    private final Set<ClientRequestAttribute> tracedAttributes;

    private ClientTracingInterceptor(Tracer tracer, GrpcTracingName operationNameConstructor, boolean streaming, boolean verbose, Set<ClientRequestAttribute> tracedAttributes) {
        this.tracer = tracer;
        this.operationNameConstructor = operationNameConstructor;
        this.streaming = streaming;
        this.verbose = verbose;
        this.tracedAttributes = tracedAttributes;
    }

    public static Builder builder(Tracer tracer) {
        return new Builder(tracer);
    }

    public Channel intercept(Channel channel) {
        return ClientInterceptors.intercept((Channel)channel, (ClientInterceptor[])new ClientInterceptor[]{this});
    }

    public <ReqT, RespT> ClientCall<ReqT, RespT> interceptCall(MethodDescriptor<ReqT, RespT> method, CallOptions callOptions, Channel next) {
        String operationName = this.operationNameConstructor.name(method);
        Span span = this.createSpanFromParent(operationName);
        for (ClientRequestAttribute attr : this.tracedAttributes) {
            switch (attr) {
                case ALL_CALL_OPTIONS: {
                    span.tag("grpc.call_options", callOptions.toString());
                    break;
                }
                case AUTHORITY: {
                    if (callOptions.getAuthority() == null) {
                        span.tag("grpc.authority", "null");
                        break;
                    }
                    span.tag("grpc.authority", callOptions.getAuthority());
                    break;
                }
                case COMPRESSOR: {
                    if (callOptions.getCompressor() == null) {
                        span.tag("grpc.compressor", "null");
                        break;
                    }
                    span.tag("grpc.compressor", callOptions.getCompressor());
                    break;
                }
                case DEADLINE: {
                    if (callOptions.getDeadline() == null) {
                        span.tag("grpc.deadline_millis", "null");
                        break;
                    }
                    span.tag("grpc.deadline_millis", (Number)callOptions.getDeadline().timeRemaining(TimeUnit.MILLISECONDS));
                    break;
                }
                case METHOD_NAME: {
                    span.tag("grpc.method_name", method.getFullMethodName());
                    break;
                }
                case METHOD_TYPE: {
                    if (method.getType() == null) {
                        span.tag("grpc.method_type", "null");
                        break;
                    }
                    span.tag("grpc.method_type", method.getType().toString());
                    break;
                }
                case HEADERS: {
                    break;
                }
            }
        }
        return new ClientTracingListener(next.newCall(method, callOptions), span);
    }

    private Span createSpanFromParent(String operationName) {
        return ((Span.Builder)this.tracer.spanBuilder(operationName).update(it -> GrpcTracingContext.activeSpan().map(Span::context).ifPresent(arg_0 -> ((Span.Builder)it).parent(arg_0)))).start();
    }

    public static class Builder {
        private final Tracer tracer;
        private GrpcTracingName operationNameConstructor;
        private boolean streaming;
        private boolean verbose;
        private Set<ClientRequestAttribute> tracedAttributes;

        public Builder(Tracer tracer) {
            this.tracer = tracer;
            this.operationNameConstructor = MethodDescriptor::getFullMethodName;
            this.streaming = false;
            this.verbose = false;
            this.tracedAttributes = new HashSet<ClientRequestAttribute>();
        }

        public Builder withOperationName(GrpcTracingName operationNameConstructor) {
            this.operationNameConstructor = operationNameConstructor;
            return this;
        }

        public Builder withStreaming() {
            this.streaming = true;
            return this;
        }

        public Builder withTracedAttributes(ClientRequestAttribute ... tracedAttributes) {
            this.tracedAttributes = new HashSet<ClientRequestAttribute>(Arrays.asList(tracedAttributes));
            return this;
        }

        public Builder withVerbosity() {
            this.verbose = true;
            return this;
        }

        public ClientTracingInterceptor build() {
            return new ClientTracingInterceptor(this.tracer, this.operationNameConstructor, this.streaming, this.verbose, this.tracedAttributes);
        }
    }

    private class ClientTracingListener<ReqT, RespT>
    extends ForwardingClientCall.SimpleForwardingClientCall<ReqT, RespT> {
        private final Span span;

        private ClientTracingListener(ClientCall<ReqT, RespT> delegate, Span span) {
            super(delegate);
            this.span = span;
        }

        public void start(ClientCall.Listener<RespT> responseListener, Metadata headers) {
            if (ClientTracingInterceptor.this.verbose) {
                this.span.addEvent("Started call");
            }
            if (ClientTracingInterceptor.this.tracedAttributes.contains((Object)ClientRequestAttribute.HEADERS)) {
                Metadata metadata = new Metadata();
                metadata.merge(headers);
                metadata.removeAll(ContextKeys.AUTHORIZATION);
                this.span.tag("grpc.headers", metadata.toString());
            }
            ClientTracingInterceptor.this.tracer.inject(this.span.context(), HeaderProvider.empty(), (HeaderConsumer)new MetadataHeaderConsumer(headers));
            ForwardingClientCallListener.SimpleForwardingClientCallListener tracingResponseListener = new ForwardingClientCallListener.SimpleForwardingClientCallListener<RespT>(responseListener){

                public void onHeaders(Metadata headers) {
                    if (ClientTracingInterceptor.this.verbose) {
                        ClientTracingListener.this.span.addEvent("headers", Map.of("Response headers received", headers.toString()));
                    }
                    this.delegate().onHeaders(headers);
                }

                public void onClose(Status status, Metadata trailers) {
                    if (ClientTracingInterceptor.this.verbose) {
                        if (status.getCode().value() == 0) {
                            ClientTracingListener.this.span.addEvent("Call closed");
                        } else {
                            String desc = String.valueOf(status.getDescription());
                            ClientTracingListener.this.span.addEvent("onClose", Map.of("Call failed", desc));
                        }
                    }
                    ClientTracingListener.this.span.end();
                    this.delegate().onClose(status, trailers);
                }

                public void onMessage(RespT message) {
                    if (ClientTracingInterceptor.this.streaming || ClientTracingInterceptor.this.verbose) {
                        ClientTracingListener.this.span.addEvent("Response received");
                    }
                    this.delegate().onMessage(message);
                }
            };
            this.delegate().start((ClientCall.Listener)tracingResponseListener, headers);
        }

        public void sendMessage(ReqT message) {
            if (ClientTracingInterceptor.this.streaming || ClientTracingInterceptor.this.verbose) {
                this.span.addEvent("Message sent");
            }
            this.delegate().sendMessage(message);
        }

        public void cancel(String message, Throwable cause) {
            String errorMessage;
            String string = errorMessage = message == null ? "Error" : message;
            if (cause == null) {
                this.span.addEvent(errorMessage);
            } else {
                this.span.addEvent("error", Map.of(errorMessage, cause.getMessage()));
            }
            this.delegate().cancel(message, cause);
        }

        public void halfClose() {
            if (ClientTracingInterceptor.this.streaming) {
                this.span.addEvent("Finished sending messages");
            }
            this.delegate().halfClose();
        }

        private class MetadataHeaderConsumer
        implements HeaderConsumer {
            private final Metadata headers;

            private MetadataHeaderConsumer(Metadata headers) {
                this.headers = headers;
            }

            public void setIfAbsent(String key, String ... values) {
                Metadata.Key<String> headerKey = this.key(key);
                if (!this.headers.containsKey(headerKey)) {
                    this.headers.put(headerKey, (Object)values[0]);
                }
            }

            public void set(String key, String ... values) {
                Metadata.Key<String> headerKey = this.key(key);
                this.headers.put(headerKey, (Object)values[0]);
            }

            public Iterable<String> keys() {
                return this.headers.keys();
            }

            public Optional<String> get(String key) {
                return Optional.ofNullable((String)this.headers.get(this.key(key)));
            }

            public Iterable<String> getAll(String key) {
                return this.get(key).map(List::of).orElseGet(List::of);
            }

            public boolean contains(String key) {
                return this.headers.containsKey(this.key(key));
            }

            private Metadata.Key<String> key(String name) {
                return Metadata.Key.of((String)name, (Metadata.AsciiMarshaller)Metadata.ASCII_STRING_MARSHALLER);
            }

            private void put(Metadata.Key<String> key, String value) {
                this.headers.put(key, (Object)value);
            }
        }
    }
}

