/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * Log entry for lifecycle related events (i.e., activation startup and deactivation shutdown).
 *
 * @deprecated Helidon inject is deprecated and will be replaced in a future version
 * @see ActivationLog
 * @see Activator
 * @see DeActivator
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ActivationLogEntryBlueprint")
public interface ActivationLogEntry extends ActivationLogEntryBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ActivationLogEntry.Builder builder() {
        return new ActivationLogEntry.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ActivationLogEntry.Builder builder(ActivationLogEntry instance) {
        return ActivationLogEntry.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ActivationLogEntry create() {
        return ActivationLogEntry.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ActivationLogEntry}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ActivationLogEntry.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ActivationLogEntry> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private ActivationResult activationResult;
        private Event event;
        private InjectionPointInfo injectionPoint;
        private Instant time;
        private long threadId = 0;
        private ServiceProvider<?> serviceProvider;
        private String message;
        private Throwable error;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ActivationLogEntry prototype) {
            event(prototype.event());
            message(prototype.message());
            activationResult(prototype.activationResult());
            serviceProvider(prototype.serviceProvider());
            injectionPoint(prototype.injectionPoint());
            time(prototype.time());
            error(prototype.error());
            threadId(prototype.threadId());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ActivationLogEntry.BuilderBase<?, ?> builder) {
            builder.event().ifPresent(this::event);
            builder.message().ifPresent(this::message);
            builder.activationResult().ifPresent(this::activationResult);
            builder.serviceProvider().ifPresent(this::serviceProvider);
            builder.injectionPoint().ifPresent(this::injectionPoint);
            builder.time().ifPresent(this::time);
            builder.error().ifPresent(this::error);
            threadId(builder.threadId());
            return self();
        }

        /**
         * The event.
         *
         * @param event the event
         * @return updated builder instance
         * @see #event()
         */
        public BUILDER event(Event event) {
            Objects.requireNonNull(event);
            this.event = event;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #message()
         */
        public BUILDER clearMessage() {
            this.message = null;
            return self();
        }

        /**
         * Optionally, any special message being logged.
         *
         * @param message the message
         * @return updated builder instance
         * @see #message()
         */
        public BUILDER message(String message) {
            Objects.requireNonNull(message);
            this.message = message;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #activationResult()
         */
        public BUILDER clearActivationResult() {
            this.activationResult = null;
            return self();
        }

        /**
         * Optionally, when this log entry pertains to a service provider activation.
         *
         * @param activationResult the activation result
         * @return updated builder instance
         * @see #activationResult()
         */
        public BUILDER activationResult(ActivationResult activationResult) {
            Objects.requireNonNull(activationResult);
            this.activationResult = activationResult;
            return self();
        }

        /**
         * Optionally, when this log entry pertains to a service provider activation.
         *
         * @param consumer the activation result
         * @return updated builder instance
         * @see #activationResult()
         */
        public BUILDER activationResult(Consumer<ActivationResult.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ActivationResult.builder();
            consumer.accept(builder);
            this.activationResult(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #serviceProvider()
         */
        public BUILDER clearServiceProvider() {
            this.serviceProvider = null;
            return self();
        }

        /**
         * Optionally, the managing service provider the event pertains to.
         *
         * @param serviceProvider the managing service provider
         * @return updated builder instance
         * @see #serviceProvider()
         */
        public BUILDER serviceProvider(ServiceProvider<?> serviceProvider) {
            Objects.requireNonNull(serviceProvider);
            this.serviceProvider = serviceProvider;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #injectionPoint()
         */
        public BUILDER clearInjectionPoint() {
            this.injectionPoint = null;
            return self();
        }

        /**
         * Optionally, the injection point that the event pertains to.
         *
         * @param injectionPoint the injection point
         * @return updated builder instance
         * @see #injectionPoint()
         */
        public BUILDER injectionPoint(InjectionPointInfo injectionPoint) {
            Objects.requireNonNull(injectionPoint);
            this.injectionPoint = injectionPoint;
            return self();
        }

        /**
         * Optionally, the injection point that the event pertains to.
         *
         * @param consumer the injection point
         * @return updated builder instance
         * @see #injectionPoint()
         */
        public BUILDER injectionPoint(Consumer<InjectionPointInfo.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = InjectionPointInfo.builder();
            consumer.accept(builder);
            this.injectionPoint(builder.build());
            return self();
        }

        /**
         * The time this event was generated.
         *
         * @param time the time of the event
         * @return updated builder instance
         * @see #time()
         */
        public BUILDER time(Instant time) {
            Objects.requireNonNull(time);
            this.time = time;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #error()
         */
        public BUILDER clearError() {
            this.error = null;
            return self();
        }

        /**
         * Any observed error during activation.
         *
         * @param error any observed error
         * @return updated builder instance
         * @see #error()
         */
        public BUILDER error(Throwable error) {
            Objects.requireNonNull(error);
            this.error = error;
            return self();
        }

        /**
         * The thread id that the event occurred on.
         *
         * @param threadId the thread id
         * @return updated builder instance
         * @see #threadId()
         */
        public BUILDER threadId(long threadId) {
            this.threadId = threadId;
            return self();
        }

        /**
         * The event.
         *
         * @return the event
         */
        public Optional<Event> event() {
            return Optional.ofNullable(event);
        }

        /**
         * Optionally, any special message being logged.
         *
         * @return the message
         */
        public Optional<String> message() {
            return Optional.ofNullable(message);
        }

        /**
         * Optionally, when this log entry pertains to a service provider activation.
         *
         * @return the activation result
         */
        public Optional<ActivationResult> activationResult() {
            return Optional.ofNullable(activationResult);
        }

        /**
         * Optionally, the managing service provider the event pertains to.
         *
         * @return the service provider
         */
        public Optional<ServiceProvider<?>> serviceProvider() {
            return Optional.ofNullable(serviceProvider);
        }

        /**
         * Optionally, the injection point that the event pertains to.
         *
         * @return the injection point
         */
        public Optional<InjectionPointInfo> injectionPoint() {
            return Optional.ofNullable(injectionPoint);
        }

        /**
         * The time this event was generated.
         *
         * @return the time
         */
        public Optional<Instant> time() {
            return Optional.ofNullable(time);
        }

        /**
         * Any observed error during activation.
         *
         * @return the error
         */
        public Optional<Throwable> error() {
            return Optional.ofNullable(error);
        }

        /**
         * The thread id that the event occurred on.
         *
         * @return the thread id
         */
        public long threadId() {
            return threadId;
        }

        @Override
        public String toString() {
            return "ActivationLogEntryBuilder{"
                    + "event=" + event + ","
                    + "message=" + message + ","
                    + "activationResult=" + activationResult + ","
                    + "serviceProvider=" + serviceProvider + ","
                    + "injectionPoint=" + injectionPoint + ","
                    + "time=" + time + ","
                    + "error=" + error + ","
                    + "threadId=" + threadId
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new ActivationLogEntryBlueprint.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (event == null) {
                collector.fatal(getClass(), "Property \"event\" must not be null, but not set");
            }
            if (time == null) {
                collector.fatal(getClass(), "Property \"time\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Optionally, any special message being logged.
         *
         * @param message the message
         * @return updated builder instance
         * @see #message()
         */
        BUILDER message(Optional<String> message) {
            Objects.requireNonNull(message);
            this.message = message.map(java.lang.String.class::cast).orElse(this.message);
            return self();
        }

        /**
         * Optionally, when this log entry pertains to a service provider activation.
         *
         * @param activationResult the activation result
         * @return updated builder instance
         * @see #activationResult()
         */
        BUILDER activationResult(Optional<? extends ActivationResult> activationResult) {
            Objects.requireNonNull(activationResult);
            this.activationResult = activationResult.map(ActivationResult.class::cast).orElse(this.activationResult);
            return self();
        }

        /**
         * Optionally, the managing service provider the event pertains to.
         *
         * @param serviceProvider the managing service provider
         * @return updated builder instance
         * @see #serviceProvider()
         */
        BUILDER serviceProvider(Optional<ServiceProvider<?>> serviceProvider) {
            Objects.requireNonNull(serviceProvider);
            this.serviceProvider = serviceProvider.map(io.helidon.inject.api.ServiceProvider.class::cast).orElse(this.serviceProvider);
            return self();
        }

        /**
         * Optionally, the injection point that the event pertains to.
         *
         * @param injectionPoint the injection point
         * @return updated builder instance
         * @see #injectionPoint()
         */
        BUILDER injectionPoint(Optional<? extends InjectionPointInfo> injectionPoint) {
            Objects.requireNonNull(injectionPoint);
            this.injectionPoint = injectionPoint.map(InjectionPointInfo.class::cast).orElse(this.injectionPoint);
            return self();
        }

        /**
         * Any observed error during activation.
         *
         * @param error any observed error
         * @return updated builder instance
         * @see #error()
         */
        BUILDER error(Optional<? extends Throwable> error) {
            Objects.requireNonNull(error);
            this.error = error.map(java.lang.Throwable.class::cast).orElse(this.error);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ActivationLogEntryImpl implements ActivationLogEntry {

            private final Event event;
            private final Instant time;
            private final long threadId;
            private final Optional<ActivationResult> activationResult;
            private final Optional<InjectionPointInfo> injectionPoint;
            private final Optional<ServiceProvider<?>> serviceProvider;
            private final Optional<String> message;
            private final Optional<Throwable> error;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ActivationLogEntryImpl(ActivationLogEntry.BuilderBase<?, ?> builder) {
                this.event = builder.event().get();
                this.message = builder.message();
                this.activationResult = builder.activationResult();
                this.serviceProvider = builder.serviceProvider();
                this.injectionPoint = builder.injectionPoint();
                this.time = builder.time().get();
                this.error = builder.error();
                this.threadId = builder.threadId();
            }

            @Override
            public Event event() {
                return event;
            }

            @Override
            public Optional<String> message() {
                return message;
            }

            @Override
            public Optional<ActivationResult> activationResult() {
                return activationResult;
            }

            @Override
            public Optional<ServiceProvider<?>> serviceProvider() {
                return serviceProvider;
            }

            @Override
            public Optional<InjectionPointInfo> injectionPoint() {
                return injectionPoint;
            }

            @Override
            public Instant time() {
                return time;
            }

            @Override
            public Optional<Throwable> error() {
                return error;
            }

            @Override
            public long threadId() {
                return threadId;
            }

            @Override
            public String toString() {
                return "ActivationLogEntry{"
                        + "event=" + event + ","
                        + "message=" + message + ","
                        + "activationResult=" + activationResult + ","
                        + "serviceProvider=" + serviceProvider + ","
                        + "injectionPoint=" + injectionPoint + ","
                        + "time=" + time + ","
                        + "error=" + error + ","
                        + "threadId=" + threadId
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ActivationLogEntry other)) {
                    return false;
                }
                return Objects.equals(event, other.event())
                        && Objects.equals(message, other.message())
                        && Objects.equals(activationResult, other.activationResult())
                        && Objects.equals(serviceProvider, other.serviceProvider())
                        && Objects.equals(injectionPoint, other.injectionPoint())
                        && Objects.equals(time, other.time())
                        && Objects.equals(error, other.error())
                        && threadId == other.threadId();
            }

            @Override
            public int hashCode() {
                return Objects.hash(event, message, activationResult, serviceProvider, injectionPoint, time, error, threadId);
            }

        }

    }

    /**
     * Fluent API builder for {@link ActivationLogEntry}.
     */
    class Builder extends ActivationLogEntry.BuilderBase<ActivationLogEntry.Builder, ActivationLogEntry> implements io.helidon.common.Builder<ActivationLogEntry.Builder, ActivationLogEntry> {

        private Builder() {
        }

        @Override
        public ActivationLogEntry buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ActivationLogEntryImpl(this);
        }

        @Override
        public ActivationLogEntry build() {
            return buildPrototype();
        }

    }

}
