/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.types.AccessModifier;
import io.helidon.common.types.Annotation;
import io.helidon.common.types.TypeName;

/**
 * Abstractly describes method or field elements of a managed service type (i.e., fields, constructors, injectable methods, etc.).
 *
 * @deprecated Helidon inject is deprecated and will be replaced in a future version
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ElementInfoBlueprint")
public interface ElementInfo extends ElementInfoBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ElementInfo.Builder builder() {
        return new ElementInfo.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ElementInfo.Builder builder(ElementInfo instance) {
        return ElementInfo.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ElementInfo create() {
        return ElementInfo.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ElementInfo}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ElementInfo.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ElementInfo> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Set<Qualifier> qualifiers = new LinkedHashSet<>();
        private final Set<Annotation> annotations = new LinkedHashSet<>();
        private AccessModifier access;
        private boolean staticDeclaration = false;
        private ElementKind elementKind;
        private Integer elementArgs;
        private Integer elementOffset;
        private String elementName;
        private TypeName elementTypeName;
        private TypeName serviceTypeName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ElementInfo prototype) {
            elementKind(prototype.elementKind());
            access(prototype.access());
            elementTypeName(prototype.elementTypeName());
            elementName(prototype.elementName());
            elementOffset(prototype.elementOffset());
            elementArgs(prototype.elementArgs());
            staticDeclaration(prototype.staticDeclaration());
            serviceTypeName(prototype.serviceTypeName());
            addAnnotations(prototype.annotations());
            addQualifiers(prototype.qualifiers());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ElementInfo.BuilderBase<?, ?> builder) {
            builder.elementKind().ifPresent(this::elementKind);
            builder.access().ifPresent(this::access);
            builder.elementTypeName().ifPresent(this::elementTypeName);
            builder.elementName().ifPresent(this::elementName);
            builder.elementOffset().ifPresent(this::elementOffset);
            builder.elementArgs().ifPresent(this::elementArgs);
            staticDeclaration(builder.staticDeclaration());
            builder.serviceTypeName().ifPresent(this::serviceTypeName);
            addAnnotations(builder.annotations());
            addQualifiers(builder.qualifiers());
            return self();
        }

        /**
         * The injection point/receiver kind.
         *
         * @param elementKind the kind
         * @return updated builder instance
         * @see #elementKind()
         */
        public BUILDER elementKind(ElementKind elementKind) {
            Objects.requireNonNull(elementKind);
            this.elementKind = elementKind;
            return self();
        }

        /**
         * The access modifier on the injection point/receiver.
         *
         * @param access the access
         * @return updated builder instance
         * @see #access()
         */
        public BUILDER access(AccessModifier access) {
            Objects.requireNonNull(access);
            this.access = access;
            return self();
        }

        /**
         * The element type name (e.g., method type or field type).
         *
         * @param elementTypeName the target receiver type name
         * @return updated builder instance
         * @see #elementTypeName()
         */
        public BUILDER elementTypeName(TypeName elementTypeName) {
            Objects.requireNonNull(elementTypeName);
            this.elementTypeName = elementTypeName;
            return self();
        }

        /**
         * The element type name (e.g., method type or field type).
         *
         * @param consumer consumer of builder for
         *                 the target receiver type name
         * @return updated builder instance
         * @see #elementTypeName()
         */
        public BUILDER elementTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.elementTypeName(builder.build());
            return self();
        }

        /**
         * The element type name (e.g., method type or field type).
         *
         * @param supplier supplier of
         *                 the target receiver type name
         * @return updated builder instance
         * @see #elementTypeName()
         */
        public BUILDER elementTypeName(Supplier<? extends TypeName> supplier) {
            Objects.requireNonNull(supplier);
            this.elementTypeName(supplier.get());
            return self();
        }

        /**
         * The element name (e.g., method name or field name).
         *
         * @param elementName the target receiver name
         * @return updated builder instance
         * @see #elementName()
         */
        public BUILDER elementName(String elementName) {
            Objects.requireNonNull(elementName);
            this.elementName = elementName;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #elementOffset()
         */
        public BUILDER clearElementOffset() {
            this.elementOffset = null;
            return self();
        }

        /**
         * If the element is a method or constructor then this is the ordinal argument position of that argument.
         *
         * @param elementOffset the offset argument, 0 based, or empty if field type
         * @return updated builder instance
         * @see #elementOffset()
         */
        public BUILDER elementOffset(int elementOffset) {
            Objects.requireNonNull(elementOffset);
            this.elementOffset = elementOffset;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #elementArgs()
         */
        public BUILDER clearElementArgs() {
            this.elementArgs = null;
            return self();
        }

        /**
         * If the element is a method or constructor then this is the total argument count for that method.
         *
         * @param elementArgs total argument count
         * @return updated builder instance
         * @see #elementArgs()
         */
        public BUILDER elementArgs(int elementArgs) {
            Objects.requireNonNull(elementArgs);
            this.elementArgs = elementArgs;
            return self();
        }

        /**
         * True if the injection point is static.
         *
         * @param staticDeclaration true if static receiver
         * @return updated builder instance
         * @see #staticDeclaration()
         */
        public BUILDER staticDeclaration(boolean staticDeclaration) {
            this.staticDeclaration = staticDeclaration;
            return self();
        }

        /**
         * The enclosing class name for the element.
         *
         * @param serviceTypeName service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(TypeName serviceTypeName) {
            Objects.requireNonNull(serviceTypeName);
            this.serviceTypeName = serviceTypeName;
            return self();
        }

        /**
         * The enclosing class name for the element.
         *
         * @param consumer consumer of builder for
         *                 service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.serviceTypeName(builder.build());
            return self();
        }

        /**
         * The enclosing class name for the element.
         *
         * @param supplier supplier of
         *                 service type name
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Supplier<? extends TypeName> supplier) {
            Objects.requireNonNull(supplier);
            this.serviceTypeName(supplier.get());
            return self();
        }

        /**
         * The annotations on this element.
         *
         * @param annotations the annotations on this element
         * @return updated builder instance
         * @see #annotations()
         */
        public BUILDER annotations(Set<? extends Annotation> annotations) {
            Objects.requireNonNull(annotations);
            this.annotations.clear();
            this.annotations.addAll(annotations);
            return self();
        }

        /**
         * The annotations on this element.
         *
         * @param annotations the annotations on this element
         * @return updated builder instance
         * @see #annotations()
         */
        public BUILDER addAnnotations(Set<? extends Annotation> annotations) {
            Objects.requireNonNull(annotations);
            this.annotations.addAll(annotations);
            return self();
        }

        /**
         * The annotations on this element.
         *
         * @param annotation the annotations on this element
         * @return updated builder instance
         * @see #annotations()
         */
        public BUILDER addAnnotation(Annotation annotation) {
            Objects.requireNonNull(annotation);
            this.annotations.add(annotation);
            return self();
        }

        /**
         * The annotations on this element.
         *
         * @param consumer the annotations on this element
         * @return updated builder instance
         * @see #annotations()
         */
        public BUILDER addAnnotation(Consumer<Annotation.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Annotation.builder();
            consumer.accept(builder);
            this.annotations.add(builder.build());
            return self();
        }

        /**
         * The qualifier type annotations on this element.
         *
         * @param qualifiers the qualifier type annotations on this element
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER qualifiers(Set<? extends Qualifier> qualifiers) {
            Objects.requireNonNull(qualifiers);
            this.qualifiers.clear();
            this.qualifiers.addAll(qualifiers);
            return self();
        }

        /**
         * The qualifier type annotations on this element.
         *
         * @param qualifiers the qualifier type annotations on this element
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifiers(Set<? extends Qualifier> qualifiers) {
            Objects.requireNonNull(qualifiers);
            this.qualifiers.addAll(qualifiers);
            return self();
        }

        /**
         * The qualifier type annotations on this element.
         *
         * @param qualifier the qualifier type annotations on this element
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifier(Qualifier qualifier) {
            Objects.requireNonNull(qualifier);
            this.qualifiers.add(qualifier);
            return self();
        }

        /**
         * The qualifier type annotations on this element.
         *
         * @param consumer the qualifier type annotations on this element
         * @return updated builder instance
         * @see #qualifiers()
         */
        public BUILDER addQualifier(Consumer<Qualifier.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = Qualifier.builder();
            consumer.accept(builder);
            this.qualifiers.add(builder.build());
            return self();
        }

        /**
         * The injection point/receiver kind.
         *
         * @return the element kind
         */
        public Optional<ElementKind> elementKind() {
            return Optional.ofNullable(elementKind);
        }

        /**
         * The access modifier on the injection point/receiver.
         *
         * @return the access
         */
        public Optional<AccessModifier> access() {
            return Optional.ofNullable(access);
        }

        /**
         * The element type name (e.g., method type or field type).
         *
         * @return the element type name
         */
        public Optional<TypeName> elementTypeName() {
            return Optional.ofNullable(elementTypeName);
        }

        /**
         * The element name (e.g., method name or field name).
         *
         * @return the element name
         */
        public Optional<String> elementName() {
            return Optional.ofNullable(elementName);
        }

        /**
         * If the element is a method or constructor then this is the ordinal argument position of that argument.
         *
         * @return the element offset
         */
        public Optional<Integer> elementOffset() {
            return Optional.ofNullable(elementOffset);
        }

        /**
         * If the element is a method or constructor then this is the total argument count for that method.
         *
         * @return the element args
         */
        public Optional<Integer> elementArgs() {
            return Optional.ofNullable(elementArgs);
        }

        /**
         * True if the injection point is static.
         *
         * @return the static declaration
         */
        public boolean staticDeclaration() {
            return staticDeclaration;
        }

        /**
         * The enclosing class name for the element.
         *
         * @return the service type name
         */
        public Optional<TypeName> serviceTypeName() {
            return Optional.ofNullable(serviceTypeName);
        }

        /**
         * The annotations on this element.
         *
         * @return the annotations
         */
        public Set<Annotation> annotations() {
            return annotations;
        }

        /**
         * The qualifier type annotations on this element.
         *
         * @return the qualifiers
         */
        public Set<Qualifier> qualifiers() {
            return qualifiers;
        }

        @Override
        public String toString() {
            return "ElementInfoBuilder{"
                    + "elementKind=" + elementKind + ","
                    + "access=" + access + ","
                    + "elementTypeName=" + elementTypeName + ","
                    + "elementName=" + elementName + ","
                    + "elementOffset=" + elementOffset + ","
                    + "elementArgs=" + elementArgs + ","
                    + "staticDeclaration=" + staticDeclaration + ","
                    + "serviceTypeName=" + serviceTypeName + ","
                    + "annotations=" + annotations + ","
                    + "qualifiers=" + qualifiers
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (elementKind == null) {
                collector.fatal(getClass(), "Property \"elementKind\" must not be null, but not set");
            }
            if (access == null) {
                collector.fatal(getClass(), "Property \"access\" must not be null, but not set");
            }
            if (elementTypeName == null) {
                collector.fatal(getClass(), "Property \"elementTypeName\" must not be null, but not set");
            }
            if (elementName == null) {
                collector.fatal(getClass(), "Property \"elementName\" must not be null, but not set");
            }
            if (serviceTypeName == null) {
                collector.fatal(getClass(), "Property \"serviceTypeName\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * If the element is a method or constructor then this is the ordinal argument position of that argument.
         *
         * @param elementOffset the offset argument, 0 based, or empty if field type
         * @return updated builder instance
         * @see #elementOffset()
         */
        BUILDER elementOffset(Optional<Integer> elementOffset) {
            Objects.requireNonNull(elementOffset);
            this.elementOffset = elementOffset.map(java.lang.Integer.class::cast).orElse(this.elementOffset);
            return self();
        }

        /**
         * If the element is a method or constructor then this is the total argument count for that method.
         *
         * @param elementArgs total argument count
         * @return updated builder instance
         * @see #elementArgs()
         */
        BUILDER elementArgs(Optional<Integer> elementArgs) {
            Objects.requireNonNull(elementArgs);
            this.elementArgs = elementArgs.map(java.lang.Integer.class::cast).orElse(this.elementArgs);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ElementInfoImpl implements ElementInfo {

            private final AccessModifier access;
            private final boolean staticDeclaration;
            private final ElementKind elementKind;
            private final Optional<Integer> elementArgs;
            private final Optional<Integer> elementOffset;
            private final Set<Qualifier> qualifiers;
            private final Set<Annotation> annotations;
            private final String elementName;
            private final TypeName elementTypeName;
            private final TypeName serviceTypeName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ElementInfoImpl(ElementInfo.BuilderBase<?, ?> builder) {
                this.elementKind = builder.elementKind().get();
                this.access = builder.access().get();
                this.elementTypeName = builder.elementTypeName().get();
                this.elementName = builder.elementName().get();
                this.elementOffset = builder.elementOffset();
                this.elementArgs = builder.elementArgs();
                this.staticDeclaration = builder.staticDeclaration();
                this.serviceTypeName = builder.serviceTypeName().get();
                this.annotations = Collections.unmodifiableSet(new LinkedHashSet<>(builder.annotations()));
                this.qualifiers = Collections.unmodifiableSet(new LinkedHashSet<>(builder.qualifiers()));
            }

            @Override
            public ElementKind elementKind() {
                return elementKind;
            }

            @Override
            public AccessModifier access() {
                return access;
            }

            @Override
            public TypeName elementTypeName() {
                return elementTypeName;
            }

            @Override
            public String elementName() {
                return elementName;
            }

            @Override
            public Optional<Integer> elementOffset() {
                return elementOffset;
            }

            @Override
            public Optional<Integer> elementArgs() {
                return elementArgs;
            }

            @Override
            public boolean staticDeclaration() {
                return staticDeclaration;
            }

            @Override
            public TypeName serviceTypeName() {
                return serviceTypeName;
            }

            @Override
            public Set<Annotation> annotations() {
                return annotations;
            }

            @Override
            public Set<Qualifier> qualifiers() {
                return qualifiers;
            }

            @Override
            public String toString() {
                return "ElementInfo{"
                        + "elementKind=" + elementKind + ","
                        + "access=" + access + ","
                        + "elementTypeName=" + elementTypeName + ","
                        + "elementName=" + elementName + ","
                        + "elementOffset=" + elementOffset + ","
                        + "elementArgs=" + elementArgs + ","
                        + "staticDeclaration=" + staticDeclaration + ","
                        + "serviceTypeName=" + serviceTypeName + ","
                        + "annotations=" + annotations + ","
                        + "qualifiers=" + qualifiers
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ElementInfo other)) {
                    return false;
                }
                return Objects.equals(elementKind, other.elementKind())
                        && Objects.equals(access, other.access())
                        && Objects.equals(elementTypeName, other.elementTypeName())
                        && Objects.equals(elementName, other.elementName())
                        && Objects.equals(elementOffset, other.elementOffset())
                        && Objects.equals(elementArgs, other.elementArgs())
                        && staticDeclaration == other.staticDeclaration()
                        && Objects.equals(serviceTypeName, other.serviceTypeName())
                        && Objects.equals(annotations, other.annotations())
                        && Objects.equals(qualifiers, other.qualifiers());
            }

            @Override
            public int hashCode() {
                return Objects.hash(elementKind, access, elementTypeName, elementName, elementOffset, elementArgs, staticDeclaration, serviceTypeName, annotations, qualifiers);
            }

        }

    }

    /**
     * Fluent API builder for {@link ElementInfo}.
     */
    class Builder extends ElementInfo.BuilderBase<ElementInfo.Builder, ElementInfo> implements io.helidon.common.Builder<ElementInfo.Builder, ElementInfo> {

        private Builder() {
        }

        @Override
        public ElementInfo buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ElementInfoImpl(this);
        }

        @Override
        public ElementInfo build() {
            return buildPrototype();
        }

    }

}
