/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.Future;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.inject.spi.InjectionPlan;

/**
 * Represents the result of a service activation or deactivation.
 *
 * @see Activator
 * @see DeActivator
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ActivationResultBlueprint")
public interface ActivationResult extends ActivationResultBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ActivationResult.Builder builder() {
        return new ActivationResult.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ActivationResult.Builder builder(ActivationResult instance) {
        return ActivationResult.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ActivationResult create() {
        return ActivationResult.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link ActivationResult}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ActivationResult.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ActivationResult> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final Map<String, InjectionPlan> injectionPlans = new LinkedHashMap<>();
        private final Map<String, Object> resolvedDependencies = new LinkedHashMap<>();
        private ActivationStatus finishingStatus;
        private boolean wasResolved = false;
        private Future<ActivationResultBlueprint> finishedActivationResult;
        private Phase finishingActivationPhase;
        private Phase startingActivationPhase = Phase.INIT;
        private Phase targetActivationPhase = Phase.INIT;
        private ServiceProvider<?> serviceProvider;
        private Throwable error;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ActivationResult prototype) {
            serviceProvider(prototype.serviceProvider());
            finishedActivationResult(prototype.finishedActivationResult());
            startingActivationPhase(prototype.startingActivationPhase());
            targetActivationPhase(prototype.targetActivationPhase());
            finishingActivationPhase(prototype.finishingActivationPhase());
            finishingStatus(prototype.finishingStatus());
            addInjectionPlans(prototype.injectionPlans());
            addResolvedDependencies(prototype.resolvedDependencies());
            wasResolved(prototype.wasResolved());
            error(prototype.error());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ActivationResult.BuilderBase<?, ?> builder) {
            builder.serviceProvider().ifPresent(this::serviceProvider);
            builder.finishedActivationResult().ifPresent(this::finishedActivationResult);
            startingActivationPhase(builder.startingActivationPhase());
            targetActivationPhase(builder.targetActivationPhase());
            builder.finishingActivationPhase().ifPresent(this::finishingActivationPhase);
            builder.finishingStatus().ifPresent(this::finishingStatus);
            addInjectionPlans(builder.injectionPlans());
            addResolvedDependencies(builder.resolvedDependencies());
            wasResolved(builder.wasResolved());
            builder.error().ifPresent(this::error);
            return self();
        }

        /**
         * The service provider undergoing activation or deactivation.
         *
         * @param serviceProvider the service provider generating the result
         * @return updated builder instance
         * @see #serviceProvider()
         */
        public BUILDER serviceProvider(ServiceProvider<?> serviceProvider) {
            Objects.requireNonNull(serviceProvider);
            this.serviceProvider = serviceProvider;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #finishedActivationResult()
         */
        public BUILDER clearFinishedActivationResult() {
            this.finishedActivationResult = null;
            return self();
        }

        /**
         * Optionally, given by the implementation provider to indicate the future completion when the provider's
         * {@link ActivationStatus} is {@link ActivationStatus#WARNING_SUCCESS_BUT_NOT_READY}.
         *
         * @param finishedActivationResult the future result, assuming how activation can be async in nature
         * @return updated builder instance
         * @see #finishedActivationResult()
         */
        public BUILDER finishedActivationResult(Future<ActivationResultBlueprint> finishedActivationResult) {
            Objects.requireNonNull(finishedActivationResult);
            this.finishedActivationResult = finishedActivationResult;
            return self();
        }

        /**
         * The activation phase that was found at onset of the phase transition.
         *
         * @param startingActivationPhase the starting phase
         * @return updated builder instance
         * @see #startingActivationPhase()
         */
        public BUILDER startingActivationPhase(Phase startingActivationPhase) {
            Objects.requireNonNull(startingActivationPhase);
            this.startingActivationPhase = startingActivationPhase;
            return self();
        }

        /**
         * The activation phase that was requested at the onset of the phase transition.
         *
         * @param targetActivationPhase the target, desired, ultimate phase requested
         * @return updated builder instance
         * @see #targetActivationPhase()
         */
        public BUILDER targetActivationPhase(Phase targetActivationPhase) {
            Objects.requireNonNull(targetActivationPhase);
            this.targetActivationPhase = targetActivationPhase;
            return self();
        }

        /**
         * The activation phase we finished successfully on, or are otherwise currently in if not yet finished.
         *
         * @param finishingActivationPhase the finishing phase
         * @return updated builder instance
         * @see #finishingActivationPhase()
         */
        public BUILDER finishingActivationPhase(Phase finishingActivationPhase) {
            Objects.requireNonNull(finishingActivationPhase);
            this.finishingActivationPhase = finishingActivationPhase;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #finishingStatus()
         */
        public BUILDER clearFinishingStatus() {
            this.finishingStatus = null;
            return self();
        }

        /**
         * How did the activation finish.
         * Will only be populated if the lifecycle event has completed - see {@link #finishedActivationResult()}.
         *
         * @param finishingStatus the finishing status
         * @return updated builder instance
         * @see #finishingStatus()
         */
        public BUILDER finishingStatus(ActivationStatus finishingStatus) {
            Objects.requireNonNull(finishingStatus);
            this.finishingStatus = finishingStatus;
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param injectionPlans the resolved injection plan map
         * @return updated builder instance
         * @see #injectionPlans()
         */
        public BUILDER injectionPlans(Map<? extends String, ? extends InjectionPlan> injectionPlans) {
            Objects.requireNonNull(injectionPlans);
            this.injectionPlans.clear();
            this.injectionPlans.putAll(injectionPlans);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param injectionPlans the resolved injection plan map
         * @return updated builder instance
         * @see #injectionPlans()
         */
        public BUILDER addInjectionPlans(Map<? extends String, ? extends InjectionPlan> injectionPlans) {
            Objects.requireNonNull(injectionPlans);
            this.injectionPlans.putAll(injectionPlans);
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param resolvedDependencies the resolved dependency map
         * @return updated builder instance
         * @see #resolvedDependencies()
         */
        public BUILDER resolvedDependencies(Map<? extends String, ?> resolvedDependencies) {
            Objects.requireNonNull(resolvedDependencies);
            this.resolvedDependencies.clear();
            this.resolvedDependencies.putAll(resolvedDependencies);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param resolvedDependencies the resolved dependency map
         * @return updated builder instance
         * @see #resolvedDependencies()
         */
        public BUILDER addResolvedDependencies(Map<? extends String, ?> resolvedDependencies) {
            Objects.requireNonNull(resolvedDependencies);
            this.resolvedDependencies.putAll(resolvedDependencies);
            return self();
        }

        /**
         * Set to true if the injection plan in {@link #resolvedDependencies()} has been resolved and can be "trusted" as being
         * complete and accurate.
         *
         * @param wasResolved true if was resolved
         * @return updated builder instance
         * @see #wasResolved()
         */
        public BUILDER wasResolved(boolean wasResolved) {
            this.wasResolved = wasResolved;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #error()
         */
        public BUILDER clearError() {
            this.error = null;
            return self();
        }

        /**
         * Any throwable/exceptions that were observed during activation.
         *
         * @param error any captured error
         * @return updated builder instance
         * @see #error()
         */
        public BUILDER error(Throwable error) {
            Objects.requireNonNull(error);
            this.error = error;
            return self();
        }

        /**
         * The service provider undergoing activation or deactivation.
         *
         * @return the service provider
         */
        public Optional<ServiceProvider<?>> serviceProvider() {
            return Optional.ofNullable(serviceProvider);
        }

        /**
         * Optionally, given by the implementation provider to indicate the future completion when the provider's
         * {@link ActivationStatus} is {@link ActivationStatus#WARNING_SUCCESS_BUT_NOT_READY}.
         *
         * @return the finished activation result
         */
        public Optional<Future<ActivationResultBlueprint>> finishedActivationResult() {
            return Optional.ofNullable(finishedActivationResult);
        }

        /**
         * The activation phase that was found at onset of the phase transition.
         *
         * @return the starting activation phase
         */
        public Phase startingActivationPhase() {
            return startingActivationPhase;
        }

        /**
         * The activation phase that was requested at the onset of the phase transition.
         *
         * @return the target activation phase
         */
        public Phase targetActivationPhase() {
            return targetActivationPhase;
        }

        /**
         * The activation phase we finished successfully on, or are otherwise currently in if not yet finished.
         *
         * @return the finishing activation phase
         */
        public Optional<Phase> finishingActivationPhase() {
            return Optional.ofNullable(finishingActivationPhase);
        }

        /**
         * How did the activation finish.
         * Will only be populated if the lifecycle event has completed - see {@link #finishedActivationResult()}.
         *
         * @return the finishing status
         */
        public Optional<ActivationStatus> finishingStatus() {
            return Optional.ofNullable(finishingStatus);
        }

        /**
         * The injection plan that was found or determined, key'ed by each element's {@link ServiceProvider#id()}.
         *
         * @return the injection plans
         */
        public Map<String, InjectionPlan> injectionPlans() {
            return injectionPlans;
        }

        /**
         * The dependencies that were resolved or loaded, key'ed by each element's {@link ServiceProvider#id()}.
         *
         * @return the resolved dependencies
         */
        public Map<String, Object> resolvedDependencies() {
            return resolvedDependencies;
        }

        /**
         * Set to true if the injection plan in {@link #resolvedDependencies()} has been resolved and can be "trusted" as being
         * complete and accurate.
         *
         * @return the was resolved
         */
        public boolean wasResolved() {
            return wasResolved;
        }

        /**
         * Any throwable/exceptions that were observed during activation.
         *
         * @return the error
         */
        public Optional<Throwable> error() {
            return Optional.ofNullable(error);
        }

        @Override
        public String toString() {
            return "ActivationResultBuilder{"
                    + "serviceProvider=" + serviceProvider + ","
                    + "finishedActivationResult=" + finishedActivationResult + ","
                    + "startingActivationPhase=" + startingActivationPhase + ","
                    + "targetActivationPhase=" + targetActivationPhase + ","
                    + "finishingActivationPhase=" + finishingActivationPhase + ","
                    + "finishingStatus=" + finishingStatus + ","
                    + "injectionPlans=" + injectionPlans + ","
                    + "resolvedDependencies=" + resolvedDependencies + ","
                    + "wasResolved=" + wasResolved + ","
                    + "error=" + error
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceProvider == null) {
                collector.fatal(getClass(), "Property \"serviceProvider\" must not be null, but not set");
            }
            if (finishingActivationPhase == null) {
                collector.fatal(getClass(), "Property \"finishingActivationPhase\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Optionally, given by the implementation provider to indicate the future completion when the provider's
         * {@link ActivationStatus} is {@link ActivationStatus#WARNING_SUCCESS_BUT_NOT_READY}.
         *
         * @param finishedActivationResult the future result, assuming how activation can be async in nature
         * @return updated builder instance
         * @see #finishedActivationResult()
         */
        BUILDER finishedActivationResult(Optional<Future<ActivationResultBlueprint>> finishedActivationResult) {
            Objects.requireNonNull(finishedActivationResult);
            this.finishedActivationResult = finishedActivationResult.map(java.util.concurrent.Future.class::cast).orElse(this.finishedActivationResult);
            return self();
        }

        /**
         * How did the activation finish.
         * Will only be populated if the lifecycle event has completed - see {@link #finishedActivationResult()}.
         *
         * @param finishingStatus the finishing status
         * @return updated builder instance
         * @see #finishingStatus()
         */
        BUILDER finishingStatus(Optional<? extends ActivationStatus> finishingStatus) {
            Objects.requireNonNull(finishingStatus);
            this.finishingStatus = finishingStatus.map(io.helidon.inject.api.ActivationStatus.class::cast).orElse(this.finishingStatus);
            return self();
        }

        /**
         * Any throwable/exceptions that were observed during activation.
         *
         * @param error any captured error
         * @return updated builder instance
         * @see #error()
         */
        BUILDER error(Optional<? extends Throwable> error) {
            Objects.requireNonNull(error);
            this.error = error.map(java.lang.Throwable.class::cast).orElse(this.error);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ActivationResultImpl implements ActivationResult {

            private final boolean wasResolved;
            private final Map<String, InjectionPlan> injectionPlans;
            private final Map<String, Object> resolvedDependencies;
            private final Optional<ActivationStatus> finishingStatus;
            private final Optional<Throwable> error;
            private final Optional<Future<ActivationResultBlueprint>> finishedActivationResult;
            private final Phase finishingActivationPhase;
            private final Phase startingActivationPhase;
            private final Phase targetActivationPhase;
            private final ServiceProvider<?> serviceProvider;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ActivationResultImpl(ActivationResult.BuilderBase<?, ?> builder) {
                this.serviceProvider = builder.serviceProvider().get();
                this.finishedActivationResult = builder.finishedActivationResult();
                this.startingActivationPhase = builder.startingActivationPhase();
                this.targetActivationPhase = builder.targetActivationPhase();
                this.finishingActivationPhase = builder.finishingActivationPhase().get();
                this.finishingStatus = builder.finishingStatus();
                this.injectionPlans = Collections.unmodifiableMap(new LinkedHashMap<>(builder.injectionPlans()));
                this.resolvedDependencies = Collections.unmodifiableMap(new LinkedHashMap<>(builder.resolvedDependencies()));
                this.wasResolved = builder.wasResolved();
                this.error = builder.error();
            }

            @Override
            public ServiceProvider<?> serviceProvider() {
                return serviceProvider;
            }

            @Override
            public Optional<Future<ActivationResultBlueprint>> finishedActivationResult() {
                return finishedActivationResult;
            }

            @Override
            public Phase startingActivationPhase() {
                return startingActivationPhase;
            }

            @Override
            public Phase targetActivationPhase() {
                return targetActivationPhase;
            }

            @Override
            public Phase finishingActivationPhase() {
                return finishingActivationPhase;
            }

            @Override
            public Optional<ActivationStatus> finishingStatus() {
                return finishingStatus;
            }

            @Override
            public Map<String, InjectionPlan> injectionPlans() {
                return injectionPlans;
            }

            @Override
            public Map<String, Object> resolvedDependencies() {
                return resolvedDependencies;
            }

            @Override
            public boolean wasResolved() {
                return wasResolved;
            }

            @Override
            public Optional<Throwable> error() {
                return error;
            }

            @Override
            public String toString() {
                return "ActivationResult{"
                        + "serviceProvider=" + serviceProvider + ","
                        + "finishedActivationResult=" + finishedActivationResult + ","
                        + "startingActivationPhase=" + startingActivationPhase + ","
                        + "targetActivationPhase=" + targetActivationPhase + ","
                        + "finishingActivationPhase=" + finishingActivationPhase + ","
                        + "finishingStatus=" + finishingStatus + ","
                        + "injectionPlans=" + injectionPlans + ","
                        + "resolvedDependencies=" + resolvedDependencies + ","
                        + "wasResolved=" + wasResolved + ","
                        + "error=" + error
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ActivationResult other)) {
                    return false;
                }
                return Objects.equals(serviceProvider, other.serviceProvider())
                        && Objects.equals(finishedActivationResult, other.finishedActivationResult())
                        && Objects.equals(startingActivationPhase, other.startingActivationPhase())
                        && Objects.equals(targetActivationPhase, other.targetActivationPhase())
                        && Objects.equals(finishingActivationPhase, other.finishingActivationPhase())
                        && Objects.equals(finishingStatus, other.finishingStatus())
                        && Objects.equals(injectionPlans, other.injectionPlans())
                        && Objects.equals(resolvedDependencies, other.resolvedDependencies())
                        && wasResolved == other.wasResolved()
                        && Objects.equals(error, other.error());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceProvider, finishedActivationResult, startingActivationPhase, targetActivationPhase, finishingActivationPhase, finishingStatus, injectionPlans, resolvedDependencies, wasResolved, error);
            }

        }

    }

    /**
     * Fluent API builder for {@link ActivationResult}.
     */
    class Builder extends ActivationResult.BuilderBase<ActivationResult.Builder, ActivationResult> implements io.helidon.common.Builder<ActivationResult.Builder, ActivationResult> {

        private Builder() {
        }

        @Override
        public ActivationResult buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ActivationResultImpl(this);
        }

        @Override
        public ActivationResult build() {
            return buildPrototype();
        }

    }

}
