/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * This is the bootstrap needed to provide to {@code Services} initialization.
 *
 * @see io.helidon.inject.spi.InjectionServicesProvider
 * @see io.helidon.inject.api.InjectionServices#globalBootstrap()
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.BootstrapBlueprint")
public interface Bootstrap extends BootstrapBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static Bootstrap.Builder builder() {
        return new Bootstrap.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static Bootstrap.Builder builder(Bootstrap instance) {
        return Bootstrap.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static Bootstrap create() {
        return Bootstrap.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link Bootstrap}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends Bootstrap.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends Bootstrap> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Config config;
        private Phase limitRuntimePhase;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Bootstrap prototype) {
            this.config = prototype.config().orElse(null);
            limitRuntimePhase(prototype.limitRuntimePhase());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(Bootstrap.BuilderBase<?, ?> builder) {
            builder.config().ifPresent(this::config);
            builder.limitRuntimePhase().ifPresent(this::limitRuntimePhase);
            return self();
        }

        /**
         * Config to use.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #limitRuntimePhase()
         */
        public BUILDER clearLimitRuntimePhase() {
            this.limitRuntimePhase = null;
            return self();
        }

        /**
         * In certain conditions Injection services should be initialized but not started (i.e., avoiding calls to {@code PostConstruct}
         * etc.). This can be used in special cases where the normal Injection startup should limit lifecycle up to a given phase. Normally
         * one should not use this feature - it is mainly used in Injection tooling (e.g., the injection maven-plugin).
         *
         * @param limitRuntimePhase the phase to stop at during lifecycle
         * @return updated builder instance
         * @see #limitRuntimePhase()
         */
        public BUILDER limitRuntimePhase(Phase limitRuntimePhase) {
            Objects.requireNonNull(limitRuntimePhase);
            this.limitRuntimePhase = limitRuntimePhase;
            return self();
        }

        /**
         * Provides the base primordial bootstrap configuration to the {@link io.helidon.inject.spi.InjectionServicesProvider}.
         * The provider will then bootstrap {@link InjectionServices} using this bootstrap instance.
         * then default values will be used accordingly.
         *
         * @return the config
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        /**
         * In certain conditions Injection services should be initialized but not started (i.e., avoiding calls to {@code PostConstruct}
         * etc.). This can be used in special cases where the normal Injection startup should limit lifecycle up to a given phase. Normally
         * one should not use this feature - it is mainly used in Injection tooling (e.g., the injection maven-plugin).
         *
         * @return the limit runtime phase
         */
        public Optional<Phase> limitRuntimePhase() {
            return Optional.ofNullable(limitRuntimePhase);
        }

        @Override
        public String toString() {
            return "BootstrapBuilder{"
                    + "config=" + config + ","
                    + "limitRuntimePhase=" + limitRuntimePhase
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * In certain conditions Injection services should be initialized but not started (i.e., avoiding calls to {@code PostConstruct}
         * etc.). This can be used in special cases where the normal Injection startup should limit lifecycle up to a given phase. Normally
         * one should not use this feature - it is mainly used in Injection tooling (e.g., the injection maven-plugin).
         *
         * @param limitRuntimePhase the phase to stop at during lifecycle
         * @return updated builder instance
         * @see #limitRuntimePhase()
         */
        BUILDER limitRuntimePhase(Optional<? extends Phase> limitRuntimePhase) {
            Objects.requireNonNull(limitRuntimePhase);
            this.limitRuntimePhase = limitRuntimePhase.map(io.helidon.inject.api.Phase.class::cast).orElse(this.limitRuntimePhase);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class BootstrapImpl implements Bootstrap {

            private final Optional<Config> config;
            private final Optional<Phase> limitRuntimePhase;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected BootstrapImpl(Bootstrap.BuilderBase<?, ?> builder) {
                this.config = builder.config();
                this.limitRuntimePhase = builder.limitRuntimePhase();
            }

            @Override
            public Optional<Config> config() {
                return config;
            }

            @Override
            public Optional<Phase> limitRuntimePhase() {
                return limitRuntimePhase;
            }

            @Override
            public String toString() {
                return "Bootstrap{"
                        + "config=" + config + ","
                        + "limitRuntimePhase=" + limitRuntimePhase
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof Bootstrap other)) {
                    return false;
                }
                return Objects.equals(config, other.config())
                        && Objects.equals(limitRuntimePhase, other.limitRuntimePhase());
            }

            @Override
            public int hashCode() {
                return Objects.hash(config, limitRuntimePhase);
            }

        }

    }

    /**
     * Fluent API builder for {@link Bootstrap}.
     */
    class Builder extends Bootstrap.BuilderBase<Bootstrap.Builder, Bootstrap> implements io.helidon.common.Builder<Bootstrap.Builder, Bootstrap> {

        private Builder() {
        }

        @Override
        public Bootstrap buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new BootstrapImpl(this);
        }

        @Override
        public Bootstrap build() {
            return buildPrototype();
        }

    }

}
