/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * For internal use only to Helidon. Applicable when {@link InjectionServices#TAG_DEBUG} is enabled.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.CallingContextBlueprint")
public interface CallingContext extends CallingContextBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static CallingContext.Builder builder() {
        return new CallingContext.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static CallingContext.Builder builder(CallingContext instance) {
        return CallingContext.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static CallingContext create() {
        return CallingContext.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link CallingContext}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends CallingContext.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends CallingContext> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private StackTraceElement[] stackTrace;
        private String moduleName;
        private String threadName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CallingContext prototype) {
            stackTrace(prototype.stackTrace());
            moduleName(prototype.moduleName());
            threadName(prototype.threadName());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(CallingContext.BuilderBase<?, ?> builder) {
            builder.stackTrace().ifPresent(this::stackTrace);
            builder.moduleName().ifPresent(this::moduleName);
            builder.threadName().ifPresent(this::threadName);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #stackTrace()
         */
        public BUILDER clearStackTrace() {
            this.stackTrace = null;
            return self();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @param stackTrace the stack trace for who initialized
         * @return updated builder instance
         * @see #stackTrace()
         */
        public BUILDER stackTrace(StackTraceElement[] stackTrace) {
            Objects.requireNonNull(stackTrace);
            this.stackTrace = stackTrace;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #moduleName()
         */
        public BUILDER clearModuleName() {
            this.moduleName = null;
            return self();
        }

        /**
         * Only populated when {@code module} is set.
         *
         * @param moduleName the module name
         * @return updated builder instance
         * @see #moduleName()
         */
        public BUILDER moduleName(String moduleName) {
            Objects.requireNonNull(moduleName);
            this.moduleName = moduleName;
            return self();
        }

        /**
         * The thread that created the calling context.
         *
         * @param threadName thread creating the calling context
         * @return updated builder instance
         * @see #threadName()
         */
        public BUILDER threadName(String threadName) {
            Objects.requireNonNull(threadName);
            this.threadName = threadName;
            return self();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @return the stack trace
         */
        public Optional<StackTraceElement[]> stackTrace() {
            return Optional.ofNullable(stackTrace);
        }

        /**
         * Only populated when {@code module} is set.
         *
         * @return the module name
         */
        public Optional<String> moduleName() {
            return Optional.ofNullable(moduleName);
        }

        /**
         * The thread that created the calling context.
         *
         * @return the thread name
         */
        public Optional<String> threadName() {
            return Optional.ofNullable(threadName);
        }

        @Override
        public String toString() {
            return "CallingContextBuilder{"
                    + "stackTrace=" + stackTrace + ","
                    + "moduleName=" + moduleName + ","
                    + "threadName=" + threadName
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new CallingContextBlueprint.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (threadName == null) {
                collector.fatal(getClass(), "Property \"threadName\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Only populated when {@link InjectionServices#TAG_DEBUG} is set.
         *
         * @param stackTrace the stack trace for who initialized
         * @return updated builder instance
         * @see #stackTrace()
         */
        BUILDER stackTrace(Optional<StackTraceElement[]> stackTrace) {
            Objects.requireNonNull(stackTrace);
            this.stackTrace = stackTrace.map(java.lang.StackTraceElement[].class::cast).orElse(this.stackTrace);
            return self();
        }

        /**
         * Only populated when {@code module} is set.
         *
         * @param moduleName the module name
         * @return updated builder instance
         * @see #moduleName()
         */
        BUILDER moduleName(Optional<String> moduleName) {
            Objects.requireNonNull(moduleName);
            this.moduleName = moduleName.map(java.lang.String.class::cast).orElse(this.moduleName);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class CallingContextImpl implements CallingContext {

            private final Optional<StackTraceElement[]> stackTrace;
            private final Optional<String> moduleName;
            private final String threadName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected CallingContextImpl(CallingContext.BuilderBase<?, ?> builder) {
                this.stackTrace = builder.stackTrace();
                this.moduleName = builder.moduleName();
                this.threadName = builder.threadName().get();
            }

            @Override
            public Optional<StackTraceElement[]> stackTrace() {
                return stackTrace;
            }

            @Override
            public Optional<String> moduleName() {
                return moduleName;
            }

            @Override
            public String threadName() {
                return threadName;
            }

            @Override
            public String toString() {
                return "CallingContext{"
                        + "stackTrace=" + stackTrace + ","
                        + "moduleName=" + moduleName + ","
                        + "threadName=" + threadName
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof CallingContext other)) {
                    return false;
                }
                return Objects.equals(stackTrace, other.stackTrace())
                        && Objects.equals(moduleName, other.moduleName())
                        && Objects.equals(threadName, other.threadName());
            }

            @Override
            public int hashCode() {
                return Objects.hash(stackTrace, moduleName, threadName);
            }

        }

    }

    /**
     * Fluent API builder for {@link CallingContext}.
     */
    class Builder extends CallingContext.BuilderBase<CallingContext.Builder, CallingContext> implements io.helidon.common.Builder<CallingContext.Builder, CallingContext> {

        private Builder() {
        }

        @Override
        public CallingContext buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new CallingContextImpl(this);
        }

        @Override
        public CallingContext build() {
            return buildPrototype();
        }

    }

}
